package com.stringee.database;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import com.stringee.common.Constant;
import com.stringee.common.Utils;
import com.stringee.messaging.ChannelType;
import com.stringee.messaging.ChatStatus;
import com.stringee.messaging.Conversation;
import com.stringee.messaging.Conversation.State;
import com.stringee.messaging.Message;
import com.stringee.messaging.Message.MsgType;
import com.stringee.messaging.User;
import com.stringee.messaging.User.Role;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class DBHandler extends SQLiteOpenHelper {

    private static final int DATABASE_VERSION = 18;
    private static final String DATABASE_NAME = "com.stringee.db";

    private static DBHandler dbHandler;
    private final SQLiteDatabase writeDatabase;
    private final SQLiteDatabase readDatabase;

    public static DBHandler getInstance(Context context) {
        synchronized (DBHandler.class) {
            if (dbHandler == null) {
                dbHandler = new DBHandler(context);
            }
            return dbHandler;
        }
    }

    public DBHandler(Context context) {
        super(context, context.getPackageName() + DATABASE_NAME, null, DATABASE_VERSION);
        writeDatabase = getWritableDatabase();
        readDatabase = getReadableDatabase();
    }

    @Override
    public void onCreate(SQLiteDatabase sqLiteDatabase) {
        createAllTable(sqLiteDatabase);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        if (oldVersion == 1 && newVersion >= 2) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.LAST_SEQUENCE + " INTEGER");
        }

        if (oldVersion <= 2 && newVersion > 2) {
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.STICKER_CATEGORY + " TEXT");
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.STICKER_NAME + " TEXT");
        }

        if (oldVersion <= 3 && newVersion > 3) {
            try {
                db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                        ConversationConstant.LAST_MESSAGE_ID + " TEXT");
            } catch (Exception ex) {
                Utils.reportException(DBHandler.class, ex);
            }
        }

        if (oldVersion <= 4 && newVersion > 4) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.CREATED_AT + " INTEGER");
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.LAST_MESSAGE + " TEXT");
        }

        if (oldVersion <= 5 && newVersion > 5) {
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.CUSTOM_DATA + " TEXT");
        }

        if (oldVersion <= 6 && newVersion > 6) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.LAST_MSG_SEQ_RECEIVED + " INTEGER");
        }

        if (oldVersion <= 7 && newVersion > 7) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.PINNED_MSG_ID + " TEXT");
        }

        if (oldVersion <= 8 && newVersion > 8) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.CHANNEL_TYPE + " INTEGER");
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.ENDED + " INTEGER");
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.DELETED + " INTEGER");
            db.execSQL("ALTER TABLE " + ParticipantConstant.TABLE_NAME + " ADD COLUMN " +
                    ParticipantConstant.ROLE + " TEXT");
        }

        if (oldVersion <= 9 && newVersion > 9) {
            db.execSQL("ALTER TABLE " + ChatRequestConstant.TABLE_NAME + " ADD COLUMN " +
                    ChatRequestConstant.CHANNEL_TYPE + " INTEGER");
            db.execSQL("ALTER TABLE " + ChatRequestConstant.TABLE_NAME + " ADD COLUMN " +
                    ChatRequestConstant.REQUEST_TYPE + " INTEGER");
            db.execSQL("ALTER TABLE " + ChatRequestConstant.TABLE_NAME + " ADD COLUMN " +
                    ChatRequestConstant.TRANSFER_FROM + " TEXT");
        }

        if (oldVersion <= 10 && newVersion > 10) {
            db.execSQL("ALTER TABLE " + ChatRequestConstant.TABLE_NAME + " ADD COLUMN " +
                    ChatRequestConstant.TRANSFER_FROM_NAME + " TEXT");
            db.execSQL("ALTER TABLE " + ChatRequestConstant.TABLE_NAME + " ADD COLUMN " +
                    ChatRequestConstant.TRANSFER_FROM_AVATAR + " TEXT");
        }

        if (oldVersion <= 11 && newVersion > 11) {
            db.execSQL("DROP TABLE " + ChatRequestConstant.TABLE_NAME);
        }

        if (oldVersion <= 12 && newVersion > 12) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.OA_ID + " INTEGER");
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.CUSTOM_DATA + " INTEGER");
        }

        if (oldVersion <= 13 && newVersion > 13) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.LAST_SENDER_NAME + " TEXT");
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.LAST_SENDER_AVATAR + " TEXT");
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.AUTHOR_NAME + " TEXT");
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.AUTHOR_AVATAR + " TEXT");
        }

        if (oldVersion <= 14 && newVersion > 14) {
            db.execSQL(
                    "ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " + UserConstant.EMAIL +
                            " TEXT");
            db.execSQL(
                    "ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " + UserConstant.PHONE +
                            " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.LOCATION + " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.BROWSER + " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.PLATFORM + " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.DEVICE + " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.IP_ADDRESS + " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.HOST_NAME + " TEXT");
            db.execSQL("ALTER TABLE " + UserConstant.TABLE_NAME + " ADD COLUMN " +
                    UserConstant.USER_AGENT + " TEXT");
        }

        if (oldVersion <= 15 && newVersion > 15) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.LAST_MSG_SEQ_SEEN + " INTEGER");
        }

        if (oldVersion <= 16 && newVersion > 16) {
            db.execSQL("ALTER TABLE " + MessageConstant.TABLE_NAME + " ADD COLUMN " +
                    MessageConstant.FILE_ID + " TEXT");
        }

        if (oldVersion <= 17 && newVersion > 17) {
            db.execSQL("ALTER TABLE " + ConversationConstant.TABLE_NAME + " ADD COLUMN " +
                    ConversationConstant.CHAT_STATUS + " INTEGER");
        }
    }

    private void createAllTable(SQLiteDatabase db) {
        db.execSQL(ConversationConstant.CREATE_TABLE);
        db.execSQL(MessageConstant.CREATE_TABLE);
        db.execSQL(ParticipantConstant.CREATE_TABLE);
        db.execSQL(UserConstant.CREATE_TABLE);
        db.execSQL(ChatRequestConstant.CREATE_TABLE);

        db.execSQL(ConversationConstant.CREATE_CONVERSATION_INDEX);
        db.execSQL(MessageConstant.CREATE_SENDER_ID_INDEX);
        db.execSQL(ParticipantConstant.CREATE_CONVERSATION_INDEX);
        db.execSQL(ChatRequestConstant.CREATE_CHAT_REQUEST_INDEX);
    }

    //********************* Conversation ************************//

    /**
     * Save a conversation to local db
     */
    public long insertConversation(Conversation conversation) {
        ContentValues values = getConversationContentValues(conversation);
        return writeDatabase.insert(ConversationConstant.TABLE_NAME, null, values);
    }

    /**
     * Update conversation when load from server
     */
    public long updateConversation(Conversation conversation) {
        ContentValues values = getConversationContentValues(conversation);
        return writeDatabase.update(ConversationConstant.TABLE_NAME, values,
                ConversationConstant.ID +
                        " = ? ", new String[]{String.valueOf(conversation.getDbId())});
    }

    /**
     * Update conversation when send a new message
     */
    public void updateConversationWhenSendMsg(Conversation conversation, String userId) {
        ContentValues values = new ContentValues();
        values.put(ConversationConstant.LAST_UPDATE_TIME, conversation.getUpdateAt());
        Message lastMessage = conversation.getLastMessage();
        User lastMessageSender = lastMessage.getSender();
        values.put(ConversationConstant.LAST_MESSAGE_TEXT, lastMessage.getText());
        values.put(ConversationConstant.LAST_SENDER_ID, lastMessageSender.getUserId());
        values.put(ConversationConstant.LAST_SENDER_NAME, lastMessageSender.getName());
        values.put(ConversationConstant.LAST_SENDER_AVATAR, lastMessageSender.getAvatarUrl());
        values.put(ConversationConstant.LAST_MESSAGE_TYPE, lastMessage.getType().getValue());
        values.put(ConversationConstant.LAST_MESSAGE_STATE, lastMessage.getState().getValue());
        values.put(ConversationConstant.LAST_TIME_NEW_MESSAGE, conversation.getLastTimeNewMsg());
        values.put(ConversationConstant.LAST_MESSAGE, conversation.getLastMsg());
        values.put(ConversationConstant.LAST_MSG_SEQ_RECEIVED, conversation.getLastMsgSeqReceived());
        values.put(ConversationConstant.LAST_MSG_SEQ_SEEN, conversation.getLastMsgSeqSeen());
        values.put(ConversationConstant.PINNED_MSG_ID, conversation.getPinnedMsgId());
        values.put(ConversationConstant.LAST_UPDATE_TIME, conversation.getUpdateAt());
        values.put(ConversationConstant.CHAT_STATUS, conversation.getChatStatus().getValue());
        writeDatabase.update(ConversationConstant.TABLE_NAME, values,
                ConversationConstant.CONVERSATION_ID + " = ? AND " + ConversationConstant.USER_ID +
                        " = ?", new String[]{String.valueOf(conversation.getId()), userId});
    }

    private ContentValues getConversationContentValues(Conversation conversation) {
        ContentValues values = new ContentValues();
        String localId = conversation.getLocalId();
        if (!Utils.isEmpty(localId)) {
            values.put(ConversationConstant.LOCAL_ID, conversation.getLocalId());
        }
        values.put(ConversationConstant.CONVERSATION_ID, conversation.getId());
        values.put(ConversationConstant.NAME, conversation.getName());
        values.put(ConversationConstant.IS_GROUP, conversation.isGroup() ? 1 : 0);
        values.put(ConversationConstant.IS_DISTINCT, conversation.isDistinct() ? 1 : 0);
        values.put(ConversationConstant.USER_ID, conversation.getClientId());
        values.put(ConversationConstant.LAST_UPDATE_TIME, conversation.getUpdateAt());
        Message lastMessage = conversation.getLastMessage();
        User lastMessageSender = lastMessage.getSender();
        values.put(ConversationConstant.LAST_MESSAGE_TEXT, lastMessage.getText());
        values.put(ConversationConstant.LAST_SENDER_ID, lastMessageSender.getUserId());
        values.put(ConversationConstant.LAST_SENDER_NAME, lastMessageSender.getName());
        values.put(ConversationConstant.LAST_SENDER_AVATAR, lastMessageSender.getAvatarUrl());
        values.put(ConversationConstant.LAST_MESSAGE_TYPE, lastMessage.getType().getValue());
        values.put(ConversationConstant.LAST_MESSAGE_STATE, lastMessage.getState().getValue());
        values.put(ConversationConstant.TOTAL_UNREAD, conversation.getTotalUnread());
        String creator = conversation.getCreator();
        if (!Utils.isEmpty(creator)) {
            values.put(ConversationConstant.CREATOR, creator);
        }
        values.put(ConversationConstant.LAST_TIME_NEW_MESSAGE, conversation.getLastTimeNewMsg());
        values.put(ConversationConstant.STATE, conversation.getState().getValue());
        values.put(ConversationConstant.TYPE, conversation.getType().getValue());
        values.put(ConversationConstant.LAST_SEQUENCE, conversation.getLastSequence());
        values.put(ConversationConstant.LAST_MESSAGE_ID, lastMessage.getId());
        values.put(ConversationConstant.LAST_MESSAGE, conversation.getLastMsg());
        values.put(ConversationConstant.LAST_MSG_SEQ_RECEIVED, conversation.getLastMsgSeqReceived());
        values.put(ConversationConstant.PINNED_MSG_ID, conversation.getPinnedMsgId());
        values.put(ConversationConstant.ENDED, conversation.isEnded() ? 1 : 0);
        values.put(ConversationConstant.OA_ID, conversation.getOaId());
        values.put(ConversationConstant.CUSTOM_DATA, conversation.getCustomData());
        values.put(ConversationConstant.CREATED_AT, conversation.getCreateAt());
        values.put(ConversationConstant.CHANNEL_TYPE, conversation.getChannelType().getValue());
        values.put(ConversationConstant.LAST_MSG_SEQ_SEEN, conversation.getLastMsgSeqSeen());
        values.put(ConversationConstant.CHAT_STATUS, conversation.getChatStatus().getValue());
        return values;
    }

//    /**
//     * Get conversation by participants
//     *
//     * @param userId
//     * @param participants
//     * @return
//     */
//    public Conversation getConversationByParticipants(String userId, List<User> participants) {
//        if (participants == null || participants.size() == 0) {
//            return null;
//        }
//        Conversation conversation = null;
//        String query = "SELECT " + ConversationConstant.TABLE_NAME + ".*,members.*" + " FROM " + ConversationConstant.TABLE_NAME + " INNER JOIN (SELECT " + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.CONVERSATION_ID + "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.USER_ID + "," + UserConstant.TABLE_NAME + "." + UserConstant.FULL_NAME + "," + UserConstant.TABLE_NAME + "." + UserConstant.AVATAR + " FROM " + ParticipantConstant.TABLE_NAME + " LEFT JOIN " + UserConstant.TABLE_NAME + " ON " + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.USER_ID + " = " + UserConstant.TABLE_NAME + "." + UserConstant.USER_ID + ") AS members ON " + ConversationConstant.TABLE_NAME + "." + ConversationConstant.CONVERSATION_ID + " = " + "members." + ParticipantConstant.CONVERSATION_ID + "  WHERE " + ConversationConstant.TABLE_NAME + "." + ConversationConstant.USER_ID + " = ?  AND " + ConversationConstant.IS_DISTINCT + " = 1  ORDER BY " + ConversationConstant.TABLE_NAME + "." + ConversationConstant.CONVERSATION_ID + ",members." + UserConstant.USER_ID + " ASC";
//
//        String[] selectionArgs = {userId};
//        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
//        if (cursor != null) {
//            if (cursor.moveToFirst()) {
//                List<Conversation> conversations = new ArrayList<>();
//                do {
//                    String conversationId = cursor.getString(2);
//                    String parUserId = cursor.getString(15);
//                    if (conversations.size() > 0) {
//                        Conversation lastConv = conversations.get(conversations.size() - 1);
//                        String lastConvId = lastConv.getId();
//                        if (conversationId.equals(lastConvId)) {
//                            User user = new User(parUserId);
//                            user.setName(cursor.getString(16));
//                            user.setAvatarUrl(cursor.getString(17));
//                            lastConv.addParticipants(user);
//                            continue;
//                        }
//                    }
//                    Conversation tempConv = new Conversation();
//                    tempConv.setDbId(cursor.getInt(0));
//                    tempConv.setLocalId(cursor.getString(1));
//                    tempConv.setId(conversationId);
//                    tempConv.setName(cursor.getString(3));
//                    tempConv.setGroup(cursor.getInt(4) == 1 ? true : false);
//                    tempConv.setDistinct(cursor.getInt(5) == 1 ? true : false);
//                    tempConv.setClientId(cursor.getString(6));
//                    tempConv.setUpdateAt(cursor.getLong(7));
//                    tempConv.setText(cursor.getString(8));
//                    tempConv.setLastMsgSender(cursor.getString(9));
//                    tempConv.setLastMsgType(cursor.getInt(10));
//                    tempConv.setTotalUnread(cursor.getInt(11));
//                    tempConv.setCreator(cursor.getString(12));
//                    tempConv.setLastTimeNewMsg(cursor.getLong(13));
//                    List<User> pars = new ArrayList<>();
//                    User user = new User(parUserId);
//                    user.setName(cursor.getString(16));
//                    user.setAvatarUrl(cursor.getString(17));
//                    pars.add(user);
//                    tempConv.setParticipants(pars);
//                    conversations.add(tempConv);
//                } while (cursor.moveToNext());
//
//                Collections.sort(participants, new Comparator<User>() {
//                    @Override
//                    public int compare(User identity, User t1) {
//                        return identity.getClientId().compareTo(t1.getClientId());
//                    }
//                });
//                if (conversations.size() > 0) {
//                    for (int i = 0; i < conversations.size(); i++) {
//                        Conversation conversation1 = conversations.get(i);
//                        List<User> users = conversation1.getParticipants();
//                        if (users.equals(participants)) {
//                            return conversation1;
//                        }
//                    }
//                }
//            }
//            if (!cursor.isClosed()) {
//                cursor.close();
//            }
//        }
//        return conversation;
//    }

    /**
     * Get a conversation by remote id
     */
    public Conversation getConversationByRemoteId(String remoteId, String clientId) {
        Conversation conversation = null;
        String query =
                "SELECT " + ConversationConstant.TABLE_NAME + ".*," + " members.*" + " FROM " +
                        ConversationConstant.TABLE_NAME + " LEFT JOIN " + "(SELECT " +
                        ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.CONVERSATION_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.USER_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.ROLE +
                        "," + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_SEEN_SEQ + "," +
                        ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_RECEIVED_SEQ + "," + UserConstant.TABLE_NAME +
                        ".* " + " FROM " + ParticipantConstant.TABLE_NAME + " LEFT JOIN " +
                        UserConstant.TABLE_NAME + " ON " + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.USER_ID + " = " + UserConstant.TABLE_NAME + "." +
                        UserConstant.USER_ID + ")" + " AS members ON " +
                        ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + " = " + "members." +
                        ParticipantConstant.CONVERSATION_ID + " WHERE " +
                        ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + " = ?" + " AND " +
                        ConversationConstant.TABLE_NAME + "." + ConversationConstant.USER_ID +
                        " = ?" + " ORDER BY members." + ParticipantConstant.USER_ID + " ASC";

        String[] selectionArgs = {remoteId, clientId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            conversation = getConversationFromCursor(cursor);
            List<User> pars = new ArrayList<>();
            User user = getParticipantFromCursor(cursor);
            if (user != null) {
                pars.add(user);
                conversation.setParticipants(pars);
            }

            while (cursor.moveToNext()) {
                User nextUser = getParticipantFromCursor(cursor);
                if (nextUser != null) {
                    conversation.addParticipants(nextUser);
                }
            }
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return conversation;
    }

    public List<Conversation> getAllConversations(String userId) {
        List<Conversation> conversations = new ArrayList<>();
        String query =
                "SELECT " + ConversationConstant.TABLE_NAME + ".*," + " members.*" + " FROM " +
                        ConversationConstant.TABLE_NAME + " LEFT JOIN " + "(SELECT " +
                        ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.CONVERSATION_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.USER_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.ROLE +
                        "," + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_SEEN_SEQ + "," +
                        ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_RECEIVED_SEQ + "," + UserConstant.TABLE_NAME +
                        ".* " + " FROM " + ParticipantConstant.TABLE_NAME + " LEFT JOIN " +
                        UserConstant.TABLE_NAME + " ON " + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.USER_ID + " = " + UserConstant.TABLE_NAME + "." +
                        UserConstant.USER_ID + ")" + " AS members ON " +
                        ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + " = " + "members." +
                        ParticipantConstant.CONVERSATION_ID + " WHERE " +
                        ConversationConstant.TABLE_NAME + "." + ConversationConstant.USER_ID +
                        " = ?" + " ORDER BY " + ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + "," + " members." +
                        ParticipantConstant.USER_ID + " ASC";

        String[] selectionArgs = {userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            do {
                String conversationId = cursor.getString(2);
                if (!conversations.isEmpty()) {
                    Conversation lastConv = conversations.get(conversations.size() - 1);
                    String lastConvId = lastConv.getId();
                    if (conversationId.equals(lastConvId)) {
                        User user = getParticipantFromCursor(cursor);
                        if (user != null) {
                            lastConv.addParticipants(user);
                        }
                        continue;
                    }
                }
                Conversation conversation = getConversationFromCursor(cursor);

                List<User> pars = new ArrayList<>();
                User user = getParticipantFromCursor(cursor);
                if (user != null) {
                    pars.add(user);
                    conversation.setParticipants(pars);
                }

                if (Utils.isEmpty(conversation.getOaId())) {
                    conversations.add(conversation);
                } else {
                    if (conversation.getCreator().equals(userId)) {
                        conversations.add(conversation);
                    }
                }

            } while (cursor.moveToNext());
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }

        Collections.sort(conversations, (conversation, t1) -> {
            if (conversation.getUpdateAt() > t1.getUpdateAt()) {
                return -1;
            } else {
                return 1;
            }
        });
        return conversations;
    }

    public List<Conversation> getAllOaConversations(String userId, String oaId) {
        List<Conversation> conversations = new ArrayList<>();
        String query =
                "SELECT " + ConversationConstant.TABLE_NAME + ".*," + " members.*" + " FROM " +
                        ConversationConstant.TABLE_NAME + " LEFT JOIN " + "(SELECT " +
                        ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.CONVERSATION_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.USER_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.ROLE +
                        "," + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_SEEN_SEQ + "," +
                        ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_RECEIVED_SEQ + "," + UserConstant.TABLE_NAME +
                        ".* " + " FROM " + ParticipantConstant.TABLE_NAME + " LEFT JOIN " +
                        UserConstant.TABLE_NAME + " ON " + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.USER_ID + " = " + UserConstant.TABLE_NAME + "." +
                        UserConstant.USER_ID + ")" + " AS members ON " +
                        ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + " = " + "members." +
                        ParticipantConstant.CONVERSATION_ID + " WHERE " +
                        ConversationConstant.TABLE_NAME + "." + ConversationConstant.USER_ID +
                        " = ?" + " AND " + ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.OA_ID + " = ?" + " ORDER BY " +
                        ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + "," + " members." +
                        ParticipantConstant.USER_ID + " ASC";

        String[] selectionArgs = {userId, oaId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            do {
                String conversationId = cursor.getString(2);
                if (!conversations.isEmpty()) {
                    Conversation lastConv = conversations.get(conversations.size() - 1);
                    String lastConvId = lastConv.getId();
                    if (conversationId.equals(lastConvId)) {
                        User user = getParticipantFromCursor(cursor);
                        if (user != null) {
                            lastConv.addParticipants(user);
                        }
                        continue;
                    }
                }
                Conversation conversation = getConversationFromCursor(cursor);
                List<User> pars = new ArrayList<>();
                User user = getParticipantFromCursor(cursor);
                if (user != null) {
                    pars.add(user);
                    conversation.setParticipants(pars);
                }

                conversations.add(conversation);
            } while (cursor.moveToNext());
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }

        Collections.sort(conversations, (conversation, t1) -> {
            if (conversation.getUpdateAt() > t1.getUpdateAt()) {
                return -1;
            } else {
                return 1;
            }
        });
        return conversations;
    }

    public List<Conversation> getConversationByChannelType(String userId, boolean ended, String channelType) {
        List<Conversation> conversations = new ArrayList<>();
        String query =
                "SELECT " + ConversationConstant.TABLE_NAME + ".*," + " members.*" + " FROM " +
                        ConversationConstant.TABLE_NAME + " LEFT JOIN " + "(SELECT " +
                        ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.CONVERSATION_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.USER_ID +
                        "," + ParticipantConstant.TABLE_NAME + "." + ParticipantConstant.ROLE +
                        "," + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_SEEN_SEQ + "," +
                        ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.LAST_MSG_RECEIVED_SEQ + "," + UserConstant.TABLE_NAME +
                        ".* " + " FROM " + ParticipantConstant.TABLE_NAME + " LEFT JOIN " +
                        UserConstant.TABLE_NAME + " ON " + ParticipantConstant.TABLE_NAME + "." +
                        ParticipantConstant.USER_ID + " = " + UserConstant.TABLE_NAME + "." +
                        UserConstant.USER_ID + ")" + " AS members ON " +
                        ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + " = " + "members." +
                        ParticipantConstant.CONVERSATION_ID + " WHERE " +
                        ConversationConstant.TABLE_NAME + "." + ConversationConstant.USER_ID +
                        " = ?" + " AND " + ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.TYPE + " in (" + channelType + ")" + " AND " +
                        ConversationConstant.TABLE_NAME + "." + ConversationConstant.ENDED +
                        " = ?" + " ORDER BY " + ConversationConstant.TABLE_NAME + "." +
                        ConversationConstant.CONVERSATION_ID + "," + " members." +
                        ParticipantConstant.USER_ID + " ASC";

        int argEnded = 0;
        if (ended) {
            argEnded = 1;
        }
        String[] selectionArgs = {userId, String.valueOf(argEnded)};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            do {
                String conversationId = cursor.getString(2);
                if (!conversations.isEmpty()) {
                    Conversation lastConv = conversations.get(conversations.size() - 1);
                    String lastConvId = lastConv.getId();
                    if (conversationId.equals(lastConvId)) {
                        User user = getParticipantFromCursor(cursor);
                        if (user != null) {
                            lastConv.addParticipants(user);
                        }
                        continue;
                    }
                }
                Conversation conversation = getConversationFromCursor(cursor);
                List<User> pars = new ArrayList<>();
                User user = getParticipantFromCursor(cursor);
                if (user != null) {
                    pars.add(user);
                    conversation.setParticipants(pars);
                }
                conversations.add(conversation);
            } while (cursor.moveToNext());
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }

        Collections.sort(conversations, (conversation, t1) -> {
            if (conversation.getUpdateAt() > t1.getUpdateAt()) {
                return -1;
            } else {
                return 1;
            }
        });
        return conversations;
    }

    private Conversation getConversationFromCursor(Cursor cursor) {
        Conversation conversation = new Conversation();
        conversation.setDbId(cursor.getInt(0));
        conversation.setLocalId(cursor.getString(1));
        conversation.setId(cursor.getString(2));
        conversation.setName(cursor.getString(3));
        conversation.setGroup(cursor.getInt(4) == 1);
        conversation.setDistinct(cursor.getInt(5) == 1);
        conversation.setClientId(cursor.getString(6));
        conversation.setUpdateAt(cursor.getLong(7));
        conversation.setTotalUnread(cursor.getInt(11));
        String creator = cursor.getString(12);
        if (!Utils.isEmpty(creator)) {
            conversation.setCreator(creator);
        }
        conversation.setLastTimeNewMsg(cursor.getLong(13));
        conversation.setState(State.getState(cursor.getInt(16)));
        conversation.setLastSequence(cursor.getInt(17));
        conversation.setCreateAt(cursor.getLong(19));
        conversation.setLastMsgSeqReceived(cursor.getLong(21));
        conversation.setPinnedMsgId(cursor.getString(22));
        conversation.setChannelType(ChannelType.getType(cursor.getInt(23)));
        conversation.setEnded(cursor.getInt(24) == 1);
        String oaId = cursor.getString(25);
        if (!Utils.isEmpty(oaId)) {
            conversation.setOaId(oaId);
        }
        conversation.setCustomData(cursor.getString(26));
        conversation.setLastMsgSeqSeen(cursor.getLong(29));
        conversation.setChatStatus(ChatStatus.getStatus(cursor.getInt(30)));

        String lastMsgSenderId = cursor.getString(9);
        String lastMsgSenderName = cursor.getString(27);
        String lastMsgSenderAvatar = cursor.getString(28);

        String text = cursor.getString(8);
        Message.Type lastMsgType = Message.Type.getType(cursor.getInt(10));
        Message.State lastMsgState = Message.State.getState(cursor.getInt(14));
        String lastMsgId = cursor.getString(18);
        String lastMsg = cursor.getString(20);
        conversation.setText(text);

        conversation.setLastMsg(lastMsg);

        try {
            JSONObject lastMsgObj = new JSONObject(conversation.getLastMsg());
            String content = lastMsgObj.optString("content", "");
            if (Utils.isEmpty(content)) {
                content = lastMsgObj.optString("text");
            }
            double latitude = 0;
            double longitude = 0;
            String fileUrl = "";
            String fileName = "";
            long fileLength = 0;
            String fileId = "";
            String thumbnailUrl = "";
            int duration = 0;
            float ratio = 0;
            String stickerName = "";
            String stickerCategory = "";
            String contact = "";
            switch (lastMsgType) {
                case RENAME_CONVERSATION:
                case CREATE_CONVERSATION:
                case RATING:
                case NOTIFICATION:
                    content = lastMsgObj.toString();
                    break;
                case LOCATION:
                    JSONObject locationObject = lastMsgObj.optJSONObject("location");
                    if (locationObject != null) {
                        latitude = locationObject.optDouble("lat");
                        longitude = locationObject.optDouble("lon");
                    }
                    break;
                case CONTACT:
                    JSONObject contactObject = lastMsgObj.optJSONObject("contact");
                    if (contactObject != null) {
                        contact = contactObject.optString("vcard");
                    }
                    break;
                case PHOTO:
                    JSONObject photoObject = lastMsgObj.optJSONObject("photo");
                    if (photoObject != null) {
                        ratio = (float) photoObject.optDouble("ratio", 0);
                        fileUrl = photoObject.optString("filePath");
                        thumbnailUrl = photoObject.optString("thumbnail");
                        fileId = photoObject.optString("file_id");
                    }
                    break;
                case VIDEO:
                    JSONObject videoObject = lastMsgObj.optJSONObject("video");
                    if (videoObject != null) {
                        ratio = (float) videoObject.optDouble("ratio", 0);
                        fileUrl = videoObject.optString("filePath");
                        duration = videoObject.optInt("duration");
                        thumbnailUrl = videoObject.optString("thumbnail");
                        fileId = videoObject.optString("file_id");
                        fileName = videoObject.optString("filename");
                    }
                    break;
                case AUDIO:
                    JSONObject audioObject = lastMsgObj.optJSONObject("audio");
                    if (audioObject != null) {
                        fileUrl = audioObject.optString("filePath");
                        duration = audioObject.optInt("duration");
                        fileId = audioObject.optString("file_id");
                        fileName = audioObject.optString("filename");
                    }
                    break;
                case FILE:
                    JSONObject fileObject = lastMsgObj.optJSONObject("file");
                    if (fileObject != null) {
                        fileUrl = fileObject.optString("filePath");
                        fileName = fileObject.optString("filename");
                        fileLength = fileObject.optLong("length");
                        fileId = fileObject.optString("file_id");
                    }
                    break;
                case STICKER:
                    JSONObject stickerObject = lastMsgObj.optJSONObject("sticker");
                    if (stickerObject != null) {
                        stickerCategory = stickerObject.optString("category");
                        stickerName = stickerObject.optString("name");
                    }
                    break;
            }
            Message message = new Message();
            message.setConversationId(conversation.getId());
            message.setConvLocalId(conversation.getLocalId());
            message.setId(lastMsgId);
            message.setCreatedAt(conversation.getLastTimeNewMsg());
            message.setState(lastMsgState);
            message.setSequence(conversation.getLastSequence());
            message.setText(content);
            message.setType(lastMsgType);
            message.setLatitude(latitude);
            message.setLongitude(longitude);
            message.setFileUrl(fileUrl);
            message.setThumbnailUrl(thumbnailUrl);
            message.setDuration(duration);
            message.setImageRatio(ratio);
            message.setContact(contact);
            message.setFileName(fileName);
            message.setFileLength(fileLength);
            message.setFileId(fileId);
            message.setClientId(conversation.getClientId());
            message.setStickerCategory(stickerCategory);
            message.setStickerName(stickerName);

            User lastSender = new User(lastMsgSenderId);
            lastSender.setName(lastMsgSenderName);
            lastSender.setAvatarUrl(lastMsgSenderAvatar);
            message.setSender(lastSender);

            JSONObject customObject = lastMsgObj.optJSONObject("metadata");
            if (customObject != null) {
                message.setCustomData(customObject);
            }
            conversation.setLastMessage(message);
        } catch (JSONException e) {
            Utils.reportException(DBHandler.class, e);
        }
        return conversation;
    }

    private User getParticipantFromCursor(Cursor cursor) {
        String userId = cursor.getString(32);
        if (userId != null) {
            User user = new User(userId);
            user.setRole(Role.getRole(cursor.getString(33)));
            user.setLastMsgSeqSeen(cursor.getLong(34));
            user.setLastMsgSeqReceived(cursor.getLong(35));
            user.setName(cursor.getString(38));
            user.setAvatarUrl(cursor.getString(39));
            user.setEmail(cursor.getString(40));
            user.setPhone(cursor.getString(41));
            user.setLocation(cursor.getString(42));
            user.setBrowser(cursor.getString(43));
            user.setPlatform(cursor.getString(44));
            user.setDevice(cursor.getString(45));
            user.setIpAddress(cursor.getString(46));
            user.setHostName(cursor.getString(47));
            user.setUserAgent(cursor.getString(48));
            return user;
        }
        return null;
    }

    /**
     * Delete conversation by conversation's id
     */
    public void deleteConversationByConvId(String convId, String userId) {
        String query = "DELETE FROM " + ConversationConstant.TABLE_NAME + " WHERE " +
                ConversationConstant.CONVERSATION_ID + " = ?" + " AND " +
                ConversationConstant.USER_ID + " = ?";
        String[] selectionArgs = {convId, userId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    /**
     * Get last sequence of message in conversation
     */
    public long getLastMsgSeqInConversation(String convId, String userId) {
        long seq = 0;
        String query =
                "SELECT " + MessageConstant.SEQUENCE + " FROM " + MessageConstant.TABLE_NAME +
                        " WHERE " + MessageConstant.CONVERSATION_ID + " = ?" + " AND " +
                        MessageConstant.USER_ID + " = ?" + " ORDER BY " + MessageConstant.SEQUENCE +
                        " DESC LIMIT 1";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            seq = cursor.getLong(0);
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }

        return seq;
    }

//    public void updateTotalUnread(String convId, String userId, int num) {
//        String query = "UPDATE " + ConversationConstant.TABLE_NAME
//                + " SET "
//                + ConversationConstant.TOTAL_UNREAD + " = ?"
//                + " WHERE " + ConversationConstant.CONVERSATION_ID + " = ?"
//                + " AND " + ConversationConstant.USER_ID + " = ?";
//        String[] selectionArgs = {String.valueOf(num), convId, userId};
//        writeDatabase.execSQL(query, selectionArgs);
//    }
//
//    public void decreaseTotalUnread(String convId, String userId, int num) {
//        String query = "UPDATE " + ConversationConstant.TABLE_NAME
//                + " SET "
//                + ConversationConstant.TOTAL_UNREAD + " = (" + ConversationConstant.TOTAL_UNREAD + " - ?)"
//                + " WHERE " + ConversationConstant.CONVERSATION_ID + " = ?"
//                + " AND " + ConversationConstant.USER_ID + " = ?";
//        String[] selectionArgs = {String.valueOf(num), convId, userId};
//        writeDatabase.execSQL(query, selectionArgs);
//    }

    //********************* Message ************************//

    /**
     * Save a message to local db
     */
    public void insertMessage(Message message) {
        ContentValues values = getMessageContentValues(message);
        values.put(MessageConstant.LOCAL_ID, message.getLocalId());
        values.put(MessageConstant.CONV_LOCAL_ID, message.getConversationLocalId());
        values.put(MessageConstant.AUTHOR, message.getSender().getUserId());
        values.put(MessageConstant.CREATED_AT, message.getCreatedOnLocalAt());
        values.put(MessageConstant.TYPE, message.getType().getValue());
        values.put(MessageConstant.THUMBNAIL, message.getThumbnail());
        values.put(MessageConstant.LATITUDE, message.getLatitude());
        values.put(MessageConstant.LONGITUDE, message.getLongitude());
        values.put(MessageConstant.ADDRESS, message.getAddress());
        values.put(MessageConstant.MESSAGE_TYPE, message.getMsgType().getValue());
        values.put(MessageConstant.FILE_PATH, message.getFilePath());
        values.put(MessageConstant.FILE_URL, message.getFileUrl());
        values.put(MessageConstant.THUMBNAIL_URL, message.getThumbnailUrl());
        values.put(MessageConstant.DURATION, message.getDuration());
        values.put(MessageConstant.RATIO, message.getImageRatio());
        values.put(MessageConstant.CONTACT, message.getContact());
        values.put(MessageConstant.FILE_PATH, message.getFilePath());
        values.put(MessageConstant.FILE_NAME, message.getFileName());
        values.put(MessageConstant.FILE_LENGTH, message.getFileLength());
        values.put(MessageConstant.USER_ID, message.getClientId());
        values.put(MessageConstant.STICKER_CATEGORY, message.getStickerCategory());
        values.put(MessageConstant.STICKER_NAME, message.getStickerName());
        if (message.getCustomData() != null) {
            values.put(MessageConstant.CUSTOM_DATA, message.getCustomData().toString());
        }
        values.put(MessageConstant.FILE_ID, message.getFileId());

        writeDatabase.insert(MessageConstant.TABLE_NAME, null, values);
    }

    /**
     * Update a message by local id
     */
    public void updateMessageByLocalId(Message message) {
        ContentValues values = getMessageContentValues(message);
        String selection = MessageConstant.LOCAL_ID + " = ?";
        String[] selectionArgs = {message.getLocalId()};
        writeDatabase.update(MessageConstant.TABLE_NAME, values, selection, selectionArgs);
    }

    /**
     * Update a message by remote id
     */
    public void updateMessageByMsgId(Message message) {
        ContentValues values = getMessageContentValues(message);
        String selection =
                MessageConstant.MESSAGE_ID + " = ? AND " + MessageConstant.USER_ID + " = ?";
        String[] selectionArgs = {message.getId(), message.getClientId()};
        writeDatabase.update(MessageConstant.TABLE_NAME, values, selection, selectionArgs);
    }

    private ContentValues getMessageContentValues(Message message) {
        ContentValues values = new ContentValues();
        values.put(MessageConstant.MESSAGE_ID, message.getId());
        values.put(MessageConstant.CONVERSATION_ID, message.getConversationId());
        values.put(MessageConstant.SERVER_CREATED_AT, message.getCreatedAt());
        values.put(MessageConstant.UPDATE_AT, message.getUpdateAt());
        values.put(MessageConstant.STATE, message.getState().getValue());
        values.put(MessageConstant.SEQUENCE, message.getSequence());
        values.put(MessageConstant.TEXT, message.getText());
        values.put(MessageConstant.AUTHOR_NAME, message.getSender().getName());
        values.put(MessageConstant.AUTHOR_AVATAR, message.getSender().getAvatarUrl());
        values.put(MessageConstant.DELETED, message.isDeleted() ? 1 : 0);
        return values;
    }

    public void updateLocalConvIdForMsg(String oldLocalConvId, String newLocalConvId) {
        String query =
                "UPDATE " + MessageConstant.TABLE_NAME + " SET " + MessageConstant.CONV_LOCAL_ID +
                        " = ?" + " WHERE " + MessageConstant.CONV_LOCAL_ID + " = ? ";
        String[] selectionArgs = {newLocalConvId, oldLocalConvId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    public void updateFileUrl(Message message) {
        String query =
                "UPDATE " + MessageConstant.TABLE_NAME + " SET " + MessageConstant.FILE_PATH +
                        " = ?," + MessageConstant.THUMBNAIL + " = ?," + MessageConstant.FILE_URL +
                        " = ?" + " WHERE " + MessageConstant.LOCAL_ID + " = ?" + " AND " +
                        MessageConstant.USER_ID + " = ?";
        String[] selectionArgs =
                {message.getFilePath(), message.getThumbnail(), message.getFileUrl(), message.getLocalId(), message.getClientId()};
        writeDatabase.execSQL(query, selectionArgs);
    }

    public void updateFilePath(Message message) {
        String query =
                "UPDATE " + MessageConstant.TABLE_NAME + " SET " + MessageConstant.FILE_PATH +
                        " = ?," + MessageConstant.THUMBNAIL + " = ?," + MessageConstant.FILE_URL +
                        " = ?" + " WHERE " + MessageConstant.MESSAGE_ID + " = ?" + " AND " +
                        MessageConstant.USER_ID + " = ?";
        String[] selectionArgs =
                {message.getFilePath(), message.getThumbnail(), message.getFileUrl(), message.getId(), message.getClientId()};
        writeDatabase.execSQL(query, selectionArgs);
    }

    /**
     * Get a message by local id
     */
    public Message getMessageByMsgId(String convId, String id, String userId, boolean isLocal) {
        String query = "SELECT *" + " FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                MessageConstant.CONVERSATION_ID + " = ?" + " AND " + MessageConstant.USER_ID +
                " = ?";
        if (isLocal) {
            query = query + " AND " + MessageConstant.LOCAL_ID + " = ?";
        } else {
            query = query + " AND " + MessageConstant.MESSAGE_ID + " = ?";
        }

        String[] selectionArgs = {convId, userId, id};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            Message message = getMessageFromCursor(cursor);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return message;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return null;
    }

    /**
     * Get last messages by count
     */
    public List<Message> getLastLocalMessages(int count, String convId, String userId) {
        List<Message> messages = new ArrayList<>();
        String query = "SELECT *" + " FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                MessageConstant.CONVERSATION_ID + " = ?" + " AND " + MessageConstant.USER_ID +
                " = ?" + " ORDER BY " + MessageConstant.SERVER_CREATED_AT + " DESC LIMIT ?";

        String[] selectionArgs = {convId, userId, String.valueOf(count)};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            do {
                Message message = getMessageFromCursor(cursor);
                messages.add(message);
            } while (cursor.moveToNext());
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        Collections.sort(messages, (message, t1) -> {
            long createdTime = message.getCreatedAt();
            if (createdTime == 0) {
                createdTime = message.getCreatedOnLocalAt();
            }
            long createdTime1 = t1.getCreatedAt();
            if (createdTime1 == 0) {
                createdTime1 = t1.getCreatedOnLocalAt();
            }
            if (createdTime > createdTime1) {
                return 1;
            } else {
                return -1;
            }
        });
        return messages;
    }

//    public List<Message> getLocalMessagesBefore(int count, String convId, String userId, long seq) {
//        List<Message> messages = new ArrayList<>();
//        String query = "SELECT *"
//                + " FROM " + MessageConstant.TABLE_NAME
//                + " WHERE " + MessageConstant.CONVERSATION_ID + " = ?"
//                + " AND " + MessageConstant.USER_ID + " = ?"
//                + " AND " + MessageConstant.SEQUENCE + " < ?"
//                + " ORDER BY " + MessageConstant.SERVER_CREATED_AT
//                + " DESC LIMIT ?";
//
//        String[] selectionArgs = {convId, userId, String.valueOf(seq), String.valueOf(count)};
//        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
//        if (cursor.moveToFirst()) {
//            do {
//                Message message = getMessageFromCursor(cursor);
//                messages.add(message);
//            } while (cursor.moveToNext());
//        }
//        if (!cursor.isClosed()) {
//            cursor.close();
//        }
//        Collections.sort(messages, (message, t1) -> {
//            long createdTime = message.getCreatedAt();
//            if (createdTime == 0) {
//                createdTime = message.getCreatedOnLocalAt();
//            }
//            long createdTime1 = t1.getCreatedAt();
//            if (createdTime1 == 0) {
//                createdTime1 = t1.getCreatedOnLocalAt();
//            }
//            if (createdTime > createdTime1) {
//                return 1;
//            } else {
//                return -1;
//            }
//        });
//        return messages;
//    }

    /**
     * Get last messages by count, local conversation id
     */
//    public List<Message> getLastMessagesByLocalConvId(int count, String localId) {
//        List<Message> messages = new ArrayList<>();
//        String query = "SELECT *"
//                + " FROM " + MessageConstant.TABLE_NAME
//                + " WHERE " + MessageConstant.CONV_LOCAL_ID + " = ?"
//                + " ORDER BY " + MessageConstant.CREATED_AT
//                + " DESC LIMIT ?";
//
//        String[] selectionArgs = {localId, String.valueOf(count)};
//        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
//        if (cursor.moveToFirst()) {
//            do {
//                Message message = getMessageFromCursor(cursor);
//                messages.add(message);
//            } while (cursor.moveToNext());
//        }
//        if (!cursor.isClosed()) {
//            cursor.close();
//        }
//        Collections.sort(messages, (message, t1) -> {
//            long createdTime = message.getCreatedAt();
//            if (createdTime == 0) {
//                createdTime = message.getCreatedOnLocalAt();
//            }
//            long createdTime1 = t1.getCreatedAt();
//            if (createdTime1 == 0) {
//                createdTime1 = t1.getCreatedOnLocalAt();
//            }
//            if (createdTime > createdTime1) {
//                return 1;
//            } else {
//                return -1;
//            }
//        });
//        return messages;
//    }

    /**
     * Update messages state
     */
    public void updateMessagesState(int state, long lastMsgSeq, String convId, String userId, int messageType) {
        int finalState = 0;
        switch (state) {
            case Constant.MESSAGE_STATE_DELIVERED:
                finalState = Message.State.DELIVERED.getValue();
                break;
            case Constant.MESSAGE_STATE_READ:
                finalState = Message.State.READ.getValue();
                break;
        }
        String query =
                "UPDATE " + MessageConstant.TABLE_NAME + " SET " + MessageConstant.STATE + " = ? " +
                        " WHERE " + MessageConstant.MESSAGE_TYPE + " = ? " + " AND " +
                        MessageConstant.CONVERSATION_ID + " = ? " + " AND " +
                        MessageConstant.USER_ID + " = ? " + " AND " + MessageConstant.SEQUENCE +
                        " <= ? " + " AND " + MessageConstant.STATE + " < ? " + " AND " +
                        MessageConstant.SEQUENCE + "  <> 0";
        String[] selectionArgs =
                {String.valueOf(finalState), String.valueOf(messageType), convId, userId, String.valueOf(lastMsgSeq), String.valueOf(finalState)};
        writeDatabase.execSQL(query, selectionArgs);
    }

    /**
     * Get message by sequence
     */
    public Message getMessageBySequence(String convId, String userId, long seq) {
        String query = "SELECT *" + " FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                MessageConstant.CONVERSATION_ID + " = ?" + " AND " + MessageConstant.USER_ID +
                " = ?" + " AND " + MessageConstant.SEQUENCE + " = ? ";

        String[] selectionArgs = {convId, userId, String.valueOf(seq)};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            Message message = getMessageFromCursor(cursor);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return message;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return null;
    }

    /**
     * Get the last received message
     */
    public Message getLastMessage(String convId, String userId) {
        String query = "SELECT *" + " FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                MessageConstant.CONVERSATION_ID + " = ?" + " AND " + MessageConstant.USER_ID +
                " = ?" + " ORDER BY " + MessageConstant.SERVER_CREATED_AT + " DESC LIMIT 1";

        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            Message message = getMessageFromCursor(cursor);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return message;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return null;
    }

    private Message getMessageFromCursor(Cursor cursor) {
        Message message = new Message();
        message.setDbId(cursor.getInt(0));
        message.setId(cursor.getString(1));
        message.setLocalId(cursor.getString(2));
        message.setConvLocalId(cursor.getString(3));
        message.setConversationId(cursor.getString(4));
        message.setCreatedOnLocalAt(cursor.getLong(6));
        message.setCreatedAt(cursor.getLong(7));
        message.setUpdateAt(cursor.getLong(8));
        message.setState(Message.State.getState(cursor.getInt(9)));
        message.setType(Message.Type.getType(cursor.getInt(10)));
        message.setFilePath(cursor.getString(11));
        message.setThumbnail(cursor.getString(12));
        message.setLatitude(cursor.getDouble(13));
        message.setLongitude(cursor.getDouble(14));
        message.setAddress(cursor.getString(15));
        message.setSequence(cursor.getLong(16));
        message.setMsgType(MsgType.getType(cursor.getInt(17)));
        message.setText(cursor.getString(18));
        message.setFileUrl(cursor.getString(19));
        message.setThumbnailUrl(cursor.getString(20));
        message.setDuration(cursor.getInt(21));
        message.setImageRatio(cursor.getFloat(22));
        message.setContact(cursor.getString(23));
        message.setFileName(cursor.getString(24));
        message.setFileLength(cursor.getLong(25));
        message.setClientId(cursor.getString(26));
        message.setStickerCategory(cursor.getString(27));
        message.setStickerName(cursor.getString(28));
        String customData = cursor.getString(29);
        if (!Utils.isEmpty(customData)) {
            try {
                message.setCustomData(new JSONObject(customData));
            } catch (JSONException e) {
                Utils.reportException(DBHandler.class, e);
            }
        }
        message.setDeleted(cursor.getInt(30) == 1);
        String senderId = cursor.getString(5);
        String senderName = cursor.getString(31);
        String senderAvatar = cursor.getString(32);
        User sender = new User(senderId);
        sender.setName(senderName);
        sender.setAvatarUrl(senderAvatar);
        message.setSender(sender);
        message.setFileId(cursor.getString(33));
        return message;
    }

//    public int getTotalUnread(String convId, String userId, long seq) {
//        String query = "SELECT"
//                + " *"
//                + " FROM " + MessageConstant.TABLE_NAME
//                + " WHERE " + MessageConstant.CONVERSATION_ID + " = ?"
//                + " AND " + MessageConstant.USER_ID + " = ?"
//                + " AND " + MessageConstant.SEQUENCE + " <= ?"
//                + " AND " + MessageConstant.MESSAGE_TYPE + " = ?"
//                + " AND " + MessageConstant.STATE + " < ?";
//        String[] selectionArgs = {convId, userId, String.valueOf(seq), String.valueOf(MsgType.RECEIVE.getValue()), String.valueOf(Message.State.READ.getValue())};
//        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
//        int count = cursor.getCount();
//        if (!cursor.isClosed()) {
//            cursor.close();
//        }
//        return count;
//    }

    public long getMaxTime(String convId, String userId, String[] msgIds) {
        StringBuilder query = new StringBuilder(
                "SELECT " + MessageConstant.SERVER_CREATED_AT + " FROM " +
                        MessageConstant.TABLE_NAME + " WHERE " + MessageConstant.CONVERSATION_ID +
                        " = ?" + " AND " + MessageConstant.USER_ID + " = ?" + " AND " +
                        MessageConstant.MESSAGE_ID + " IN(");

        for (int i = 0; i < msgIds.length; i++) {
            query.append("?,");
        }
        query = new StringBuilder(query.substring(0, query.length() - 1));
        query.append(") ORDER BY " + MessageConstant.SERVER_CREATED_AT + " DESC");

        List<String> list = new ArrayList<>();
        list.add(convId);
        list.add(userId);
        Collections.addAll(list, msgIds);
        String[] selectionArgs = list.toArray(new String[0]);

        Cursor cursor = readDatabase.rawQuery(query.toString(), selectionArgs);
        if (cursor.moveToFirst()) {
            long maxTime = cursor.getLong(0);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return maxTime;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return 0;
    }

    public long getConvMaxTime(String convId, String userId) {
        String query = "SELECT " + MessageConstant.SERVER_CREATED_AT + " FROM " +
                MessageConstant.TABLE_NAME + " WHERE " + MessageConstant.CONVERSATION_ID + " = ?" +
                " AND " + MessageConstant.USER_ID + " = ?" + " ORDER BY " +
                MessageConstant.SERVER_CREATED_AT + " DESC LIMIT 1";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            long maxTime = cursor.getLong(0);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return maxTime;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return 0;
    }

    //********************* Participants ************************//

    public boolean isParticipantExist(String convId, String userId) {
        String query = "SELECT" + " *" + " FROM " + ParticipantConstant.TABLE_NAME + " WHERE " +
                ParticipantConstant.CONVERSATION_ID + " = ?" + " AND " +
                ParticipantConstant.USER_ID + " = ? ";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return true;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return false;
    }


    /**
     * Save participant last msg received, seen
     */
    public void insertParticipant(String convId, String userId, long lastMsgReceivedSeq, long lastMsgSeenSeq, Role role) {
        ContentValues values = new ContentValues();
        values.put(ParticipantConstant.CONVERSATION_ID, convId);
        values.put(ParticipantConstant.USER_ID, userId);
        values.put(ParticipantConstant.LAST_MSG_RECEIVED_SEQ, lastMsgReceivedSeq);
        values.put(ParticipantConstant.LAST_MSG_SEEN_SEQ, lastMsgSeenSeq);
        values.put(ParticipantConstant.ROLE, role.getValue());
        writeDatabase.insert(ParticipantConstant.TABLE_NAME, null, values);
    }

//    public void insertParticipant(String convId, User user) {
//        ContentValues values = new ContentValues();
//        values.put(ParticipantConstant.CONVERSATION_ID, convId);
//        values.put(ParticipantConstant.USER_ID, user.getUserId());
//        values.put(ParticipantConstant.LAST_MSG_RECEIVED_SEQ, user.getLastMsgSeqReceived());
//        values.put(ParticipantConstant.LAST_MSG_SEEN_SEQ, user.getLastMsgSeqSeen());
//        values.put(ParticipantConstant.ROLE, user.getRole().getValue());
//        writeDatabase.insert(ParticipantConstant.TABLE_NAME, null, values);
//    }

    /**
     * Save participants
     */
//    public void saveParticipants(String convId, JSONArray jsonArray) {
//        StringBuilder query = new StringBuilder("INSERT INTO " + ParticipantConstant.TABLE_NAME
//                + "("
//                + ParticipantConstant.CONVERSATION_ID + ","
//                + ParticipantConstant.USER_ID + ","
//                + ParticipantConstant.LAST_MSG_RECEIVED_SEQ + ","
//                + ParticipantConstant.LAST_MSG_SEEN_SEQ + ","
//                + ParticipantConstant.ROLE
//                + ") VALUES");
//        int length = jsonArray.length();
//        String[] args = new String[5 * length];
//        for (int i = 0; i < length; i++) {
//            query.append("(?,?,?,?,?)");
//            if (i == length - 1) {
//                query.append(";");
//            } else {
//                query.append(",");
//            }
//
//            try {
//                JSONObject jsonObject = jsonArray.getJSONObject(i);
//                args[4 * i] = convId;
//                args[4 * i + 1] = jsonObject.getString("user");
//                args[4 * i + 2] = "0";
//                args[4 * i + 3] = "0";
//                args[4 * i + 4] = jsonObject.getString("role");
//            } catch (JSONException e) {
//                Utils.reportException(DBHandler.class, e);
//            }
//        }
//        writeDatabase.execSQL(query.toString(), args);
//    }

    /**
     * Update participant last msg received, seen
     */
    public void updateParticipant(String convId, String userId, long lastMsgReceivedSeq, long lastMsgSeenSeq) {
        String query = "UPDATE " + ParticipantConstant.TABLE_NAME + " SET " +
                ParticipantConstant.LAST_MSG_RECEIVED_SEQ + " = ?, " +
                ParticipantConstant.LAST_MSG_SEEN_SEQ + " = ?" + " WHERE " +
                ParticipantConstant.CONVERSATION_ID + " = ?" + " AND " +
                ParticipantConstant.USER_ID + " =? ";
        String[] selectionArgs =
                {String.valueOf(lastMsgReceivedSeq), String.valueOf(lastMsgSeenSeq), convId, userId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    /**
     * Update last msg received
     */
    public void updateLastMsgReceived(String convId, String userId, long lastMsgReceivedSeq) {
        String query = "UPDATE " + ParticipantConstant.TABLE_NAME + " SET " +
                ParticipantConstant.LAST_MSG_RECEIVED_SEQ + " = ?" + " WHERE " +
                ParticipantConstant.CONVERSATION_ID + " = ?" + " AND " +
                ParticipantConstant.USER_ID + " =? ";
        String[] selectionArgs = {String.valueOf(lastMsgReceivedSeq), convId, userId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    /**
     * Update last msg seen
     */
    public void updateLastMsgSeen(String convId, String userId, long lastMsgSeenSeq) {
        String query = "UPDATE " + ParticipantConstant.TABLE_NAME + " SET " +
                ParticipantConstant.LAST_MSG_SEEN_SEQ + " = ?" + " WHERE " +
                ParticipantConstant.CONVERSATION_ID + " = ?" + " AND " +
                ParticipantConstant.USER_ID + " =? ";
        String[] selectionArgs = {String.valueOf(lastMsgSeenSeq), convId, userId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    public void deleteAllParticipants(String convId) {
        String query = "DELETE FROM " + ParticipantConstant.TABLE_NAME + " WHERE " +
                ParticipantConstant.CONVERSATION_ID + " = ?";
        String[] selectionArgs = {convId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    public void deleteParticipants(String convId, String[] userIds) {
        StringBuilder query = new StringBuilder(
                "DELETE FROM " + ParticipantConstant.TABLE_NAME + " WHERE " +
                        ParticipantConstant.CONVERSATION_ID + " = ?" + " AND " +
                        ParticipantConstant.USER_ID + " IN (");
        for (int i = 0; i < userIds.length; i++) {
            query.append("?,");
        }
        query = new StringBuilder(query.substring(0, query.length() - 1));
        query.append(")");

        List<String> list = new ArrayList<>();
        list.add(convId);
        Collections.addAll(list, userIds);
        String[] selectionArgs = list.toArray(new String[0]);
        writeDatabase.execSQL(query.toString(), selectionArgs);
    }

    public long getLastMsgReceivedSeq(String convId, String userId) {
        String query = "SELECT " + ParticipantConstant.LAST_MSG_RECEIVED_SEQ + " FROM " +
                ParticipantConstant.TABLE_NAME + " WHERE " + ParticipantConstant.CONVERSATION_ID +
                " = ?" + " AND " + ParticipantConstant.USER_ID + " <> ?" + " ORDER BY " +
                ParticipantConstant.LAST_MSG_RECEIVED_SEQ + " DESC";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            long lastMsgReceiveSeq = cursor.getLong(0);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return lastMsgReceiveSeq;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return 0;
    }

    public long getLastMsgSeenSeq(String convId, String userId) {
        String query = "SELECT " + ParticipantConstant.LAST_MSG_SEEN_SEQ + " FROM " +
                ParticipantConstant.TABLE_NAME + " WHERE " + ParticipantConstant.CONVERSATION_ID +
                " = ?" + " AND " + ParticipantConstant.USER_ID + " <> ?" + " ORDER BY " +
                ParticipantConstant.LAST_MSG_SEEN_SEQ + " DESC";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            long lastMsgSeenSeq = cursor.getLong(0);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return lastMsgSeenSeq;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return 0;
    }

    public long getMyLastMsgReceivedSeq(String convId, String userId) {
        String query = "SELECT " + ParticipantConstant.LAST_MSG_RECEIVED_SEQ + " FROM " +
                ParticipantConstant.TABLE_NAME + " WHERE " + ParticipantConstant.CONVERSATION_ID +
                " = ?" + " AND " + ParticipantConstant.USER_ID + " = ?" + " ORDER BY " +
                ParticipantConstant.LAST_MSG_RECEIVED_SEQ + " DESC";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            long myLastMsgReceiveSeq = cursor.getLong(0);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return myLastMsgReceiveSeq;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return 0;
    }

    public long getMyLastMsgSeenSeq(String convId, String userId) {
        String query = "SELECT " + ParticipantConstant.LAST_MSG_SEEN_SEQ + " FROM " +
                ParticipantConstant.TABLE_NAME + " WHERE " + ParticipantConstant.CONVERSATION_ID +
                " = ?" + " AND " + ParticipantConstant.USER_ID + " = ?" + " ORDER BY " +
                ParticipantConstant.LAST_MSG_SEEN_SEQ + " DESC";
        String[] selectionArgs = {convId, userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            long myLastMsgSeenSeq = cursor.getLong(0);
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return myLastMsgSeenSeq;
        }
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return 0;
    }

    /**
     * Delete all messages of a conversation
     */
    public void deleteMessagesByConvId(String convId, String userId) {
        String query = "DELETE FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                MessageConstant.CONVERSATION_ID + " = ?" + " AND " + MessageConstant.USER_ID +
                " = ?";
        String[] selectionArgs = {convId, userId};
        writeDatabase.execSQL(query, selectionArgs);
    }

    public void deleteMessages(String convId, String userId, String[] msgIds) {
        StringBuilder query = new StringBuilder(
                "DELETE FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                        MessageConstant.CONVERSATION_ID + " = ?" + " AND " +
                        MessageConstant.USER_ID + " =?" + " AND " + MessageConstant.MESSAGE_ID +
                        " IN (");
        for (int i = 0; i < msgIds.length; i++) {
            query.append("?,");
        }
        query = new StringBuilder(query.substring(0, query.length() - 1));
        query.append(")");

        List<String> list = new ArrayList<>();
        list.add(convId);
        list.add(userId);
        Collections.addAll(list, msgIds);
        String[] selectionArgs = list.toArray(new String[0]);
        writeDatabase.execSQL(query.toString(), selectionArgs);
    }

    public void deleteLocalMessages(String convId, String userId, String[] msgIds) {
        StringBuilder query = new StringBuilder(
                "DELETE FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                        MessageConstant.CONVERSATION_ID + " = ?" + " AND " +
                        MessageConstant.USER_ID + " =?" + " AND " + MessageConstant.LOCAL_ID +
                        " IN (");
        for (int i = 0; i < msgIds.length; i++) {
            query.append("?,");
        }
        query = new StringBuilder(query.substring(0, query.length() - 1));
        query.append(")");

        List<String> list = new ArrayList<>();
        list.add(convId);
        list.add(userId);
        Collections.addAll(list, msgIds);
        String[] selectionArgs = list.toArray(new String[0]);
        writeDatabase.execSQL(query.toString(), selectionArgs);
    }

    public int getTotalUnread(String userId) {
        String query = "SELECT" + " *" + " FROM " + MessageConstant.TABLE_NAME + " WHERE " +
                MessageConstant.USER_ID + " = ?" + " AND " + MessageConstant.MESSAGE_TYPE + " = ?" +
                " AND " + MessageConstant.STATE + " < ? ";
        String[] selectionArgs =
                {userId, String.valueOf(MsgType.RECEIVE.getValue()), String.valueOf(Message.State.READ.getValue())};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        int count = cursor.getCount();
        if (!cursor.isClosed()) {
            cursor.close();
        }
        return count;
    }

    //********************* User ************************//

    /**
     * Save user to db
     */
    public void insertUser(User user) {
        ContentValues values = new ContentValues();
        values.put(UserConstant.USER_ID, user.getUserId());
        values.put(UserConstant.FULL_NAME, user.getName());
        values.put(UserConstant.AVATAR, user.getAvatarUrl());
        values.put(UserConstant.EMAIL, user.getEmail());
        values.put(UserConstant.PHONE, user.getPhone());
        values.put(UserConstant.LOCATION, user.getLocation());
        values.put(UserConstant.BROWSER, user.getBrowser());
        values.put(UserConstant.PLATFORM, user.getPlatform());
        values.put(UserConstant.DEVICE, user.getDevice());
        values.put(UserConstant.IP_ADDRESS, user.getIpAddress());
        values.put(UserConstant.HOST_NAME, user.getHostName());
        values.put(UserConstant.USER_AGENT,
                user.getUserAgent() != null ? user.getUserAgent() : null);

        writeDatabase.insert(UserConstant.TABLE_NAME, null, values);
    }

    /**
     * Update user
     */
    public void updateUser(User user) {
        ContentValues values = new ContentValues();
        if (user.getName() != null) {
            values.put(UserConstant.FULL_NAME, user.getName());
        }
        if (user.getAvatarUrl() != null) {
            values.put(UserConstant.AVATAR, user.getAvatarUrl());
        }
        if (user.getEmail() != null) {
            values.put(UserConstant.EMAIL, user.getEmail());
        }
        if (user.getPhone() != null) {
            values.put(UserConstant.PHONE, user.getPhone());
        }
        if (user.getLocation() != null) {
            values.put(UserConstant.LOCATION, user.getLocation());
        }
        if (user.getBrowser() != null) {
            values.put(UserConstant.BROWSER, user.getBrowser());
        }
        if (user.getPlatform() != null) {
            values.put(UserConstant.PLATFORM, user.getPlatform());
        }
        if (user.getDevice() != null) {
            values.put(UserConstant.DEVICE, user.getDevice());
        }
        if (user.getIpAddress() != null) {
            values.put(UserConstant.IP_ADDRESS, user.getIpAddress());
        }
        if (user.getHostName() != null) {
            values.put(UserConstant.HOST_NAME, user.getHostName());
        }
        if (user.getUserAgent() != null) {
            values.put(UserConstant.USER_AGENT, user.getUserAgent());
        }
        if (values.size() != 0) {
            String[] args = {user.getUserId()};
            writeDatabase.update(UserConstant.TABLE_NAME, values,
                    UserConstant.USER_ID + " = ?", args);
        }
    }

//    public void insertOrUpdateUser(User user) {
//        ContentValues values = new ContentValues();
//        values.put(UserConstant.USER_ID, user.getUserId());
//        values.put(UserConstant.FULL_NAME, user.getName());
//        values.put(UserConstant.AVATAR, user.getAvatarUrl());
//        values.put(UserConstant.EMAIL, user.getEmail());
//        values.put(UserConstant.PHONE, user.getPhone());
//        try {
//            long rowId = writeDatabase.insertWithOnConflict(UserConstant.TABLE_NAME, null, values, SQLiteDatabase.CONFLICT_REPLACE);
//            if (rowId == -1) {
//                // Insert failed, try updating
//                writeDatabase.update(UserConstant.TABLE_NAME, values, UserConstant.USER_ID + " = ?", new String[]{user.getUserId()});
//            }
//        } catch (Exception e) {
//            Utils.reportException(DBHandler.class, e);
//        }
//    }

    /**
     * Get a user
     */
    public User getUser(String userId) {
        String query = "SELECT" + " *" + " FROM " + UserConstant.TABLE_NAME + " WHERE " +
                UserConstant.USER_ID + " = ? ";
        String[] selectionArgs = {userId};
        Cursor cursor = readDatabase.rawQuery(query, selectionArgs);
        if (cursor.moveToFirst()) {
            User user = new User(userId);
            user.setName(cursor.getString(2));
            user.setAvatarUrl(cursor.getString(3));
            user.setEmail(cursor.getString(4));
            user.setPhone(cursor.getString(5));
            user.setLocation(cursor.getString(6));
            user.setBrowser(cursor.getString(7));
            user.setPlatform(cursor.getString(8));
            user.setDevice(cursor.getString(9));
            user.setIpAddress(cursor.getString(10));
            user.setHostName(cursor.getString(11));
            user.setUserAgent(cursor.getString(12));
            if (!cursor.isClosed()) {
                cursor.close();
            }
            return user;
        }

        return null;
    }

    /**
     * Clear local db
     */
    public void clearDb() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM " + ConversationConstant.TABLE_NAME);
        db.execSQL("DELETE FROM " + MessageConstant.TABLE_NAME);
        db.execSQL("DELETE FROM " + ParticipantConstant.TABLE_NAME);
        db.execSQL("DELETE FROM " + UserConstant.TABLE_NAME);
    }

    public void syncConversationToLocalDb(String currentUserId, Conversation conversation) {
        long lastConvReceivedSeq = 0;
        long lastConvSeenSeq = 0;

        List<User> participants = conversation.getParticipants();
        if (!Utils.isEmpty(participants)) {
            for (int i = 0; i < participants.size(); i++) {
                User user = participants.get(i);
                syncUser(user);

                boolean isParExist =
                        dbHandler.isParticipantExist(conversation.getId(), user.getUserId());
                if (isParExist) {
                    dbHandler.updateParticipant(conversation.getId(), user.getUserId(), user.getLastMsgSeqReceived(), user.getLastMsgSeqSeen());
                } else {
                    dbHandler.insertParticipant(conversation.getId(), user.getUserId(), user.getLastMsgSeqReceived(), user.getLastMsgSeqSeen(), user.getRole());
                }

                if (!user.getUserId().equals(currentUserId)) {
                    if (user.getLastMsgSeqReceived() > lastConvReceivedSeq) {
                        lastConvReceivedSeq = user.getLastMsgSeqReceived();
                    }
                    if (user.getLastMsgSeqSeen() > lastConvSeenSeq) {
                        lastConvSeenSeq = user.getLastMsgSeqSeen();
                    }
                }
            }
        }

        updateMessagesState(Constant.MESSAGE_STATE_DELIVERED, lastConvReceivedSeq, conversation.getId(), currentUserId, MsgType.SEND.getValue());
        updateMessagesState(Constant.MESSAGE_STATE_READ, lastConvSeenSeq, conversation.getId(), currentUserId, MsgType.SEND.getValue());

        Conversation localConversation =
                getConversationByRemoteId(conversation.getId(), currentUserId);
        if (localConversation == null) {
            conversation.setDbId((int) insertConversation(conversation));
        } else {
            conversation.setDbId(localConversation.getDbId());
            if (localConversation.getUpdateAt() <= conversation.getUpdateAt()) {
                updateConversation(conversation);
            }
            String localId2 = localConversation.getLocalId();
            if (localId2 != null && !conversation.getLocalId().equals(localId2)) {
                updateLocalConvIdForMsg(localId2, conversation.getLocalId());
            }
        }
    }

    public void syncUser(User user) {
        if (getUser(user.getUserId()) == null) {
            insertUser(user);
        } else {
            updateUser(user);
        }
    }

    public void syncMessage(Message message) {
        if (getMessageByMsgId(message.getConversationId(), message.getId(), message.getClientId(), false) !=
                null) {
            updateMessageByMsgId(message);
        } else {
            insertMessage(message);
        }
    }
}
