package com.stringee.network.processor;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.Constant;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.event.EventManager;
import com.stringee.database.DBHandler;
import com.stringee.messaging.Conversation;
import com.stringee.messaging.Message;
import com.stringee.messaging.Message.MsgType;
import com.stringee.messaging.Message.Type;
import com.stringee.messaging.StringeeChange;
import com.stringee.messaging.User;
import com.stringee.messaging.User.Role;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.network.tcpclient.packet.Packet;

import org.json.JSONObject;

public class ChatMessageFromServer extends ProcessorBase {
    @Override
    public void process(final StringeeClient client, final Packet packet) {
        client.executeExecutor(() -> {
            packet.decodeJson();
            DBHandler dbHandler = DBHandler.getInstance(client.getContext());
            String from = packet.optFieldString("from");
            String name = packet.optFieldString("displayName");
            String avatar = packet.optFieldString("avatarUrl");
            String role = packet.optFieldString("role");
            User sender = new User(from);
            sender.setName(name);
            sender.setAvatarUrl(avatar);

            String convId = packet.optFieldString("convId");
            String msgId = packet.optFieldString("msgId");
            long createdTime = packet.optFieldLong("createdTime");
            long seq = packet.optFieldLong("seq");
            JSONObject messageObject = packet.optFieldJSONObject("message");

            MsgType msgType = MsgType.RECEIVE;
            if (from.equals(client.getUserId())) {
                msgType = MsgType.SEND;
            }

            client.executeDbExecutor(() -> {
                dbHandler.syncUser(sender);
                MsgType msgType1 = MsgType.RECEIVE;
                if (from.equals(client.getUserId())) {
                    msgType1 = MsgType.SEND;
                }
                if (!dbHandler.isParticipantExist(convId, from)) {
                    dbHandler.insertParticipant(convId, from, seq, seq, Role.getRole(role));
                }

                if (!dbHandler.isParticipantExist(convId, client.getUserId())) {
                    dbHandler.insertParticipant(convId, client.getUserId(), seq, 0, Role.MEMBER);
                } else {
                    if (msgType1 == MsgType.RECEIVE) {
                        dbHandler.updateLastMsgReceived(convId, client.getUserId(), seq);
                    }
                }
            });

            if (messageObject != null) {
                String text;
                double latitude = 0;
                double longitude = 0;
                String fileUrl = "";
                String fileName = "";
                long fileLength = 0;
                String fileId = "";
                String thumbnailUrl = "";
                int duration = 0;
                float ratio = 0;
                String stickerName = "";
                String stickerCategory = "";
                String contact = "";
                text = messageObject.optString("content", "");
                Type type = Type.getType(packet.optFieldInt("type"));
                switch (type) {
                    case RENAME_CONVERSATION:
                    case CREATE_CONVERSATION:
                    case RATING:
                    case NOTIFICATION:
                        text = messageObject.toString();
                        break;
                    case LOCATION:
                        JSONObject locationObject = messageObject.optJSONObject("location");
                        if (locationObject != null) {
                            latitude = locationObject.optDouble("lat");
                            longitude = locationObject.optDouble("lon");
                        }
                        break;
                    case CONTACT:
                        JSONObject contactObject = messageObject.optJSONObject("contact");
                        if (contactObject != null) {
                            contact = contactObject.optString("vcard");
                        }
                        break;
                    case PHOTO:
                        JSONObject photoObject = messageObject.optJSONObject("photo");
                        if (photoObject != null) {
                            ratio = (float) photoObject.optDouble("ratio");
                            fileUrl = photoObject.optString("filePath");
                            thumbnailUrl = photoObject.optString("thumbnail");
                            fileId = photoObject.optString("file_id");
                        }
                        break;
                    case VIDEO:
                        JSONObject videoObject = messageObject.optJSONObject("video");
                        if (videoObject != null) {
                            ratio = (float) videoObject.optDouble("ratio");
                            fileUrl = videoObject.optString("filePath");
                            duration = videoObject.optInt("duration");
                            thumbnailUrl = videoObject.optString("thumbnail");
                            fileId = videoObject.optString("file_id");
                            fileName = videoObject.optString("filename");
                        }
                        break;
                    case AUDIO:
                        JSONObject audioObject = messageObject.optJSONObject("audio");
                        if (audioObject != null) {
                            fileUrl = audioObject.optString("filePath");
                            duration = audioObject.optInt("duration");
                            fileId = audioObject.optString("file_id");
                            fileName = audioObject.optString("filename");
                        }
                        break;
                    case FILE:
                        JSONObject fileObject = messageObject.optJSONObject("file");
                        if (fileObject != null) {
                            fileUrl = fileObject.optString("filePath");
                            fileName = fileObject.optString("filename");
                            fileLength = fileObject.optLong("length");
                            fileId = fileObject.optString("file_id");
                        }
                        break;
                    case STICKER:
                        JSONObject stickerObject = messageObject.optJSONObject("sticker");
                        if (stickerObject != null) {
                            stickerCategory = stickerObject.optString("category");
                            stickerName = stickerObject.optString("name");
                        }
                        break;
                }

                Message message = new Message();
                message.setConversationId(convId);
                message.setId(msgId);
                message.setCreatedAt(createdTime);
                message.setSender(sender);
                message.setSequence(seq);
                message.setText(text);
                message.setType(type);
                message.setMsgType(msgType);
                if (msgType == MsgType.RECEIVE) {
                    message.setState(Message.State.DELIVERED);
                } else {
                    message.setState(Message.State.SENT);
                }
                message.setLatitude(latitude);
                message.setLongitude(longitude);
                message.setFileUrl(fileUrl);
                message.setThumbnailUrl(thumbnailUrl);
                message.setDuration(duration);
                message.setImageRatio(ratio);
                message.setContact(contact);
                message.setFileName(fileName);
                message.setFileLength(fileLength);
                message.setFileId(fileId);
                message.setClientId(client.getUserId());
                message.setStickerCategory(stickerCategory);
                message.setStickerName(stickerName);
                JSONObject customObject = messageObject.optJSONObject("metadata");
                if (customObject != null) {
                    message.setCustomData(customObject);
                }

                // Send delivered report
                if (msgType == MsgType.RECEIVE) {
                    int requestId;
                    synchronized (Common.lock) {
                        requestId = ++Common.requestId;
                    }
                    SendPacketUtils.sendMessageReport(client, requestId, message, Constant.MESSAGE_STATE_DELIVERED);
                }

                client.executeDbExecutor(() -> dbHandler.insertMessage(message));
                EventManager.sendChatChangeEvent(client, new StringeeChange(StringeeChange.Type.INSERT, message));
                client.getConversationFromServer(convId, new CallbackListener<Conversation>() {
                    @Override
                    public void onSuccess(Conversation conversation) {
                        Conversation localConv = dbHandler.getConversationByRemoteId(convId, client.getUserId());
                        dbHandler.syncConversationToLocalDb(client.getUserId(), conversation);
                        EventManager.sendChatChangeEvent(client, new StringeeChange(localConv != null ? StringeeChange.Type.UPDATE : StringeeChange.Type.INSERT, conversation));
                    }
                });
            }
        });
    }
}
