package com.stringee.network.tcpclient.ssl;

import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.List;
import java.util.StringTokenizer;

/**
 * @author Alex
 */
public class SunHostnameChecker {

    // constants for subject alt names of type DNS and IP
    private final static int ALTNAME_DNS = 2;

    /**
     * Perform the check.
     *
     * @param expectedName the expected name
     * @param cert         the certificate to check
     * @throws CertificateException if the name does not match any of the names specified in the certificate
     */
    public void match(String expectedName, X509Certificate cert) throws CertificateException {
        matchDNS(expectedName, cert);
    }

    /**
     * Check if the certificate allows use of the given DNS name.
     * <p/>
     * From RFC2818: If a subjectAltName extension of type dNSName is present, that MUST be used as the identity.
     * Otherwise, the (most specific) Common Name field in the Subject field of the certificate MUST be used. Although
     * the use of the Common Name is existing practice, it is deprecated and Certification Authorities are encouraged to
     * use the dNSName instead.
     * <p/>
     * Matching is performed using the matching rules specified by [RFC2459]. If more than one identity of a given type
     * is present in the certificate (e.g., more than one dNSName name, a match in any one of the set is considered
     * acceptable.)
     */
    private void matchDNS(String expectedName, X509Certificate cert) throws CertificateException {
        Collection<List<?>> subjAltNames = cert.getSubjectAlternativeNames();
        if (subjAltNames != null) {
            boolean foundDNS = false;
            for (List<?> next : subjAltNames) {
                if (((Integer) next.get(0)) == ALTNAME_DNS) {
                    foundDNS = true;
                    String dnsName = (String) next.get(1);

                    if (isMatched(expectedName, dnsName)) {
                        return;
                    }
                }
            }
            if (foundDNS) {
                // if certificate contains any subject alt names of type DNS
                // but none match, reject
                throw new CertificateException("No subject alternative DNS " + "name matching " + expectedName + " found.");
            }
        }

        String msg = "No name matching " + expectedName + " found";
        throw new CertificateException(msg);
    }

    /**
     * Returns true if name matches against template.<p>
     * <p/>
     * The matching is performed as per RFC 2818 rules for TLS and RFC 2830 rules for LDAP.<p>
     * <p/>
     * The <code>name</code> parameter should represent a DNS name. The <code>template</code> parameter may contain the
     * wildcard character
     *
     * @param name     the name to match
     * @param template the template to match against
     * @return true if name matches against template
     */
    public boolean isMatched(String name, String template) {
        return matchAllWildcards(name, template);
    }

    /**
     * Returns true if name matches against template.<p>
     * <p/>
     * According to RFC 2818, section 3.1 - Names may contain the wildcard character * which is considered to match any
     * single domain name component or component fragment. E.g., *.a.com matches foo.a.com but not bar.foo.a.com. f*.com
     * matches foo.com but not bar.com.
     */
    private static boolean matchAllWildcards(String name, String template) {
        name = name.toLowerCase();
        template = template.toLowerCase();
        StringTokenizer nameSt = new StringTokenizer(name, ".");
        StringTokenizer templateSt = new StringTokenizer(template, ".");

        if (nameSt.countTokens() != templateSt.countTokens()) {
            return false;
        }

        while (nameSt.hasMoreTokens()) {
            if (!matchWildCards(nameSt.nextToken(),
                    templateSt.nextToken())) {
                return false;
            }
        }
        return true;
    }

    /**
     * Returns true if the name matches against the template that may contain wildcard char
     * <p/>
     */
    private static boolean matchWildCards(String name, String template) {
        int wildcardIdx = template.indexOf("*");
        if (wildcardIdx == -1) {
            return name.equals(template);
        }

        boolean isBeginning = true;
        String beforeWildcard;
        String afterWildcard = template;

        while (wildcardIdx != -1) {
            // match in sequence the non-wildcard chars in the template.
            beforeWildcard = afterWildcard.substring(0, wildcardIdx);
            afterWildcard = afterWildcard.substring(wildcardIdx + 1);

            int beforeStartIdx = name.indexOf(beforeWildcard);
            if ((beforeStartIdx == -1)
                    || (isBeginning && beforeStartIdx != 0)) {
                return false;
            }
            isBeginning = false;

            // update the match scope
            name = name.substring(beforeStartIdx + beforeWildcard.length());
            wildcardIdx = afterWildcard.indexOf("*");
        }
        return name.endsWith(afterWildcard);
    }
}
