package com.stringee.video;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.Utils;
import com.stringee.listener.StatusListener;
import com.stringee.listener.StringeeRoomListener;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by luannguyen on 10/19/2017.
 */

public class StringeeRoom {
    private String id;
    private String token;
    private State state;
    private boolean isRecorded;
    private LocalParticipant localParticipant;
    private final ConcurrentHashMap<String, RemoteParticipant> participantMap = new ConcurrentHashMap<>();
    private StringeeRoomListener listener;
    private final StringeeClient client;
    private final ConcurrentHashMap<String, StringeeVideoTrack> videoTrackMap = new ConcurrentHashMap<>();

    public StringeeRoom(StringeeClient client) {
        this.client = client;
        this.state = State.DISCONNECTED;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public boolean isRecorded() {
        return isRecorded;
    }

    public void setRecorded(boolean recorded) {
        isRecorded = recorded;
    }

    public State getState() {
        return state;
    }

    public void setState(State state) {
        this.state = state;
    }

    public LocalParticipant getLocalParticipant() {
        return localParticipant;
    }

    public void setLocalParticipant(LocalParticipant localParticipant) {
        this.localParticipant = localParticipant;
    }

    public Map<String, RemoteParticipant> getParticipantMap() {
        return participantMap;
    }

    /**
     * Get all remote participants
     *
     * @return List<RemoteParticipant>
     */
    public List<RemoteParticipant> getRemoteParticipants() {
        return new ArrayList<>(participantMap.values());
    }

    public RemoteParticipant getRemoteParticipant(String id) {
        return participantMap.get(id);
    }

    public void addRemoteParticipant(RemoteParticipant remoteParticipant) {
        participantMap.put(remoteParticipant.getId(), remoteParticipant);
    }

    public void setListener(StringeeRoomListener listener) {
        this.listener = listener;
    }

    public StringeeRoomListener getListener() {
        return listener;
    }

    public StringeeClient getClient() {
        return client;
    }

    public ConcurrentHashMap<String, StringeeVideoTrack> getVideoTrackMap() {
        return videoTrackMap;
    }

    public List<StringeeVideoTrack> getVideoTracks() {
        return new ArrayList<>(videoTrackMap.values());
    }

    /**
     * Get remote video tracks
     *
     * @return List<StringeeVideoTrack>
     */
    public List<StringeeVideoTrack> getRemoteVideoTracks() {
        List<StringeeVideoTrack> videoTracks = getVideoTracks();
        if (!videoTracks.isEmpty()) {
            for (int i = videoTracks.size() - 1; i >= 0; i--) {
                StringeeVideoTrack videoTrack = videoTracks.get(i);
                if (videoTrack.isLocal()) {
                    videoTracks.remove(i);
                }
            }
        }
        return videoTracks;
    }

    /**
     * Get local video tracks
     *
     * @return List<StringeeVideoTrack>
     */
    public List<StringeeVideoTrack> getLocalVideoTracks() {
        List<StringeeVideoTrack> videoTracks = getVideoTracks();
        if (!videoTracks.isEmpty()) {
            for (int i = videoTracks.size() - 1; i >= 0; i--) {
                StringeeVideoTrack videoTrack = videoTracks.get(i);
                if (videoTrack.isLocal()) {
                    videoTracks.remove(i);
                }
            }
        }
        return videoTracks;
    }

    /**
     * Publish a video track
     *
     * @param videoTrack video track
     * @param listener   callback listener
     */
    public void publish(StringeeVideoTrack videoTrack, StatusListener listener) {
        videoTrack.setClient(client);
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        videoTrackMap.put(videoTrack.getLocalId(), videoTrack);
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        client.getLocalVideoTrackMap().put(requestId, videoTrack);
        SendPacketUtils.publishTrack(client, requestId, id, videoTrack);
    }

    /**
     * Publish a video track with options
     *
     * @param videoTrack video track
     * @param options    options
     * @param listener   callback listener
     */
    public void publish(StringeeVideoTrack videoTrack, StringeeVideoTrack.Options options, StatusListener listener) {
        videoTrack.setAudio(options.audioEnabled());
        videoTrack.setVideo(options.videoEnabled());
        videoTrack.setScreenCapture(options.isScreenCapture());
        this.publish(videoTrack, listener);
    }

    /**
     * Unpublish a video track
     *
     * @param videoTrack video track
     * @param listener   callback listener
     */
    public void unpublish(StringeeVideoTrack videoTrack, StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.unpublish(client, requestId, id, videoTrack.getId());
    }

    /**
     * Subscribe a video track
     *
     * @param videoTrack video track
     * @param options    options
     * @param listener   callback listener
     */
    public void subscribe(StringeeVideoTrack videoTrack, StringeeVideoTrack.Options options, StatusListener listener) {
        boolean audio = true;
        boolean video = false;
        boolean screen = false;
        if (options != null) {
            audio = options.audioEnabled();
            video = options.videoEnabled();
            screen = options.isScreenCapture();
        }
        videoTrack.setAudio(audio);
        videoTrack.setVideo(video);
        videoTrack.setScreenCapture(screen);
        String localId = Utils.getDeviceId(client.getContext()) + "-" + videoTrack.getId() + "-" + System.currentTimeMillis();
        videoTrack.setLocalId(localId);
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.subscribeTrack(client, requestId, id, videoTrack);
    }

    /**
     * Unsubscribe a video track
     *
     * @param videoTrack video track
     * @param listener   callback listener
     */
    public void unsubscribe(StringeeVideoTrack videoTrack, StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.unsubscribe(client, requestId, id, videoTrack);
    }

    /**
     * Leave a room
     *
     * @param listener  callback listener
     * @param allClient leave all clients
     */
    public void leave(boolean allClient, StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.leaveRoom(client, requestId, id, allClient);
    }

    /**
     * Send a message to the room
     *
     * @param msg      message
     * @param listener callback listener
     */
    public void sendMessage(JSONObject msg, StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendRoomMsg(client, requestId, id, msg);
    }

    public enum State {
        CONNECTING,
        CONNECTED,
        RECONNECTING,
        DISCONNECTED
    }
}
