package com.stringee.video;

import android.content.Context;
import android.graphics.Bitmap;

import com.stringee.call.StringeeCallFactory;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.video.StringeeVideoTrack.Options;

import org.webrtc.AudioSource;
import org.webrtc.AudioTrack;
import org.webrtc.CapturerObserver;
import org.webrtc.DefaultVideoDecoderFactory;
import org.webrtc.DefaultVideoEncoderFactory;
import org.webrtc.EglBase;
import org.webrtc.MediaConstraints;
import org.webrtc.MediaStream;
import org.webrtc.PeerConnection;
import org.webrtc.PeerConnection.Observer;
import org.webrtc.PeerConnection.RTCConfiguration;
import org.webrtc.PeerConnectionFactory;
import org.webrtc.RendererCommon.RendererEvents;
import org.webrtc.RendererCommon.ScalingType;
import org.webrtc.ScreenCapturerAndroid;
import org.webrtc.SurfaceTextureHelper;
import org.webrtc.SurfaceViewRenderer;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoDecoderFactory;
import org.webrtc.VideoEncoderFactory;
import org.webrtc.VideoFrame;
import org.webrtc.VideoSource;
import org.webrtc.VideoTrack;

public class StringeeVideoFactory {
    private static final String AUDIO_ECHO_CANCELLATION_CONSTRAINT = "googEchoCancellation";
    private static final String AUDIO_EXPERIMENTAL__ECHO_CANCELLATION_CONSTRAINT = "googExperimentalEchoCancellation";
    private static final String AUDIO_AUTO_GAIN_CONTROL_CONSTRAINT = "googAutoGainControl";
    private static final String AUDIO_EXPERIMENTAL_AUTO_GAIN_CONTROL_CONSTRAINT = "googExperimentalAutoGainControl";
    private static final String AUDIO_HIGH_PASS_FILTER_CONSTRAINT = "googHighpassFilter";
    private static final String AUDIO_NOISE_SUPPRESSION_CONSTRAINT = "googNoiseSuppression";
    private static final String AUDIO_EXPERIMENTAL_NOISE_SUPPRESSION_CONSTRAINT = "googExperimentalNoiseSuppression";
    private static final String AUDIO_MIRROR_CONSTRAINT = "googAudioMirroring";
    private static final String AUDIO_TRACK_ID = "ARDAMSa0";
    private static final String VIDEO_TRACK_ID = "ARDAMSv0";

    private static EglBase eglBase;
    private static EglBase.Context rootEglBaseContext;
    private static PeerConnectionFactory sFactory;

    private static StringeeVideoFactory videoFactory;

    private VideoFrame snapshotLocalFrame;
    private VideoFrame snapshotScreenFrame;
    private CallbackListener<Bitmap> snapshotLocalListener;
    private CallbackListener<Bitmap> snapshotScreenListener;

    public static synchronized StringeeVideoFactory getInstance(Context context) {
        if (videoFactory == null) {
            videoFactory = new StringeeVideoFactory();
            PeerConnectionFactory.InitializationOptions.Builder optionBuilder =
                    PeerConnectionFactory.InitializationOptions.builder(context.getApplicationContext());
            PeerConnectionFactory.initialize(optionBuilder.createInitializationOptions());

            eglBase = EglBase.create();
            rootEglBaseContext = eglBase.getEglBaseContext();
            VideoEncoderFactory encoderFactory =
                    new DefaultVideoEncoderFactory(
                            rootEglBaseContext, false, false);
            VideoDecoderFactory decoderFactory =
                    new DefaultVideoDecoderFactory(rootEglBaseContext);
            sFactory = PeerConnectionFactory.builder().setVideoEncoderFactory(encoderFactory).setVideoDecoderFactory(decoderFactory).createPeerConnectionFactory();
        }
        return videoFactory;
    }

    /**
     * Create local video track
     *
     * @param context    context
     * @param videoTrack video track
     * @param options    options
     */
    public void createLocalVideoTrack(Context context, StringeeVideoTrack videoTrack, Options options) {
        MediaConstraints audioConstraints = new MediaConstraints();
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_ECHO_CANCELLATION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_EXPERIMENTAL__ECHO_CANCELLATION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_AUTO_GAIN_CONTROL_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_EXPERIMENTAL_AUTO_GAIN_CONTROL_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_HIGH_PASS_FILTER_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_NOISE_SUPPRESSION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_EXPERIMENTAL_NOISE_SUPPRESSION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(
                new MediaConstraints.KeyValuePair(AUDIO_MIRROR_CONSTRAINT, "false"));

        MediaStream mediaStream = sFactory.createLocalMediaStream("ARDAMS" + System.currentTimeMillis());
        videoTrack.setMediaStream(mediaStream);
        if (options.audioEnabled()) {
            AudioSource audioSource = sFactory.createAudioSource(audioConstraints);
            AudioTrack localAudioTrack = sFactory.createAudioTrack(AUDIO_TRACK_ID + System.currentTimeMillis(), audioSource);
            videoTrack.setLocalAudioTrack(localAudioTrack);
            videoTrack.setAudioSource(audioSource);
            mediaStream.addTrack(localAudioTrack);
        }
        if (options.videoEnabled()) {
            VideoCapturer videoCapturer = StringeeCallFactory.createVideoCapturer(context);
            SurfaceTextureHelper surfaceTextureHelper =
                    SurfaceTextureHelper.create("CaptureThread", rootEglBaseContext);
            VideoSource videoSource = sFactory.createVideoSource(videoCapturer.isScreencast());
            int minWidth = VideoDimensions.WVGA_VIDEO_WIDTH;
            int minHeight = VideoDimensions.WVGA_VIDEO_HEIGHT;
            if (options.getVideoDimensions() != null) {
                minWidth = options.getVideoDimensions().width;
                minHeight = options.getVideoDimensions().height;
            }
            videoCapturer.initialize(surfaceTextureHelper, context, new CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean b) {
                    videoSource.getCapturerObserver().onCapturerStarted(b);
                    if (videoTrack.getCaptureSessionListener() != null) {
                        videoTrack.getCaptureSessionListener().onCapturerStarted(StringeeVideoTrack.TrackType.CAMERA);
                    }
                }

                @Override
                public void onCapturerStopped() {
                    videoSource.getCapturerObserver().onCapturerStopped();
                    if (videoTrack.getCaptureSessionListener() != null) {
                        videoTrack.getCaptureSessionListener().onCapturerStopped(StringeeVideoTrack.TrackType.CAMERA);
                    }
                }

                @Override
                public void onFrameCaptured(VideoFrame videoFrame) {
                    videoSource.getCapturerObserver().onFrameCaptured(videoFrame);
                    if (snapshotLocalListener != null) {
                        if (snapshotLocalFrame != null) {
                            return;
                        }
                        snapshotLocalFrame = videoFrame;

                        Bitmap bitmap = Utils.generateBitmap(snapshotLocalFrame);
                        Bitmap emptyBitmap = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), bitmap.getConfig());
                        if (!bitmap.sameAs(emptyBitmap)) {
                            snapshotLocalListener.onSuccess(bitmap);
                        } else {
                            snapshotLocalListener.onError(new StringeeError(1, "Snapshot false"));
                        }
                        snapshotLocalListener = null;
                        snapshotLocalFrame = null;
                    }
                }
            });
            videoCapturer.startCapture(minWidth, minHeight, 30);
            VideoTrack localVideoTrack = sFactory.createVideoTrack(VIDEO_TRACK_ID + System.currentTimeMillis(), videoSource);
            videoTrack.setLocalVideoTrack(localVideoTrack);
            videoTrack.setVideoSource(videoSource);
            videoTrack.setVideoCapturer(videoCapturer);
            mediaStream.addTrack(localVideoTrack);
        }
    }

    public void createCapture(Context context, StringeeVideoTrack videoTrack, VideoDimensions videoDimensions, ScreenCapturerAndroid videoCapturer) {
        MediaStream mediaStream = sFactory.createLocalMediaStream("ARDAMS" + System.currentTimeMillis());
        videoTrack.setMediaStream(mediaStream);
        SurfaceTextureHelper surfaceTextureHelper =
                SurfaceTextureHelper.create("CaptureScreenThread", rootEglBaseContext);

        VideoSource videoSource = sFactory.createVideoSource(videoCapturer.isScreencast());
        videoCapturer.initialize(surfaceTextureHelper, context, new CapturerObserver() {
            @Override
            public void onCapturerStarted(boolean b) {
                videoSource.getCapturerObserver().onCapturerStarted(b);
                if (videoTrack.getCaptureSessionListener() != null) {
                    videoTrack.getCaptureSessionListener().onCapturerStarted(StringeeVideoTrack.TrackType.SCREEN);
                }
            }

            @Override
            public void onCapturerStopped() {
                videoSource.getCapturerObserver().onCapturerStopped();
                if (videoTrack.getCaptureSessionListener() != null) {
                    videoTrack.getCaptureSessionListener().onCapturerStopped(StringeeVideoTrack.TrackType.SCREEN);
                }
            }

            @Override
            public void onFrameCaptured(VideoFrame videoFrame) {
                videoSource.getCapturerObserver().onFrameCaptured(videoFrame);
                if (snapshotScreenListener != null) {
                    if (snapshotScreenFrame != null) {
                        return;
                    }
                    snapshotScreenFrame = videoFrame;

                    Bitmap bitmap = Utils.generateBitmap(snapshotScreenFrame);
                    Bitmap emptyBitmap = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), bitmap.getConfig());
                    if (!bitmap.sameAs(emptyBitmap)) {
                        snapshotScreenListener.onSuccess(bitmap);
                    } else {
                        snapshotScreenListener.onError(new StringeeError(1, "Snapshot false"));
                    }
                    snapshotScreenListener = null;
                    snapshotScreenFrame = null;
                }
            }
        });
        videoCapturer.startCapture(videoDimensions.width, videoDimensions.height, 30);
        VideoTrack localVideoTrack = sFactory.createVideoTrack(VIDEO_TRACK_ID + System.currentTimeMillis(), videoSource);
        videoTrack.setLocalVideoTrack(localVideoTrack);
        videoTrack.setVideoSource(videoSource);
        videoTrack.setVideoCapturer(videoCapturer);

        mediaStream.addTrack(localVideoTrack);
    }

    /**
     * Create a peer connection
     *
     * @param rtcConfiguration rtc configuration
     * @param observer         observer
     * @return peer connection
     */
    public synchronized PeerConnection createPeerConnection(RTCConfiguration rtcConfiguration, Observer observer) {
        return sFactory.createPeerConnection(rtcConfiguration, observer);
    }

    public synchronized void release() {
        if (sFactory != null) {
            sFactory.dispose();
            sFactory = null;
        }

        if (eglBase != null) {
            eglBase.release();
            eglBase = null;
        }

        rootEglBaseContext = null;
        videoFactory = null;
    }


    public void renderView(VideoTrack videoTrack, SurfaceViewRenderer renderer, boolean isOverlay, ScalingType scalingType, RendererEvents rendererEvents) {
        renderer.release();
        renderer.setEnableHardwareScaler(true);
        renderer.init(rootEglBaseContext, rendererEvents);
        if (scalingType != null) {
            switch (scalingType) {
                case SCALE_ASPECT_FIT:
                    renderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);
                    break;
                case SCALE_ASPECT_FILL:
                    renderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);
                    break;
                case SCALE_ASPECT_BALANCED:
                    renderer.setScalingType(ScalingType.SCALE_ASPECT_BALANCED);
                    break;
            }
        }
        renderer.setZOrderMediaOverlay(isOverlay);
        videoTrack.addSink(renderer);
    }

    public void renderView2(VideoTrack videoTrack, TextureViewRenderer renderer, ScalingType scalingType, RendererEvents rendererEvents) {
        renderer.release();
        renderer.setEnableHardwareScaler(true);
        renderer.init(rootEglBaseContext, rendererEvents);
        if (scalingType != null) {
            switch (scalingType) {
                case SCALE_ASPECT_FIT:
                    renderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);
                    break;
                case SCALE_ASPECT_FILL:
                    renderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);
                    break;
                case SCALE_ASPECT_BALANCED:
                    renderer.setScalingType(ScalingType.SCALE_ASPECT_BALANCED);
                    break;
            }
        }
        videoTrack.addSink(renderer);
    }

    public void snapshotLocal(CallbackListener<Bitmap> snapshotLocalListener) {
        this.snapshotLocalListener = snapshotLocalListener;
    }

    public void snapshotScreen(CallbackListener<Bitmap> snapshotScreenListener) {
        this.snapshotScreenListener = snapshotScreenListener;
    }
}
