package com.stringee.video;

import android.content.Context;
import android.graphics.Bitmap;

import com.stringee.StringeeClient;
import com.stringee.call.StringeeCallFactory;
import com.stringee.call.StringeeIceCandidate;
import com.stringee.common.Common;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StatusListener;
import com.stringee.messaging.listeners.CallbackListener;

import org.webrtc.AudioSource;
import org.webrtc.AudioTrack;
import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.IceCandidate;
import org.webrtc.MediaConstraints;
import org.webrtc.MediaStream;
import org.webrtc.PeerConnection;
import org.webrtc.PeerConnection.SdpSemantics;
import org.webrtc.RendererCommon.RendererEvents;
import org.webrtc.RendererCommon.ScalingType;
import org.webrtc.ScreenCapturerAndroid;
import org.webrtc.SessionDescription;
import org.webrtc.SurfaceViewRenderer;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoSource;
import org.webrtc.VideoTrack;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;

public class StringeeVideoTrack {
    public enum MediaState {
        CONNECTED(0), DISCONNECTED(1);

        private final short value;

        MediaState(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    public enum TrackType {
        CAMERA(0), SCREEN(1), PLAYER(2);

        private final short value;

        TrackType(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    public enum MediaType {
        AUDIO(1), VIDEO(2);

        private final short value;

        MediaType(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static MediaType getType(int value) {
            if (value == 2) {
                return VIDEO;
            }
            return AUDIO;
        }
    }

    public static class Options {
        private boolean audio = true;
        private boolean video = true;
        private boolean screen = false;
        private VideoDimensions videoDimensions;

        public Options() {
        }

        public boolean audioEnabled() {
            return audio;
        }

        public void audio(boolean audio) {
            this.audio = audio;
        }

        public boolean videoEnabled() {
            return video;
        }

        public void video(boolean video) {
            this.video = video;
        }

        public boolean isScreenCapture() {
            return screen;
        }

        public void screen(boolean screen) {
            this.screen = screen;
        }

        public VideoDimensions getVideoDimensions() {
            return videoDimensions;
        }

        public void videoDimensions(VideoDimensions videoDimensions) {
            this.videoDimensions = videoDimensions;
        }
    }

    public interface Listener {
        void onMediaAvailable();

        void onMediaStateChange(MediaState state);
    }

    public interface CaptureSessionListener {
        void onCapturerStarted(TrackType trackType);

        void onCapturerStopped(TrackType trackType);
    }

    private String id;
    private boolean audio = true;
    private boolean video = true;
    private boolean screen = false;
    private boolean isPlayer = false;
    private String userId;
    private String roomId;
    private String pcId;
    private String localId;
    private StringeeClient client;
    private final LinkedBlockingQueue<StringeeIceCandidate> candidatesQueue = new LinkedBlockingQueue<>();
    private boolean localSDPSet;
    private boolean remoteSDPSet;
    private Listener listener;
    private MediaStream mediaStream;
    private PeerConnection peerConnection;
    private SurfaceViewRenderer renderer;
    private TextureViewRenderer renderer2;
    private StringeeVideoFactory videoFactory;
    private AudioTrack localAudioTrack;
    private VideoTrack localVideoTrack;
    private boolean isLocal;
    private AudioSource audioSource;
    private VideoSource videoSource;
    private VideoCapturer videoCapturer;
    private TrackType trackType = TrackType.CAMERA;
    private CaptureSessionListener captureSessionListener;
    private boolean canSwitch = true;
    private boolean isFrontCamera = true;
    private String frontCameraName = "";
    private String rearCameraName = "";

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public boolean audioEnabled() {
        return audio;
    }

    public void setAudio(boolean audio) {
        this.audio = audio;
    }

    public boolean videoEnabled() {
        return video;
    }

    public void setVideo(boolean video) {
        this.video = video;
    }

    public boolean isScreenCapture() {
        return screen;
    }

    public void setScreenCapture(boolean screen) {
        this.screen = screen;
    }

    public boolean isPlayer() {
        return isPlayer;
    }

    public void setPlayer(boolean player) {
        isPlayer = player;
    }

    public String getUserId() {
        return userId;
    }

    public void setUserId(String userId) {
        this.userId = userId;
    }

    public String getRoomId() {
        return roomId;
    }

    public void setRoomId(String roomId) {
        this.roomId = roomId;
    }

    public String getPcId() {
        return pcId;
    }

    public void setPcId(String pcId) {
        this.pcId = pcId;
    }

    public String getLocalId() {
        return localId;
    }

    public void setLocalId(String localId) {
        this.localId = localId;
    }

    public StringeeClient getClient() {
        return client;
    }

    public void setClient(StringeeClient client) {
        this.client = client;
    }

    public LinkedBlockingQueue<StringeeIceCandidate> getCandidatesQueue() {
        return candidatesQueue;
    }

    public void setLocalSDPSet(boolean localSDPSet) {
        this.localSDPSet = localSDPSet;
    }

    public void setRemoteSDPSet(boolean remoteSDPSet) {
        this.remoteSDPSet = remoteSDPSet;
    }

    public Listener getListener() {
        return listener;
    }

    public void setListener(Listener listener) {
        this.listener = listener;
    }

    public MediaStream getMediaStream() {
        return mediaStream;
    }

    public void setMediaStream(MediaStream mediaStream) {
        this.mediaStream = mediaStream;
    }

    public PeerConnection getPeerConnection() {
        return peerConnection;
    }

    public void setPeerConnection(PeerConnection peerConnection) {
        this.peerConnection = peerConnection;
    }

    public void setLocalAudioTrack(AudioTrack localAudioTrack) {
        this.localAudioTrack = localAudioTrack;
    }

    public void setLocalVideoTrack(VideoTrack localVideoTrack) {
        this.localVideoTrack = localVideoTrack;
    }

    public boolean isLocal() {
        return isLocal;
    }

    public void setLocal(boolean local) {
        isLocal = local;
    }

    public void setAudioSource(AudioSource audioSource) {
        this.audioSource = audioSource;
    }

    public void setVideoSource(VideoSource videoSource) {
        this.videoSource = videoSource;
    }

    public void setVideoCapturer(VideoCapturer videoCapturer) {
        this.videoCapturer = videoCapturer;
    }

    public TrackType getTrackType() {
        return trackType;
    }

    public void setTrackType(TrackType trackType) {
        this.trackType = trackType;
    }

    public CaptureSessionListener getCaptureSessionListener() {
        return captureSessionListener;
    }

    public void setCaptureSessionListener(CaptureSessionListener captureSessionListener) {
        this.captureSessionListener = captureSessionListener;
    }

    /**
     * Create local video track
     *
     * @param context context
     * @param options options
     */
    public void create(Context context, Options options) {
        videoFactory = StringeeVideoFactory.getInstance(context);
        videoFactory.createLocalVideoTrack(context, this, options);

        // get cameraId
        CameraEnumerator enumerator = Camera2Enumerator.isSupported(context) ? new Camera2Enumerator(context) : new Camera1Enumerator();
        String[] cameraNames = enumerator.getDeviceNames();
        if (cameraNames.length > 0) {
            // first front id is main front camera
            for (String name : cameraNames) {
                boolean isFrontFace = enumerator.isFrontFacing(name);
                if (isFrontFace) {
                    frontCameraName = name;
                    break;
                }
            }

            // first rear id is main rear camera
            for (String cameraName : cameraNames) {
                boolean isBackFace = enumerator.isBackFacing(cameraName);
                if (isBackFace) {
                    rearCameraName = cameraName;
                    break;
                }
            }
        }

        canSwitch = (!Utils.isEmpty(frontCameraName) && !Utils.isEmpty(rearCameraName));
    }

    public void createCapture(Context context, VideoDimensions videoDimensions, ScreenCapturerAndroid videoCapturer) {
        videoFactory = StringeeVideoFactory.getInstance(context);
        videoFactory.createCapture(context, this, videoDimensions, videoCapturer);

        canSwitch = false;
    }

    /**
     * Release resource
     */
    public void release() {
        if (client != null) {
            StringeeRoom stringeeRoom = client.getRoomMap().get(roomId);
            if (stringeeRoom != null) {
                stringeeRoom.getVideoTrackMap().remove(id);
                stringeeRoom.getVideoTrackMap().remove(localId);
            }
        }
        if (peerConnection != null) {
            peerConnection.dispose();
            peerConnection = null;
        }

        if (renderer != null) {
            renderer.release();
            renderer = null;
        }

        if (renderer2 != null) {
            renderer2.release();
            renderer2 = null;
        }

        if (isLocal) {
            if (audioSource != null) {
                audioSource.dispose();
                audioSource = null;
            }

            if (videoCapturer != null) {
                try {
                    videoCapturer.stopCapture();
                } catch (Exception e) {
                    Utils.reportException(StringeeVideoTrack.class, e);
                }
                videoCapturer.dispose();
                videoCapturer = null;
            }

            if (videoSource != null) {
                videoSource.dispose();
                videoSource = null;
            }
        }
    }

    /**
     * Return video view
     *
     * @return SurfaceViewRenderer
     */
    public SurfaceViewRenderer getView(Context context) {
        if (renderer == null) {
            renderer = new SurfaceViewRenderer(context);
        }
        return renderer;
    }

    /**
     * Return video view
     */
    public TextureViewRenderer getView2(Context context) {
        if (renderer2 == null) {
            renderer2 = new TextureViewRenderer(context);
        }
        return renderer2;
    }

    /**
     * Render video view
     */
    public void renderView(boolean isOverlay) {
        this.renderView(isOverlay, ScalingType.SCALE_ASPECT_FIT, null);
    }

    /**
     * Render video view
     */
    public void renderView(boolean isOverlay, RendererEvents rendererEvents) {
        this.renderView(isOverlay, ScalingType.SCALE_ASPECT_FIT, rendererEvents);
    }

    /**
     * Render video view
     */
    public void renderView(boolean isOverlay, ScalingType scalingType) {
        this.renderView(isOverlay, scalingType, null);
    }

    /**
     * Render video view with scale
     */
    public void renderView(boolean isOverlay, ScalingType scalingType, RendererEvents rendererEvents) {
        if (isLocal) {
            videoFactory.renderView(localVideoTrack, renderer, isOverlay, scalingType, rendererEvents);
        } else {
            if (mediaStream != null && mediaStream.videoTracks != null && !mediaStream.videoTracks.isEmpty()) {
                videoFactory.renderView(mediaStream.videoTracks.get(0), renderer, isOverlay, scalingType, rendererEvents);
            }
        }
    }

    /**
     * Render video view
     */
    public void renderView2() {
        this.renderView2(ScalingType.SCALE_ASPECT_FIT, null);
    }

    /**
     * Render video view
     */
    public void renderView2(RendererEvents rendererEvents) {
        this.renderView2(ScalingType.SCALE_ASPECT_FIT, rendererEvents);
    }

    /**
     * Render video view
     */
    public void renderView2(ScalingType scalingType) {
        this.renderView2(scalingType, null);
    }

    /**
     * Render video view with scale
     */
    public void renderView2(ScalingType scalingType, RendererEvents rendererEvents) {
        if (isLocal) {
            videoFactory.renderView2(localVideoTrack, renderer2, scalingType, rendererEvents);
        } else {
            if (mediaStream != null && mediaStream.videoTracks != null && !mediaStream.videoTracks.isEmpty()) {
                videoFactory.renderView2(mediaStream.videoTracks.get(0), renderer2, scalingType, rendererEvents);
            }
        }
    }

    /**
     * Mute/unmute the audio
     *
     * @param mute true to mute, false to unmute
     */
    public void mute(boolean mute) {
        if (localAudioTrack != null) {
            boolean enableAudio = !mute;
            localAudioTrack.setEnabled(enableAudio);
        }
    }

    /**
     * Turn video on/off
     *
     * @param enable true to turn on, false to turn off
     */
    public void enableVideo(boolean enable) {
        if (localVideoTrack != null) {
            localVideoTrack.setEnabled(enable);
        }
    }

    /**
     * Switch camera
     *
     * @param listener callback listener
     */
    public void switchCamera(StatusListener listener) {
        if (canSwitch) {
            if (videoCapturer != null && videoCapturer instanceof CameraVideoCapturer) {
                CameraVideoCapturer cameraVideoCapturer = (CameraVideoCapturer) videoCapturer;
                cameraVideoCapturer.switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean b) {
                        isFrontCamera = !isFrontCamera;
                        if (listener != null) {
                            listener.onSuccess();
                        }
                    }

                    @Override
                    public void onCameraSwitchError(String s) {
                        if (listener != null) {
                            listener.onError(new StringeeError(-1, s));
                        }
                    }
                }, isFrontCamera ? rearCameraName : frontCameraName);
            }
        } else {
            if (Utils.isEmpty(frontCameraName) && Utils.isEmpty(rearCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have any cameras"));
            } else if (Utils.isEmpty(rearCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have a rear camera"));
            } else if (Utils.isEmpty(frontCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have a front camera"));
            }
        }
    }

    /**
     * Send audio enable notification
     *
     * @param enable   true to enable, false to disable
     * @param listener callback listener
     */
    public void sendAudioEnableNotification(boolean enable, StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendVideoRoomEnableDisableAudioVideo(client, requestId, roomId, 1, enable, id);
    }

    /**
     * Send video enable notification
     *
     * @param enable   true to enable, false to disable
     * @param listener callback listener
     */
    public void sendVideoEnableNotification(boolean enable, StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendVideoRoomEnableDisableAudioVideo(client, requestId, roomId, 2, enable, id);
    }

    public void switchCamera(StatusListener listener, String cameraName) {
        if (videoCapturer != null && videoCapturer instanceof CameraVideoCapturer) {
            CameraVideoCapturer cameraVideoCapturer = (CameraVideoCapturer) videoCapturer;
            cameraVideoCapturer.switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                @Override
                public void onCameraSwitchDone(boolean b) {
                    if (listener != null) {
                        listener.onSuccess();
                    }
                }

                @Override
                public void onCameraSwitchError(String s) {
                    if (listener != null) {
                        listener.onError(new StringeeError(-1, s));
                    }
                }
            }, cameraName);
        }
    }

    private void publishOrSubscribe(Context context, boolean isPublish) {
        LinkedList<PeerConnection.IceServer> iceServers = StringeeCallFactory.createIceServers(StringeeVideo.getIceServers());
        PeerConnection.RTCConfiguration rtcConfig = new PeerConnection.RTCConfiguration(iceServers);
        rtcConfig.bundlePolicy = PeerConnection.BundlePolicy.BALANCED;
        rtcConfig.rtcpMuxPolicy = PeerConnection.RtcpMuxPolicy.REQUIRE;
        rtcConfig.continualGatheringPolicy = PeerConnection.ContinualGatheringPolicy.GATHER_CONTINUALLY;
        rtcConfig.tcpCandidatePolicy = PeerConnection.TcpCandidatePolicy.DISABLED;
        rtcConfig.sdpSemantics = SdpSemantics.UNIFIED_PLAN;

        videoFactory = StringeeVideoFactory.getInstance(context);
        peerConnection = videoFactory.createPeerConnection(rtcConfig, new VideoObserver(this));
        if (isPublish) {
            List<String> mediaStreamLabels = Collections.singletonList("ARDAMS");
            if (!Utils.isEmpty(mediaStream.audioTracks)) {
                List<AudioTrack> audioTracks = mediaStream.audioTracks;
                for (int i = 0; i < audioTracks.size(); i++) {
                    AudioTrack audioTrack = audioTracks.get(i);
                    peerConnection.addTrack(audioTrack, mediaStreamLabels);
                }
            }
            if (!Utils.isEmpty(mediaStream.videoTracks)) {
                List<VideoTrack> videoTracks = mediaStream.videoTracks;
                for (int i = 0; i < videoTracks.size(); i++) {
                    VideoTrack videoTrack = videoTracks.get(i);
                    peerConnection.addTrack(videoTrack, mediaStreamLabels);
                }
            }
        }

        MediaConstraints sdpMediaConstraints = new MediaConstraints();
        if (audio) {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "true"));
        } else {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "false"));
        }
        if (video || screen) {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "true"));
        } else {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "false"));
        }
        peerConnection.createOffer(new VideoSdpObserver(this, true), sdpMediaConstraints);
    }

    /**
     * Publish local video track to server
     */
    public void publish(Context context) {
        publishOrSubscribe(context, true);
    }

    /**
     * Subscribe a remote video track
     */
    public void subscribe(Context context) {
        publishOrSubscribe(context, false);
    }

    public void setLocalDescription(SessionDescription description, VideoSdpObserver observer) {
        peerConnection.setLocalDescription(observer, description);
    }

    public void setRemoteDescription(SessionDescription description, VideoSdpObserver observer) {
        peerConnection.setRemoteDescription(observer, description);
    }

    public synchronized void addIceCandidate(final StringeeIceCandidate ice) {
        if (localSDPSet && remoteSDPSet) {
            while (!candidatesQueue.isEmpty()) {
                StringeeIceCandidate iceCandidate = candidatesQueue.poll();
                if (iceCandidate != null) {
                    setIceCandidate(iceCandidate);
                }
            }
            setIceCandidate(ice);
        } else {
            candidatesQueue.add(ice);
        }
    }

    public synchronized void setIceCandidate(StringeeIceCandidate iceCandidate) {
        IceCandidate webrtcIceCandidate = new IceCandidate(iceCandidate.sdpMid, iceCandidate.sdpMLineIndex, iceCandidate.sdp);
        peerConnection.addIceCandidate(webrtcIceCandidate);
    }

    public void restartICE() {
        MediaConstraints sdpMediaConstraints = new MediaConstraints();
        sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "true"));
        if (video) {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "true"));
        } else {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "false"));
        }

        sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("IceRestart", "true"));
        VideoSdpObserver videoSdpObserver = new VideoSdpObserver(this, true);
        peerConnection.createOffer(videoSdpObserver, sdpMediaConstraints);
    }

    public void snapshotLocal(CallbackListener<Bitmap> listener) {
        if (videoFactory != null) {
            videoFactory.snapshotLocal(listener);
        }
    }

    public void snapshotScreen(CallbackListener<Bitmap> listener) {
        if (videoFactory != null) {
            videoFactory.snapshotScreen(listener);
        }
    }
}
