package com.stringee.widget;

import android.content.Context;
import android.util.Log;

import com.stringee.StringeeClient;
import com.stringee.call.StringeeCall;
import com.stringee.call.StringeeCall2;
import com.stringee.common.SocketAddress;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StatusListener;
import com.stringee.listener.StringeeConnectionListener;
import com.stringee.widget.call.CallConfig;
import com.stringee.widget.call.StringeeCallWrapper;
import com.stringee.widget.call.receiver.GSMCallStateReceiver;
import com.stringee.widget.common.StringeeCommon;
import com.stringee.widget.common.StringeeConstant;
import com.stringee.widget.common.StringeeNotificationService;
import com.stringee.widget.common.StringeeUtils;

import org.json.JSONObject;
import org.webrtc.ThreadUtils;

import java.util.ArrayList;
import java.util.List;

public class StringeeWidget {
    private static volatile StringeeWidget instance;
    private final Context context;
    private StringeeClient client;
    private StringeeListener listener;
    private final List<SocketAddress> host = new ArrayList<>();

    public StringeeWidget(Context context) {
        this.context = context.getApplicationContext();
    }

    /**
     * Get a single instance
     *
     * @param context application context
     * @return instance
     */
    public static StringeeWidget getInstance(Context context) {
        if (instance == null) {
            synchronized (StringeeWidget.class) {
                if (instance == null) {
                    instance = new StringeeWidget(context.getApplicationContext());
                }
            }
        }
        synchronized (StringeeWidget.class) {
            if (instance.client == null) {
                instance.client = new StringeeClient(context);
            }
        }
        return instance;
    }

    public StringeeClient getClient() {
        return client;
    }

    public StringeeListener getListener() {
        return listener;
    }

    public void setListener(StringeeListener listener) {
        this.listener = listener;
    }

    /**
     * Set host
     *
     * @param addresses list of socket address
     */
    public void setHost(final List<SocketAddress> addresses) {
        if (!addresses.isEmpty()) {
            host.clear();
        }

        host.addAll(addresses);
    }

    public List<SocketAddress> getHost() {
        return host;
    }

    /**
     * Check whether client connect to Stringee server yet
     *
     * @return true if connected, otherwise false
     */
    public boolean isConnected() {
        return client != null && client.isAlreadyConnected();
    }

    private void initStringeeClient() {
        if (client == null) {
            client = new StringeeClient(context);
        }
        if (!StringeeUtils.isListEmpty(host)) {
            client.setHost(host);
        }
        client.setConnectionListener(new StringeeConnectionListener() {
            @Override
            public void onConnectionConnected(StringeeClient stringeeClient, boolean isReconnecting) {
                Log.d(StringeeConstant.TAG, "onConnectionConnected - userId: " + stringeeClient.getUserId() + " - reconnecting: " + isReconnecting);
                if (!isReconnecting && listener != null) {
                    listener.onConnectionConnected();
                }
            }

            @Override
            public void onConnectionDisconnected(StringeeClient stringeeClient, boolean isReconnecting) {
                Log.d(StringeeConstant.TAG, "onConnectionDisconnected - userId: " + stringeeClient.getUserId() + " - reconnecting: " + isReconnecting);
                if (!isReconnecting && listener != null) {
                    listener.onConnectionDisconnected();
                }
            }

            @Override
            public void onIncomingCall(StringeeCall stringeeCall) {
                Log.d(StringeeConstant.TAG, "onIncomingCall - callId: " + stringeeCall.getCallId() + " - from: " + stringeeCall.getFrom() + " - fromAlias: " + stringeeCall.getFromAlias());
                StringeeUtils.runOnUiThread(() -> {
                    if (StringeeCommon.isInCall) {
                        stringeeCall.reject("Reject when in another call", null);
                        return;
                    }
                    StringeeCallWrapper.getInstance(context).initializedIncomingCall(stringeeCall);
                    StringeeCallWrapper.getInstance(context).initAnswer();
                    StringeeNotificationService.getInstance(context).notifyIncomingCall(stringeeCall.getFrom(), stringeeCall.getFromAlias());
                });
            }

            @Override
            public void onIncomingCall2(StringeeCall2 stringeeCall2) {
                Log.d(StringeeConstant.TAG, "onIncomingCall2 - callId: " + stringeeCall2.getCallId() + " - from: " + stringeeCall2.getFrom() + " - fromAlias: " + stringeeCall2.getFromAlias());
                StringeeUtils.runOnUiThread(() -> {
                    if (StringeeCommon.isInCall) {
                        stringeeCall2.reject("Reject when in another call", null);
                        return;
                    }
                    StringeeCallWrapper.getInstance(context).initializedIncomingCall(stringeeCall2);
                    StringeeCallWrapper.getInstance(context).initAnswer();
                    StringeeNotificationService.getInstance(context).notifyIncomingCall(stringeeCall2.getFrom(), stringeeCall2.getFromAlias());
                });
            }

            @Override
            public void onConnectionError(StringeeClient stringeeClient, StringeeError stringeeError) {
                Log.d(StringeeConstant.TAG, "onConnectionError - error: " + stringeeError.getMessage());
                if (listener != null) {
                    listener.onConnectionError(stringeeError);
                }
            }

            @Override
            public void onRequestNewToken(StringeeClient stringeeClient) {
                Log.d(StringeeConstant.TAG, "onRequestNewToken");
                if (listener != null) {
                    listener.onRequestNewToken();
                }
            }

            @Override
            public void onCustomMessage(String from, JSONObject msg) {
                Log.d(StringeeConstant.TAG, "onCustomMessage - from: " + from + " - msg: " + msg.toString());
            }

            @Override
            public void onTopicMessage(String from, JSONObject msg) {
                Log.d(StringeeConstant.TAG, "onTopicMessage - from: " + from + " - msg: " + msg.toString());
            }
        });
    }

    /**
     * Connect Stringee server
     *
     * @param accessToken access token
     */
    public void connect(String accessToken) {
        String userIdFromToken = StringeeUtils.getUserId(accessToken);
        if (StringeeUtils.isStringEmpty(userIdFromToken)) {
            if (listener != null) {
                listener.onConnectionError(new StringeeError(2, "ACCESS_TOKEN_CANNOT_BE_DECODE"));
            }
            return;
        }

        initStringeeClient();

        if (!client.isAlreadyConnected()) {
            client.connect(accessToken);
        }
    }

    /**
     * Disconnect Stringee server
     */
    public void disconnect() {
        if (client != null) {
            client.disconnect();
            client = null;
        }
    }

    /**
     * Connect and make call
     *
     * @param accessToken access token
     * @param callConfig  call configuration
     */
    public void makeCall(String accessToken, CallConfig callConfig, StatusListener listener) {
        ThreadUtils.checkIsOnMainThread();
        String userIdFromToken = StringeeUtils.getUserId(accessToken);
        if (StringeeUtils.isStringEmpty(userIdFromToken)) {
            StringeeError error = new StringeeError(2, "ACCESS_TOKEN_CANNOT_BE_DECODE");
            if (listener != null) {
                listener.onError(error);
            }
            if (getListener() != null) {
                getListener().onConnectionError(error);
            }
            return;
        }
        if (StringeeCommon.isInCall) {
            if (listener != null) {
                listener.onError(new StringeeError(100, "You're on another call"));
            }
            return;
        }

        if (GSMCallStateReceiver.phoneState == GSMCallStateReceiver.PhoneState.OFFHOOK) {
            if (listener != null) {
                listener.onError(new StringeeError(101, "You're on a GSM call, and can't make a call"));
            }
            return;
        }

        connect(accessToken);
        if (StringeeUtils.isStringEmpty(callConfig.getFrom())) {
            callConfig.setFrom(userIdFromToken);
        }
        makeCall(callConfig, listener);
    }

    /**
     * Start a call
     *
     * @param callConfig call configuration
     */
    public void makeCall(CallConfig callConfig, StatusListener listener) {
        ThreadUtils.checkIsOnMainThread();
        if (StringeeCommon.isInCall) {
            if (listener != null) {
                listener.onError(new StringeeError(100, "You're on another call"));
            }
            return;
        }

        if (GSMCallStateReceiver.phoneState == GSMCallStateReceiver.PhoneState.OFFHOOK) {
            if (listener != null) {
                listener.onError(new StringeeError(101, "You're on a GSM call, and can't make a call"));
            }
            return;
        }

        StringeeCommon.isInCall = true;
        StringeeCallWrapper.getInstance(context).initializedOutgoingCall(callConfig, listener);
    }

    /**
     * Finalize
     */
    public void finalize() {
        host.clear();
        listener = null;
        disconnect();
        instance = null;
    }

    /**
     * Register push notification token
     *
     * @param token    firebase messaging token
     * @param listener callback
     */
    public void registerPushNotification(String token, StatusListener listener) {
        if (!isConnected()) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeClient is not connected yet"));
            }
            return;
        }
        client.registerPushToken(token, listener);
    }
}