package com.stringee.widget.call;

import android.content.pm.ActivityInfo;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.view.Gravity;
import android.view.View;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import androidx.activity.OnBackPressedCallback;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;

import com.stringee.common.StringeeAudioManager;
import com.stringee.widget.R;
import com.stringee.widget.common.StringeeCommon;
import com.stringee.widget.common.StringeeConstant;
import com.stringee.widget.common.StringeePermissionsUtils;
import com.stringee.widget.common.StringeeSensorManagerUtils;
import com.stringee.widget.common.StringeeUtils;
import com.stringee.widget.common.custom_view.scanner_view.ScanViewConfig;
import com.stringee.widget.databinding.StringeeActivityVideoCallBinding;
import com.stringee.widget.databinding.StringeeActivityVoiceCallBinding;
import com.stringee.widget.databinding.StringeeButtonEndCallBinding;
import com.stringee.widget.databinding.StringeeLayoutCallControlBinding;
import com.stringee.widget.databinding.StringeeLayoutCallStatusBinding;
import com.stringee.widget.databinding.StringeeLayoutIncomingCallBinding;

import java.util.Random;

public class StringeeCallActivity extends AppCompatActivity implements View.OnClickListener {
    private StringeeActivityVideoCallBinding videoCallBinding;
    private StringeeActivityVoiceCallBinding voiceCallBinding;
    private StringeeLayoutIncomingCallBinding incomingCallBinding;
    private StringeeLayoutCallStatusBinding callStatusBinding;
    private StringeeLayoutCallControlBinding callControlBinding;
    private StringeeButtonEndCallBinding endCallBinding;
    private StringeeCallWrapper callWrapper;
    private StringeeSensorManagerUtils sensorManagerUtils;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON);

        setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O_MR1) {
            setShowWhenLocked(true);
            setTurnScreenOn(true);
        }

        callWrapper = StringeeCallWrapper.getInstance(this);
        if (callWrapper.isVideoCall()) {
            videoCallBinding = StringeeActivityVideoCallBinding.inflate(getLayoutInflater());
            incomingCallBinding = videoCallBinding.vIncomingCall;
            callStatusBinding = videoCallBinding.vCallStatus;
            callControlBinding = videoCallBinding.vCallControl;
            endCallBinding = videoCallBinding.vBtnEndCall;
            setContentView(videoCallBinding.getRoot());
        } else {
            voiceCallBinding = StringeeActivityVoiceCallBinding.inflate(getLayoutInflater());
            incomingCallBinding = voiceCallBinding.vIncomingCall;
            callStatusBinding = voiceCallBinding.vCallStatus;
            callControlBinding = voiceCallBinding.vCallControl;
            endCallBinding = voiceCallBinding.vBtnEndCall;
            setContentView(voiceCallBinding.getRoot());
        }

        getOnBackPressedDispatcher().addCallback(this, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
            }
        });

        sensorManagerUtils = StringeeSensorManagerUtils.getInstance(this).initialize(getLocalClassName());

        updateView();

        if (!StringeePermissionsUtils.getInstance().isCallPermissionGranted(this, callWrapper.isVideoCall())) {
            StringeePermissionsUtils.getInstance().requestCallPermission(this, callWrapper.isVideoCall());
            return;
        }

        if (!callWrapper.isVideoCall()) {
            sensorManagerUtils.turnOn();
        }
        initCall();
    }

    private void updateView() {
        Bitmap avatar = StringeeUtils.getAvatar(this, callWrapper.getName(), new Random().nextInt(Integer.MAX_VALUE));

        // Incoming call layout
        if (callWrapper.getCallStatus() == CallStatus.INCOMING) {
            incomingCallBinding.btnAnswer.setOnClickListener(this);
            incomingCallBinding.btnReject.setOnClickListener(this);
            incomingCallBinding.tvFrom.setText(callWrapper.getName());
            incomingCallBinding.ivAvatarIncoming.setImageBitmap(avatar);
        }

        // Call status layout
        callStatusBinding.tvName.setText(callWrapper.getName());

        // Call control layout
        callControlBinding.vBtnSwitch.setVisibility(callWrapper.isVideoCall() ? View.VISIBLE : View.GONE);
        callControlBinding.vBtnCamera.setVisibility(callWrapper.isVideoCall() ? View.VISIBLE : View.GONE);
        callControlBinding.vBtnSpeaker.setVisibility(callWrapper.isVideoCall() ? View.GONE : View.VISIBLE);
        callControlBinding.btnCamera.setOnClickListener(this);
        callControlBinding.btnSwitch.setOnClickListener(this);
        callControlBinding.btnMute.setOnClickListener(this);
        callControlBinding.btnSpeaker.setOnClickListener(this);

        // Button end call
        endCallBinding.btnEnd.setOnClickListener(this);

        if (callWrapper.isVideoCall()) {
            if (callWrapper.getCallConfig() != null) {
                if (callWrapper.getCallConfig().isUseScannerView()) {
                    ScanViewConfig scanViewConfig = callWrapper.getCallConfig().getScannerViewOption();
                    videoCallBinding.vScanner.setColor(scanViewConfig.getColor());
                    videoCallBinding.vScanner.setRadius(scanViewConfig.getRadius());
                    videoCallBinding.vScanner.setWidthRatio(scanViewConfig.getWidthRatio());
                    videoCallBinding.vScanner.setAspectRatio(scanViewConfig.getAspectRatio());
                    videoCallBinding.vScanner.setBorderStrokeWidth(scanViewConfig.getBorderStrokeWidth());
                    videoCallBinding.vScanner.setBorderStrokeColor(scanViewConfig.getBorderStrokeColor());
                    videoCallBinding.vScanner.setBorderType(scanViewConfig.getBorderType());
                    videoCallBinding.vScanner.setDashLength(scanViewConfig.getDashLength());
                    videoCallBinding.vScanner.setVisibility(View.VISIBLE);
                } else {
                    videoCallBinding.vScanner.setVisibility(View.GONE);
                }
                if (callWrapper.getCallConfig().getCameraFacing() != CallConfig.CameraFacing.BOTH) {
                    callControlBinding.vBtnSwitch.setVisibility(View.GONE);
                }
            } else {
                videoCallBinding.vScanner.setVisibility(View.GONE);
            }
        } else {
            voiceCallBinding.ivAvatar.setImageBitmap(avatar);
        }
        updateCallStatus(callWrapper.getCallStatus());
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        boolean isGranted = StringeePermissionsUtils.getInstance().verifyPermissions(grantResults);
        if (requestCode == StringeePermissionsUtils.REQUEST_CALL_PERMISSION) {
            if (!isGranted) {
                StringeeUtils.reportMessage(this, R.string.stringee_recording_required);
                finish();
                StringeeCommon.isInCall = false;
            } else {
                if (!callWrapper.isVideoCall()) {
                    sensorManagerUtils.turnOn();
                }
                initCall();
            }
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        runOnUiThread(() -> {
            if (callWrapper.getCallStatus() == CallStatus.STARTED || callWrapper.getCallStatus() == CallStatus.CALLING || callWrapper.getCallStatus() == CallStatus.RINGING) {
                getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
                getWindow().clearFlags(WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD);
                getWindow().clearFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED);
                getWindow().clearFlags(WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON);

                sensorManagerUtils = StringeeSensorManagerUtils.getInstance(this).initialize(getLocalClassName());
                if (!callWrapper.isVideoCall()) {
                    sensorManagerUtils.turnOff();
                }

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O_MR1) {
                    setShowWhenLocked(false);
                    setTurnScreenOn(false);
                }
            }
        });
    }

    @Override
    protected void onResume() {
        super.onResume();
        runOnUiThread(() -> {
            if (callWrapper.getCallStatus() == CallStatus.STARTED || callWrapper.getCallStatus() == CallStatus.CALLING || callWrapper.getCallStatus() == CallStatus.RINGING) {
                getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
                getWindow().addFlags(WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD);
                getWindow().addFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED);
                getWindow().addFlags(WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON);

                sensorManagerUtils = StringeeSensorManagerUtils.getInstance(this).initialize(getLocalClassName());
                if (!callWrapper.isVideoCall()) {
                    sensorManagerUtils.turnOn();
                } else {
                    if (callWrapper.isNeedResumeVideo()) {
                        callWrapper.resumeVideo();
                    }
                }

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O_MR1) {
                    setShowWhenLocked(true);
                    setTurnScreenOn(true);
                }
            }
        });
    }

    private void initCall() {
        callWrapper.registerEvent(new OnCallListener() {
            @Override
            public void onCallStatus(CallStatus status) {
                runOnUiThread(() -> updateCallStatus(callWrapper.getCallStatus()));
            }

            @Override
            public void onError(String message) {
                runOnUiThread(() -> dismiss());
            }

            @Override
            public void onReceiveLocalStream() {
                runOnUiThread(() -> {
                    try {
                        if (callWrapper.isVideoCall()) {
                            if (callWrapper.getCallConfig() != null && callWrapper.getCallConfig().isUseScannerView()) {
                                videoCallBinding.vScanner.setVisibility(callWrapper.isVideoEnable() ? View.VISIBLE : View.GONE);
                            }

                            FrameLayout.LayoutParams childParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.WRAP_CONTENT);
                            childParams.gravity = Gravity.CENTER;

                            videoCallBinding.vLocal.removeAllViews();
                            videoCallBinding.vLocal.addView(callWrapper.getLocalView(), childParams);
                            callWrapper.renderLocalView();
                        }
                    } catch (Exception ex) {
                        StringeeUtils.reportException(StringeeCallActivity.class, ex);
                    }
                });
            }

            @Override
            public void onReceiveRemoteStream() {
                runOnUiThread(() -> {
                    try {
                        if (callWrapper.isVideoCall()) {
                            RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) videoCallBinding.dflLocal.getLayoutParams();
                            params.width = StringeeUtils.dpToPx(StringeeCallActivity.this, 100);
                            params.height = StringeeUtils.dpToPx(StringeeCallActivity.this, 150);
                            params.setMargins(StringeeUtils.dpToPx(StringeeCallActivity.this, 20),
                                    StringeeUtils.dpToPx(StringeeCallActivity.this, 20),
                                    StringeeUtils.dpToPx(StringeeCallActivity.this, 20),
                                    StringeeUtils.dpToPx(StringeeCallActivity.this, 20));
                            params.removeRule(RelativeLayout.CENTER_IN_PARENT);
                            params.addRule(RelativeLayout.ALIGN_PARENT_START);
                            params.addRule(RelativeLayout.BELOW, R.id.v_call_status);
                            videoCallBinding.dflLocal.setAllowDrag(true);
                            videoCallBinding.dflLocal.setLayoutParams(params);

                            videoCallBinding.vInCall.setOnClickListener(StringeeCallActivity.this);

                            FrameLayout.LayoutParams childParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.WRAP_CONTENT);
                            childParams.gravity = Gravity.CENTER;

                            videoCallBinding.vRemote.removeAllViews();
                            videoCallBinding.vRemote.addView(callWrapper.getRemoteView(), childParams);
                            callWrapper.renderRemoteView();

                            videoCallBinding.vRemote.setOnClickListener(StringeeCallActivity.this);
                        }
                    } catch (Exception ex) {
                        StringeeUtils.reportException(StringeeCallActivity.class, ex);
                    }
                });
            }

            @Override
            public void onSpeakerChange(StringeeAudioManager.AudioDevice selectedAudioDevice) {
                runOnUiThread(() -> {
                    switch (selectedAudioDevice) {
                        case WIRED_HEADSET:
                        case BLUETOOTH:
                            callControlBinding.btnSpeaker.setBackgroundResource(R.drawable.stringee_btn_circle_selected_selector);
                            callControlBinding.btnSpeaker.setImageResource(selectedAudioDevice == StringeeAudioManager.AudioDevice.BLUETOOTH ? R.drawable.stringee_ic_bluetooth : R.drawable.stringee_ic_speaker_off_white);
                            callControlBinding.btnSpeaker.setClickable(false);
                            break;
                        case EARPIECE:
                        case SPEAKER_PHONE:
                            callControlBinding.btnSpeaker.setBackgroundResource(selectedAudioDevice == StringeeAudioManager.AudioDevice.SPEAKER_PHONE ? R.drawable.stringee_btn_circle_selector : R.drawable.stringee_btn_circle_selected_selector);
                            callControlBinding.btnSpeaker.setImageResource(selectedAudioDevice == StringeeAudioManager.AudioDevice.SPEAKER_PHONE ? R.drawable.stringee_ic_speaker_on_black : R.drawable.stringee_ic_speaker_off_white);
                            callControlBinding.btnSpeaker.setClickable(true);
                            break;
                    }
                });
            }

            @Override
            public void onMicChange(boolean isOn) {
                runOnUiThread(() -> {
                    callControlBinding.btnMute.setBackgroundResource(!isOn ? R.drawable.stringee_btn_circle_selector : R.drawable.stringee_btn_circle_selected_selector);
                    callControlBinding.btnMute.setImageResource(!isOn ? R.drawable.stringee_ic_mic_off_black : R.drawable.stringee_ic_mic_on_white);
                });
            }

            @Override
            public void onVideoChange(boolean isOn) {
                runOnUiThread(() -> {
                    callControlBinding.btnCamera.setBackgroundResource(isOn ? R.drawable.stringee_btn_circle_selected_selector : R.drawable.stringee_btn_circle_selector);
                    callControlBinding.btnCamera.setImageResource(isOn ? R.drawable.stringee_ic_cam_on_white : R.drawable.stringee_ic_cam_off_black);
                });
            }

            @Override
            public void onTimer(String duration) {
                runOnUiThread(() -> callStatusBinding.tvTime.setText(duration));
            }

            @Override
            public void onCallBandwidth(long callBandwidth) {
                runOnUiThread(() -> {
                    if (callBandwidth <= 0) {
                        callStatusBinding.imNetwork.setImageResource(R.drawable.stringee_signal_no_connect);
                    } else {
                        if (callBandwidth < 15000) {
                            callStatusBinding.imNetwork.setImageResource(R.drawable.stringee_signal_poor);
                        } else {
                            if (callBandwidth >= 35000) {
                                callStatusBinding.imNetwork.setImageResource(R.drawable.stringee_signal_excellent);
                            } else {
                                if (callBandwidth <= 25000) {
                                    callStatusBinding.imNetwork.setImageResource(R.drawable.stringee_signal_average);
                                } else {
                                    callStatusBinding.imNetwork.setImageResource(R.drawable.stringee_signal_good);
                                }
                            }
                        }
                    }
                });
            }
        });
        if (!callWrapper.isIncomingCall()) {
            callWrapper.makeCall();
        } else {
            boolean isAnswerFromPush = getIntent().getBooleanExtra(StringeeConstant.PARAM_ANSWER, false);
            if (isAnswerFromPush) {
                callWrapper.answer();
            }
        }
    }

    private void updateCallStatus(CallStatus callStatus) {
        if (callStatus == CallStatus.INCOMING) {
            incomingCallBinding.getRoot().setVisibility(View.VISIBLE);
            if (callWrapper.isVideoCall()) {
                videoCallBinding.vInCall.setVisibility(View.GONE);
            } else {
                voiceCallBinding.vInCall.setVisibility(View.GONE);
            }
        } else {
            if (callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY) {
                incomingCallBinding.getRoot().setVisibility(View.GONE);
                if (callWrapper.isVideoCall()) {
                    videoCallBinding.vInCall.setVisibility(View.VISIBLE);
                } else {
                    voiceCallBinding.vInCall.setVisibility(View.VISIBLE);
                }
            }
        }

        incomingCallBinding.btnAnswer.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);
        incomingCallBinding.btnReject.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);

        if (callWrapper.isVideoCall()) {
            videoCallBinding.dflLocal.setAllowDrag(callWrapper.getCallStatus() == CallStatus.STARTED);
        }

        callControlBinding.btnCamera.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);
        callControlBinding.btnMute.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);
        callControlBinding.btnSpeaker.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);
        callControlBinding.btnSwitch.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);

        endCallBinding.btnEnd.setClickable(callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY);

        callStatusBinding.imNetwork.setVisibility(callStatus == CallStatus.STARTED ? View.VISIBLE : View.GONE);
        callStatusBinding.vTime.setVisibility(callStatus == CallStatus.STARTED ? View.VISIBLE : View.GONE);
        switch (callStatus) {
            case INCOMING:
                callStatusBinding.tvStatus.setText(R.string.stringee_incoming);
                break;
            case CALLING:
                callStatusBinding.tvStatus.setText(R.string.stringee_calling);
                break;
            case RINGING:
                callStatusBinding.tvStatus.setText(R.string.stringee_ringing);
                break;
            case STARTING:
                callStatusBinding.tvStatus.setText(R.string.stringee_starting);
                break;
            case STARTED:
                callStatusBinding.tvStatus.setText(R.string.stringee_started);
                break;
            case BUSY:
                callStatusBinding.tvStatus.setText(R.string.stringee_busy);
                dismiss();
                break;
            case ENDED:
                callStatusBinding.tvStatus.setText(R.string.stringee_ended);
                dismiss();
                break;
        }
    }

    @Override
    public void onClick(View view) {
        int vId = view.getId();
        if (vId == R.id.btn_answer) {
            callWrapper.answer();
        } else if (vId == R.id.btn_reject) {
            callWrapper.endCall(false, "Click on reject button");
        } else if (vId == R.id.btn_end) {
            callWrapper.endCall(true, "Click on end button");
        } else if (vId == R.id.btn_mute) {
            callWrapper.mute();
        } else if (vId == R.id.btn_speaker) {
            callWrapper.changeSpeaker();
        } else if (vId == R.id.btn_camera) {
            callWrapper.enableVideo();
        } else if (vId == R.id.btn_switch) {
            callWrapper.switchCamera();
        } else if (vId == R.id.v_in_call || vId == R.id.v_remote) {
            boolean isControlVisible = callControlBinding.getRoot().getVisibility() == View.VISIBLE;
            callControlBinding.getRoot().setVisibility(isControlVisible ? View.GONE : View.VISIBLE);
            boolean isEndCallVisible = endCallBinding.getRoot().getVisibility() == View.VISIBLE;
            endCallBinding.getRoot().setVisibility(isEndCallVisible ? View.GONE : View.VISIBLE);
        }
    }

    private void dismiss() {
        sensorManagerUtils.releaseSensor();

        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD);
        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED);
        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON);

        finish();
    }
}