package com.stringee.widget.call;

import android.content.Context;
import android.content.Intent;
import android.graphics.Color;
import android.util.Log;
import android.view.View;
import android.widget.Toast;

import com.stringee.call.CallType;
import com.stringee.call.StringeeCall;
import com.stringee.call.StringeeCall2;
import com.stringee.call.VideoQuality;
import com.stringee.common.StringeeAudioManager;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StatusListener;
import com.stringee.video.StringeeVideoTrack;
import com.stringee.widget.R;
import com.stringee.widget.StringeeListener;
import com.stringee.widget.StringeeWidget;
import com.stringee.widget.call.receiver.GSMCallStateReceiver;
import com.stringee.widget.common.StringeeAudioManagerUtils;
import com.stringee.widget.common.StringeeCommon;
import com.stringee.widget.common.StringeeConstant;
import com.stringee.widget.common.StringeeDateTimeUtils;
import com.stringee.widget.common.StringeeNotificationService;
import com.stringee.widget.common.StringeeNotify;
import com.stringee.widget.common.StringeeUtils;
import com.stringee.widget.common.custom_view.scanner_view.ScanViewConfig;
import com.stringee.widget.common.custom_view.scanner_view.ScannerOverlayView;
import com.stringee.widget.common.event.EventManager;
import com.stringee.widget.common.event.EventReceiver;

import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.RendererCommon;

import java.util.Timer;
import java.util.TimerTask;

public class StringeeCallWrapper {
    private static volatile StringeeCallWrapper instance;
    private final Context context;
    private StringeeCall stringeeCall;
    private StringeeCall2 stringeeCall2;
    private CallConfig callConfig;
    private boolean isVideoCall = false;
    private boolean isSpeakerOn = false;
    private boolean isVideoEnable = false;
    private boolean isMicOn = true;
    private boolean isIncomingCall = false;
    private boolean canSwitch = true;
    private boolean isFrontCamera = true;
    private boolean needResumeVideo = false;
    private final StringeeAudioManagerUtils audioManagerUtils;
    private OnCallListener listener;
    private final StringeeWidget stringeeWidget;
    private StringeeCall.SignalingState callSignalingState = StringeeCall.SignalingState.CALLING;
    private StringeeCall.MediaState callMediaState = StringeeCall.MediaState.DISCONNECTED;
    private StringeeCall2.SignalingState call2SignalingState = StringeeCall2.SignalingState.CALLING;
    private StringeeCall2.MediaState call2MediaState = StringeeCall2.MediaState.DISCONNECTED;
    private CallStatus callStatus = CallStatus.CALLING;
    private Timer timer;
    private Timer statsTimer;
    private double prevCallTimestamp = 0;
    private long prevCallBytes = 0;
    private long callBw = 0;
    private String frontCameraName = "";
    private String rearCameraName = "";
    private StatusListener makeCallListener;
    private final EventReceiver phoneStateReceiver;

    public StringeeCallWrapper(Context context) {
        this.context = context.getApplicationContext();
        this.audioManagerUtils = StringeeAudioManagerUtils.getInstance(context);
        this.audioManagerUtils.setAudioEvents(
                selectedAudioDevice -> StringeeUtils.runOnUiThread(() -> {
                    Log.d(StringeeConstant.TAG,
                            "onAudioEvents: selectedAudioDevice - " + selectedAudioDevice.name());
                    if (isVideoCall) {
                        switch (selectedAudioDevice) {
                            case WIRED_HEADSET:
                                audioManagerUtils.setSpeakerphoneOn(false);
                                break;
                            case BLUETOOTH:
                                audioManagerUtils.setBluetoothScoOn(true);
                                break;
                            case EARPIECE:
                            case SPEAKER_PHONE:
                                audioManagerUtils.setSpeakerphoneOn(true);
                                break;
                        }
                    } else {
                        switch (selectedAudioDevice) {
                            case WIRED_HEADSET:
                            case BLUETOOTH:
                                if (selectedAudioDevice ==
                                        StringeeAudioManager.AudioDevice.BLUETOOTH) {
                                    audioManagerUtils.setBluetoothScoOn(true);
                                } else {
                                    audioManagerUtils.setSpeakerphoneOn(false);
                                }
                                break;
                            case EARPIECE:
                            case SPEAKER_PHONE:
                                audioManagerUtils.setSpeakerphoneOn(isSpeakerOn);
                                break;
                        }
                    }
                    if (listener != null) {
                        listener.onSpeakerChange(selectedAudioDevice);
                    }
                }));
        this.stringeeWidget = StringeeWidget.getInstance(context);

        phoneStateReceiver = new EventReceiver() {
            @Override
            public void onReceive(Intent intent) {
                String phoneState = intent.getStringExtra(StringeeConstant.PARAM_TELEPHONY_STATE);
                if (phoneState != null) {
                    if (GSMCallStateReceiver.PhoneState.getState(phoneState) ==
                            GSMCallStateReceiver.PhoneState.OFFHOOK) {
                        if (callStatus != CallStatus.ENDED && callStatus != CallStatus.BUSY &&
                                callStatus != CallStatus.INCOMING) {
                            endCall(true, "End by phone state change to offhook");
                            StringeeUtils.reportMessage(context,
                                    context.getString(R.string.stringee_end_call_by_gsm));
                        }
                    }
                }
            }
        };
        EventManager.getInstance()
                .registerEvent(phoneStateReceiver,
                        StringeeNotify.TELEPHONY_STATE_CHANGE.getValue());
    }

    public static StringeeCallWrapper getInstance(Context context) {
        if (instance == null) {
            synchronized (StringeeCallWrapper.class) {
                if (instance == null) {
                    instance = new StringeeCallWrapper(context);
                }
            }
        }
        return instance;
    }

    public CallStatus getCallStatus() {
        return callStatus;
    }

    public void initializedOutgoingCall(CallConfig callConfig, StatusListener makeCallListener) {
        StringeeCommon.isInCall = true;
        this.isIncomingCall = false;
        this.callConfig = callConfig;
        this.makeCallListener = makeCallListener;
        String from = StringeeWidget.getInstance(context).getClient().getUserId();
        String callFrom = callConfig.getFrom();
        if (!StringeeUtils.isStringEmpty(callFrom)) {
            from = callFrom;
        }
        String customData = callConfig.getCustomData();
        this.isVideoCall = callConfig.isVideoCall();
        if (!callConfig.isVideoCall()) {
            this.stringeeCall =
                    new StringeeCall(stringeeWidget.getClient(), from, callConfig.getTo());
            this.stringeeCall.setVideoCall(isVideoCall);
            if (!StringeeUtils.isStringEmpty(customData)) {
                this.stringeeCall.setCustom(customData);
            }
        } else {
            this.stringeeCall2 =
                    new StringeeCall2(stringeeWidget.getClient(), from, callConfig.getTo());
            this.stringeeCall2.setVideoCall(isVideoCall);
            if (callConfig.getResolution() != null) {
                switch (callConfig.getResolution()) {
                    case NORMAL:
                        this.stringeeCall2.setVideoQuality(VideoQuality.QUALITY_480P);
                        break;
                    case HD:
                        this.stringeeCall2.setVideoQuality(VideoQuality.QUALITY_720P);
                        break;
                    case FULLHD:
                        this.stringeeCall2.setVideoQuality(VideoQuality.QUALITY_1080P);
                        break;
                }
            }
            if (!StringeeUtils.isStringEmpty(customData)) {
                this.stringeeCall2.setCustom(customData);
            }
        }
        this.isSpeakerOn = isVideoCall;
        this.isVideoEnable = isVideoCall;
        this.callStatus = CallStatus.CALLING;
        if (callConfig.isVideoCall()) {
            if (!StringeeUtils.isStringEmpty(callConfig.getCustomData()) &&
                    !StringeeUtils.isStringEmpty(callConfig.getTo())) {
                try {
                    JSONObject customDataObj = new JSONObject(callConfig.getCustomData());
                    String company = customDataObj.optString("company");
                    if (company.equals("Viettel") && callConfig.getTo().equals("callbot")) {
                        ScanViewConfig scanViewConfig = new ScanViewConfig();
                        scanViewConfig.setWidthRatio(0.8f);
                        scanViewConfig.setAspectRatio(1.5f);
                        scanViewConfig.setBorderStrokeWidth(2f);
                        scanViewConfig.setBorderStrokeColor(Color.parseColor("#e83054"));
                        scanViewConfig.setBorderType(ScannerOverlayView.BorderType.OVAL);
                        scanViewConfig.setDashLength(10f);
                        callConfig.useScannerView(scanViewConfig);
                    }
                } catch (JSONException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        startAudioManager();
        registerCallEvent();
        Intent intent = new Intent(context, StringeeCallActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }

    public void initializedIncomingCall(StringeeCall stringeeCall) {
        StringeeCommon.isInCall = true;
        this.stringeeCall = stringeeCall;
        this.isIncomingCall = true;
        this.isVideoCall = stringeeCall.isVideoCall();
        this.isSpeakerOn = stringeeCall.isVideoCall();
        this.isVideoEnable = stringeeCall.isVideoCall();
        this.callStatus = CallStatus.INCOMING;
        registerCallEvent();
    }

    public void initializedIncomingCall(StringeeCall2 stringeeCall2) {
        StringeeCommon.isInCall = true;
        this.stringeeCall2 = stringeeCall2;
        this.isIncomingCall = true;
        this.isVideoCall = stringeeCall2.isVideoCall();
        this.isSpeakerOn = stringeeCall2.isVideoCall();
        this.isVideoEnable = stringeeCall2.isVideoCall();
        this.callStatus = CallStatus.INCOMING;
        registerCallEvent();
    }

    public void registerEvent(OnCallListener listener) {
        this.listener = listener;
    }

    private void registerCallEvent() {
        if (!isVideoCall) {
            stringeeCall.setCallListener(new StringeeCall.StringeeCallListener() {
                @Override
                public void onSignalingStateChange(StringeeCall stringeeCall,
                                                   StringeeCall.SignalingState signalingState,
                                                   String reason, int sipCode, String sipReason) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onSignalingStateChange: " + signalingState);
                        StringeeListener clientListener =
                                StringeeWidget.getInstance(context).getListener();
                        if (clientListener != null) {
                            clientListener.onCallStateChange(stringeeCall, signalingState);
                        }
                        callSignalingState = signalingState;
                        switch (callSignalingState) {
                            case CALLING:
                                callStatus = CallStatus.CALLING;
                                if (makeCallListener != null) {
                                    makeCallListener.onSuccess();
                                }
                                if (stringeeCall.getCallType() != CallType.APP_TO_PHONE) {
                                    audioManagerUtils.playWaitingSound();
                                }
                                break;
                            case RINGING:
                                callStatus = CallStatus.RINGING;
                                break;
                            case ANSWERED:
                                callStatus = CallStatus.STARTING;
                                if (callMediaState == StringeeCall.MediaState.CONNECTED) {
                                    callStarted();
                                    callStatus = CallStatus.STARTED;
                                }
                                break;
                            case BUSY:
                                callStatus = CallStatus.BUSY;
                                release();
                                break;
                            case ENDED:
                                callStatus = CallStatus.ENDED;
                                release();
                                break;
                        }
                        if (listener != null) {
                            listener.onCallStatus(callStatus);
                        }
                    });
                }

                @Override
                public void onError(StringeeCall stringeeCall, int code, String desc) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onError: " + desc);
                        callStatus = CallStatus.ENDED;
                        if (makeCallListener != null) {
                            makeCallListener.onError(new StringeeError(code, desc));
                        }
                        if (listener != null) {
                            listener.onError(desc);
                            listener.onCallStatus(callStatus);
                        }
                    });
                }

                @Override
                public void onHandledOnAnotherDevice(StringeeCall stringeeCall,
                                                     StringeeCall.SignalingState signalingState,
                                                     String desc) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onHandledOnAnotherDevice: " + signalingState);
                        if (signalingState != StringeeCall.SignalingState.RINGING) {
                            callStatus = CallStatus.ENDED;
                            if (listener != null) {
                                listener.onCallStatus(callStatus);
                            }
                            String msg =
                                    context.getString(R.string.stringee_another_device_answered);
                            switch (signalingState) {
                                case BUSY:
                                    msg = context.getString(
                                            R.string.stringee_another_device_rejected);
                                    break;
                                case ENDED:
                                    msg = context.getString(R.string.stringee_another_device_ended);
                                    break;
                            }
                            StringeeUtils.reportMessage(context, msg);
                            release();
                        }
                    });
                }

                @Override
                public void onMediaStateChange(StringeeCall stringeeCall,
                                               StringeeCall.MediaState mediaState) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onMediaStateChange: " + mediaState);
                        callMediaState = mediaState;
                        if (callSignalingState == StringeeCall.SignalingState.ANSWERED) {
                            callStatus = CallStatus.STARTED;
                            callStarted();
                            if (listener != null) {
                                listener.onCallStatus(callStatus);
                            }
                        }
                    });
                }

                @Override
                public void onLocalStream(StringeeCall stringeeCall) {
                }

                @Override
                public void onRemoteStream(StringeeCall stringeeCall) {
                }

                @Override
                public void onCallInfo(StringeeCall stringeeCall, JSONObject jsonObject) {
                    StringeeUtils.runOnUiThread(() -> Log.d(StringeeConstant.TAG,
                            "onCallInfo: " + jsonObject.toString()));
                }
            });
        } else {
            stringeeCall2.setCallListener(new StringeeCall2.StringeeCallListener() {
                @Override
                public void onSignalingStateChange(StringeeCall2 stringeeCall2,
                                                   StringeeCall2.SignalingState signalingState,
                                                   String reason, int sipCode, String sipReason) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onSignalingStateChange: " + signalingState);
                        StringeeListener clientListener =
                                StringeeWidget.getInstance(context).getListener();
                        if (clientListener != null) {
                            clientListener.onCallStateChange2(stringeeCall2, signalingState);
                        }
                        call2SignalingState = signalingState;
                        switch (call2SignalingState) {
                            case CALLING:
                                callStatus = CallStatus.CALLING;
                                if (makeCallListener != null) {
                                    makeCallListener.onSuccess();
                                }
                                audioManagerUtils.playWaitingSound();
                                break;
                            case RINGING:
                                callStatus = CallStatus.RINGING;
                                break;
                            case ANSWERED:
                                callStatus = CallStatus.STARTING;
                                if (call2MediaState == StringeeCall2.MediaState.CONNECTED) {
                                    callStarted();
                                    callStatus = CallStatus.STARTED;
                                }
                                break;
                            case BUSY:
                                callStatus = CallStatus.BUSY;
                                release();
                                break;
                            case ENDED:
                                callStatus = CallStatus.ENDED;
                                release();
                                break;
                        }
                        if (listener != null) {
                            listener.onCallStatus(callStatus);
                        }
                    });
                }

                @Override
                public void onError(StringeeCall2 stringeeCall2, int code, String desc) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onError: " + desc);
                        callStatus = CallStatus.ENDED;
                        if (listener != null) {
                            listener.onError(desc);
                            listener.onCallStatus(callStatus);
                        }
                    });
                }

                @Override
                public void onHandledOnAnotherDevice(StringeeCall2 stringeeCall2,
                                                     StringeeCall2.SignalingState signalingState,
                                                     String desc) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onHandledOnAnotherDevice: " + signalingState);
                        if (signalingState != StringeeCall2.SignalingState.RINGING) {
                            callStatus = CallStatus.ENDED;
                            if (listener != null) {
                                listener.onCallStatus(callStatus);
                            }
                            String msg =
                                    context.getString(R.string.stringee_another_device_answered);
                            switch (signalingState) {
                                case BUSY:
                                    msg = context.getString(
                                            R.string.stringee_another_device_rejected);
                                    break;
                                case ENDED:
                                    msg = context.getString(R.string.stringee_another_device_ended);
                                    break;
                            }
                            StringeeUtils.reportMessage(context, msg);
                            release();
                        }
                    });
                }

                @Override
                public void onMediaStateChange(StringeeCall2 stringeeCall2,
                                               StringeeCall2.MediaState mediaState) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onMediaStateChange: " + mediaState);
                        call2MediaState = mediaState;
                        if (call2SignalingState == StringeeCall2.SignalingState.ANSWERED) {
                            callStatus = CallStatus.STARTED;
                            callStarted();
                            if (listener != null) {
                                listener.onCallStatus(callStatus);
                            }
                        }
                    });
                }

                @Override
                public void onLocalStream(StringeeCall2 stringeeCall2) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onLocalStream");
                        if (isVideoCall) {
                            if (listener != null) {
                                listener.onReceiveLocalStream();
                            }
                            try {
                                if (callConfig != null && callConfig.getCameraFacing() !=
                                        CallConfig.CameraFacing.BOTH && canSwitch) {
                                    stringeeCall.switchCamera(new StatusListener() {
                                        @Override
                                        public void onSuccess() {
                                            StringeeUtils.runOnUiThread(() -> {
                                                try {
                                                    stringeeCall2.getLocalView()
                                                            .setMirror(
                                                                    callConfig.getCameraFacing() ==
                                                                            CallConfig.CameraFacing.FRONT);
                                                } catch (Exception ex) {
                                                    StringeeUtils.reportException(
                                                            StringeeCallWrapper.class, ex);
                                                }
                                            });
                                        }
                                    }, callConfig.getCameraFacing() ==
                                            CallConfig.CameraFacing.FRONT ? frontCameraName : rearCameraName);
                                }
                            } catch (Exception ex) {
                                StringeeUtils.reportException(StringeeCallWrapper.class, ex);
                            }
                        }
                    });
                }

                @Override
                public void onRemoteStream(StringeeCall2 stringeeCall2) {
                    StringeeUtils.runOnUiThread(() -> {
                        Log.d(StringeeConstant.TAG, "onRemoteStream");
                        if (isVideoCall) {
                            if (listener != null) {
                                listener.onReceiveRemoteStream();
                            }
                        }
                    });
                }

                @Override
                public void onVideoTrackAdded(StringeeVideoTrack stringeeVideoTrack) {
                    StringeeUtils.runOnUiThread(() -> Log.d(StringeeConstant.TAG,
                            "onVideoTrackAdded: " + stringeeVideoTrack.getId()));
                }

                @Override
                public void onVideoTrackRemoved(StringeeVideoTrack stringeeVideoTrack) {
                    StringeeUtils.runOnUiThread(() -> Log.d(StringeeConstant.TAG,
                            "onVideoTrackRemoved: " + stringeeVideoTrack.getId()));
                }

                @Override
                public void onLocalTrackAdded(StringeeCall2 stringeeCall2,
                                              StringeeVideoTrack stringeeVideoTrack) {

                }

                @Override
                public void onRemoteTrackAdded(StringeeCall2 stringeeCall2,
                                               StringeeVideoTrack stringeeVideoTrack) {

                }

                @Override
                public void onCallInfo(StringeeCall2 stringeeCall2, JSONObject jsonObject) {
                    StringeeUtils.runOnUiThread(() -> Log.d(StringeeConstant.TAG,
                            "onCallInfo: " + jsonObject.toString()));
                }

                @Override
                public void onTrackMediaStateChange(String s,
                                                    StringeeVideoTrack.MediaType mediaType,
                                                    boolean b) {

                }
            });
            stringeeCall2.setCaptureSessionListener(
                    new StringeeVideoTrack.CaptureSessionListener() {
                        @Override
                        public void onCapturerStarted(StringeeVideoTrack.TrackType trackType) {

                        }

                        @Override
                        public void onCapturerStopped(StringeeVideoTrack.TrackType trackType) {
                            needResumeVideo = true;
                        }
                    });
        }
    }

    public boolean isVideoCall() {
        return isVideoCall;
    }

    public boolean isIncomingCall() {
        return isIncomingCall;
    }

    public boolean isVideoEnable() {
        return isVideoEnable;
    }

    private void startAudioManager() {
        audioManagerUtils.startAudioManager();
        audioManagerUtils.setSpeakerphoneOn(isVideoCall);
    }

    public void makeCall() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!isVideoCall) {
            stringeeCall.makeCall(new StatusListener() {
                @Override
                public void onSuccess() {
                    handleResponse("makeCall", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    handleResponse("makeCall", false, stringeeError.getMessage());
                }
            });
        } else {
            getCameraName();
            stringeeCall2.makeCall(new StatusListener() {
                @Override
                public void onSuccess() {
                    handleResponse("makeCall", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    handleResponse("makeCall", false, stringeeError.getMessage());
                }
            });
        }
    }

    private void getCameraName() {
        // get cameraId
        CameraEnumerator enumerator =
                Camera2Enumerator.isSupported(context) ? new Camera2Enumerator(
                        context) : new Camera1Enumerator();
        String[] cameraNames = enumerator.getDeviceNames();
        if (cameraNames.length > 0) {
            // first front id is main front camera
            for (String name : cameraNames) {
                boolean isFrontFace = enumerator.isFrontFacing(name);
                if (isFrontFace) {
                    frontCameraName = name;
                    break;
                }
            }

            // first rear id is main rear camera
            for (String cameraName : cameraNames) {
                boolean isBackFace = enumerator.isBackFacing(cameraName);
                if (isBackFace) {
                    rearCameraName = cameraName;
                    break;
                }
            }
        }

        canSwitch = (!StringeeUtils.isStringEmpty(frontCameraName) &&
                !StringeeUtils.isStringEmpty(rearCameraName));
    }

    public void initAnswer() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!isVideoCall) {
            stringeeCall.ringing(new StatusListener() {
                @Override
                public void onSuccess() {
                    handleResponse("initAnswer", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    handleResponse("initAnswer", false, stringeeError.getMessage());
                }
            });
        } else {
            getCameraName();
            stringeeCall2.ringing(new StatusListener() {
                @Override
                public void onSuccess() {
                    handleResponse("initAnswer", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    handleResponse("initAnswer", false, stringeeError.getMessage());
                }
            });
        }
        audioManagerUtils.startRingtoneAndVibration();
    }

    public void answer() {
        if (GSMCallStateReceiver.phoneState == GSMCallStateReceiver.PhoneState.OFFHOOK) {
            Toast.makeText(context, R.string.stringee_answer_on_gsm, Toast.LENGTH_SHORT).show();
            return;
        }
        StringeeNotificationService.getInstance(context)
                .cancelNotification(StringeeNotificationService.INCOMING_CALL_NOTIFICATION_ID);

        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }

        if (!isVideoCall) {
            stringeeCall.answer(new StatusListener() {
                @Override
                public void onSuccess() {
                    audioManagerUtils.stopRinging();
                    handleResponse("answer", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    handleResponse("answer", false, stringeeError.getMessage());
                }
            });
        } else {
            stringeeCall2.answer(new StatusListener() {
                @Override
                public void onSuccess() {
                    audioManagerUtils.stopRinging();
                    handleResponse("answer", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    handleResponse("answer", false, stringeeError.getMessage());
                }
            });
        }
    }

    public void endCall(boolean isHangUp, String reason) {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!isVideoCall) {
            if (isHangUp) {
                stringeeCall.hangup(reason, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        handleResponse("hangup", true, null);
                    }

                    @Override
                    public void onError(StringeeError stringeeError) {
                        super.onError(stringeeError);
                        handleResponse("hangup", false, stringeeError.getMessage());
                    }
                });
            } else {
                stringeeCall.reject(reason, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        handleResponse("reject", true, null);
                    }

                    @Override
                    public void onError(StringeeError stringeeError) {
                        super.onError(stringeeError);
                        handleResponse("reject", false, stringeeError.getMessage());
                    }
                });
            }
        } else {
            if (isHangUp) {
                stringeeCall2.hangup(reason, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        handleResponse("hangup", true, null);
                    }

                    @Override
                    public void onError(StringeeError stringeeError) {
                        super.onError(stringeeError);
                        handleResponse("hangup", false, stringeeError.getMessage());
                    }
                });
            } else {
                stringeeCall2.reject(reason, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        handleResponse("reject", true, null);
                    }

                    @Override
                    public void onError(StringeeError stringeeError) {
                        super.onError(stringeeError);
                        handleResponse("reject", false, stringeeError.getMessage());
                    }
                });
            }
        }
        if (listener != null) {
            listener.onCallStatus(CallStatus.ENDED);
        }
        release();
    }

    public void enableVideo() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!isVideoCall) {
            stringeeCall.enableVideo(!isVideoEnable);
        } else {
            stringeeCall2.enableVideo(!isVideoEnable);
        }
        handleResponse("enableVideo", true, null);
        isVideoEnable = !isVideoEnable;
        if (listener != null) {
            listener.onVideoChange(isVideoEnable);
        }
    }

    public void mute() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!isVideoCall) {
            stringeeCall.mute(isMicOn);
        } else {
            stringeeCall2.mute(isMicOn);
        }
        handleResponse("mute", true, null);
        isMicOn = !isMicOn;
        if (listener != null) {
            listener.onMicChange(isMicOn);
        }
    }

    public void changeSpeaker() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!isVideoCall) {
            audioManagerUtils.setSpeakerphoneOn(!isSpeakerOn);
        } else {
            audioManagerUtils.setSpeakerphoneOn(!isSpeakerOn);
        }
        handleResponse("changeSpeaker", true, null);
        isSpeakerOn = !isSpeakerOn;
        if (listener != null) {
            listener.onSpeakerChange(
                    isSpeakerOn ? StringeeAudioManager.AudioDevice.SPEAKER_PHONE : StringeeAudioManager.AudioDevice.EARPIECE);
        }
    }

    public void switchCamera() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!canSwitch) {
            return;
        }
        canSwitch = false;
        if (isVideoCall) {
            stringeeCall2.switchCamera(new StatusListener() {
                @Override
                public void onSuccess() {
                    canSwitch = true;
                    isFrontCamera = !isFrontCamera;
                    if (stringeeCall2 != null) {
                        try {
                            stringeeCall2.getLocalView().setMirror(isFrontCamera);
                        } catch (Exception ex) {
                            StringeeUtils.reportException(StringeeCallWrapper.class, ex);
                        }
                    }
                    handleResponse("switchCamera", true, null);
                }

                @Override
                public void onError(StringeeError stringeeError) {
                    super.onError(stringeeError);
                    canSwitch = true;
                    handleResponse("switchCamera", false, stringeeError.getMessage());
                }
            }, isFrontCamera ? rearCameraName : frontCameraName);
        }
    }

    private void handleResponse(String action, boolean isSuccess, String message) {
        Log.d(StringeeConstant.TAG, action + ": " + (isSuccess ? "success" : message));
        if (!isSuccess) {
            if (listener != null) {
                listener.onError(message);
            }
            release();
        }
    }

    private boolean isCallNotInitialized() {
        boolean isCallNotInitialized;
        if (!isVideoCall) {
            isCallNotInitialized = stringeeCall == null;
        } else {
            isCallNotInitialized = stringeeCall2 == null;
        }
        if (isCallNotInitialized) {
            if (listener != null) {
                listener.onError("call is not initialized");
            }
        }
        return isCallNotInitialized;
    }

    public void release() {
        Log.d(StringeeConstant.TAG, "release callManager");
        StringeeCommon.isInCall = false;
        audioManagerUtils.stopAudioManager();
        audioManagerUtils.stopRinging();
        if (callConfig != null && !callConfig.isCallOut()) {
            audioManagerUtils.stopWaitingSound();
        }
        audioManagerUtils.playEndCallSound();
        StringeeNotificationService.getInstance(context)
                .cancelNotification(StringeeNotificationService.INCOMING_CALL_NOTIFICATION_ID);
        if (timer != null) {
            timer.cancel();
        }
        if (statsTimer != null) {
            statsTimer.cancel();
        }
        stringeeCall = null;
        stringeeCall2 = null;
        audioManagerUtils.release();
        EventManager.getInstance().unregisterEvent(phoneStateReceiver);
        instance = null;
    }

    public String getName() {
        String name = "";
        if (!isIncomingCall) {
            name = callConfig.getTo();
            String toAlias = callConfig.getToAlias();
            if (StringeeUtils.isStringEmpty(name)) {
                name = toAlias;
            }
        } else {
            if (isVideoCall) {
                if (stringeeCall2 != null) {
                    name = stringeeCall2.getFromAlias();
                }
                if (StringeeUtils.isStringEmpty(name)) {
                    if (stringeeCall2 != null) {
                        name = stringeeCall2.getFrom();
                    }
                }
            } else {
                if (stringeeCall != null) {
                    name = stringeeCall.getFromAlias();
                }
                if (StringeeUtils.isStringEmpty(name)) {
                    if (stringeeCall != null) {
                        name = stringeeCall.getFrom();
                    }
                }
            }
        }
        return name;
    }

    public View getLocalView() {
        if (!isVideoCall) {
            return stringeeCall.getLocalView2();
        } else {
            return stringeeCall2.getLocalView2();
        }
    }

    public View getRemoteView() {
        if (!isVideoCall) {
            return stringeeCall.getRemoteView2();
        } else {
            return stringeeCall2.getRemoteView2();
        }
    }

    public void renderLocalView() {
        if (!isVideoCall) {
            stringeeCall.renderLocalView2(RendererCommon.ScalingType.SCALE_ASPECT_FIT);
        } else {
            stringeeCall2.renderLocalView2(RendererCommon.ScalingType.SCALE_ASPECT_FIT);
        }
    }

    public void renderRemoteView() {
        if (!isVideoCall) {
            stringeeCall.renderRemoteView2(RendererCommon.ScalingType.SCALE_ASPECT_FIT);
        } else {
            stringeeCall2.renderRemoteView2(RendererCommon.ScalingType.SCALE_ASPECT_FIT);
        }
    }

    public CallConfig getCallConfig() {
        return callConfig;
    }

    public boolean isNeedResumeVideo() {
        return needResumeVideo;
    }

    private void callStarted() {
        audioManagerUtils.stopWaitingSound();
        if (isIncomingCall()) {
            startAudioManager();
        }
        if (timer == null) {
            long startTime = System.currentTimeMillis();

            timer = new Timer();
            TimerTask timerTask = new TimerTask() {
                @Override
                public void run() {
                    StringeeUtils.runOnUiThread(() -> {
                        if (listener != null) {
                            listener.onTimer(StringeeDateTimeUtils.getInstance()
                                    .getCallTime(System.currentTimeMillis(), startTime));
                        }
                    });
                }
            };
            timer.schedule(timerTask, 0, 1000);
        }
        if (statsTimer == null) {
            statsTimer = new Timer();
            TimerTask timerTask = new TimerTask() {
                @Override
                public void run() {
                    StringeeUtils.runOnUiThread(() -> {
                        if (isVideoCall) {
                            if (stringeeCall2 != null) {
                                stringeeCall2.getStats(statsReport -> StringeeUtils.runOnUiThread(
                                        () -> checkCallStats2(statsReport)));
                            }
                        } else {
                            if (stringeeCall != null) {
                                stringeeCall.getStats(statsReport -> StringeeUtils.runOnUiThread(
                                        () -> checkCallStats(statsReport)));
                            }
                        }
                    });
                }
            };
            statsTimer.schedule(timerTask, 0, 2000);
        }
    }

    private void checkCallStats2(StringeeCall2.StringeeCallStats stats) {
        if (stringeeCall2 == null) {
            return;
        }
        double videoTimestamp = stats.timeStamp / 1000d;
        long bytesReceived =
                stringeeCall2.isVideoCall() ? (long) stats.videoBytesReceived : (long) stats.callBytesReceived;
        //initialize values
        if (prevCallTimestamp == 0) {
            prevCallTimestamp = videoTimestamp;
            prevCallBytes = bytesReceived;
        } else {
            //calculate video bandwidth
            callBw = (long) ((8 * (bytesReceived - prevCallBytes)) /
                    (videoTimestamp - prevCallTimestamp));
            prevCallTimestamp = videoTimestamp;
            prevCallBytes = bytesReceived;

            if (listener != null) {
                listener.onCallBandwidth(callBw);
            }
        }
    }

    private void checkCallStats(StringeeCall.StringeeCallStats stats) {
        if (stringeeCall == null) {
            return;
        }
        double videoTimestamp = stats.timeStamp / 1000d;
        long bytesReceived =
                stringeeCall.isVideoCall() ? (long) stats.videoBytesReceived : (long) stats.callBytesReceived;
        //initialize values
        if (prevCallTimestamp == 0) {
            prevCallTimestamp = videoTimestamp;
            prevCallBytes = bytesReceived;
        } else {
            //calculate video bandwidth
            callBw = (long) ((8 * (bytesReceived - prevCallBytes)) /
                    (videoTimestamp - prevCallTimestamp));
            prevCallTimestamp = videoTimestamp;
            prevCallBytes = bytesReceived;

            if (listener != null) {
                listener.onCallBandwidth(callBw);
            }
        }
    }

    public void resumeVideo() {
        if (isCallNotInitialized()) {
            if (listener != null) {
                listener.onCallStatus(CallStatus.ENDED);
            }
            release();
            return;
        }
        if (!needResumeVideo) {
            return;
        }
        needResumeVideo = false;
        if (isVideoCall) {
            stringeeCall2.resumeVideo();
        }
    }
}
