package com.stringee.widget.common;

import android.annotation.SuppressLint;
import android.content.Context;
import android.media.AudioAttributes;
import android.media.AudioAttributes.Builder;
import android.media.AudioDeviceInfo;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.os.VibrationEffect;
import android.os.Vibrator;

import com.stringee.common.StringeeAudioManager;
import com.stringee.widget.R;

@SuppressLint("MissingPermission")
public class StringeeAudioManagerUtils {
    private static volatile StringeeAudioManagerUtils instance;
    private final Context context;
    private OnAudioEvents audioEvents;
    private final Uri incomingRingtoneUri;
    private final Uri endCallPlayerUri;
    private final Uri waitingPlayerUri;
    private final MediaPlayer incomingRingtone;
    private final MediaPlayer endCallPlayer;
    private final MediaPlayer ringingPlayer;
    private final Vibrator incomingVibrator;
    private StringeeAudioManager audioManager;
    private AudioManager am;
    private int previousAudioModel;
    private boolean previousSpeaker;

    public interface OnAudioEvents {
        void onAudioEvents(StringeeAudioManager.AudioDevice selectedAudioDevice);
    }

    public static StringeeAudioManagerUtils getInstance(Context context) {
        if (instance == null) {
            synchronized (StringeeAudioManagerUtils.class) {
                if (instance == null) {
                    instance = new StringeeAudioManagerUtils(context.getApplicationContext());
                }
            }
        }
        return instance;
    }

    public StringeeAudioManagerUtils(Context context) {
        this.context = context.getApplicationContext();
        this.incomingRingtoneUri = Uri.parse(RingtoneManager.getDefaultUri(RingtoneManager.TYPE_RINGTONE).toString());
        this.endCallPlayerUri = Uri.parse("android.resource://" + context.getPackageName() + "/" + R.raw.stringee_call_end);
        this.waitingPlayerUri = Uri.parse("android.resource://" + context.getPackageName() + "/" + R.raw.stringee_call_ringing);

        this.incomingRingtone = new MediaPlayer();
        this.ringingPlayer = new MediaPlayer();
        this.endCallPlayer = new MediaPlayer();

        this.incomingVibrator = (Vibrator) context.getSystemService(Context.VIBRATOR_SERVICE);
    }

    public void startAudioManager() {
        StringeeUtils.runOnUiThread(() -> {
            if (audioManager == null) {
                audioManager = StringeeAudioManager.create(context);
            }

            audioManager.start((selectedAudioDevice, availableAudioDevices) -> audioEvents.onAudioEvents(selectedAudioDevice));
        });
    }

    public void setAudioEvents(OnAudioEvents onAudioEvents) {
        StringeeUtils.runOnUiThread(() -> audioEvents = onAudioEvents);
    }

    public void stopAudioManager() {
        StringeeUtils.runOnUiThread(() -> {
            if (audioManager != null) {
                audioManager.stop();
            }
        });
    }

    public void playWaitingSound() {
        StringeeUtils.runOnUiThread(() -> {
            if (ringingPlayer == null) {
                return;
            }
            if (ringingPlayer.isPlaying() || ringingPlayer.isLooping()) {
                ringingPlayer.stop();
                ringingPlayer.reset();
            }
            ringingPlayer.setOnPreparedListener(mediaPlayer -> ringingPlayer.start());
            ringingPlayer.setLooping(true);
            ringingPlayer.setAudioStreamType(AudioManager.STREAM_VOICE_CALL);
            try {
                ringingPlayer.setDataSource(context, waitingPlayerUri);
                ringingPlayer.prepareAsync();
            } catch (Exception e) {
                stopWaitingSound();
            }
        });
    }

    public void stopWaitingSound() {
        StringeeUtils.runOnUiThread(() -> {
            if (ringingPlayer != null) {
                ringingPlayer.stop();
                ringingPlayer.reset();
            }
        });
    }

    public void playEndCallSound() {
        StringeeUtils.runOnUiThread(() -> {
            if (endCallPlayer != null) {
                if (endCallPlayer.isPlaying()) {
                    return;
                }
                endCallPlayer.setOnPreparedListener(mediaPlayer -> endCallPlayer.start());
                endCallPlayer.setLooping(false);
                endCallPlayer.setAudioStreamType(AudioManager.STREAM_VOICE_CALL);
                try {
                    endCallPlayer.setDataSource(context, endCallPlayerUri);
                    endCallPlayer.prepareAsync();
                    endCallPlayer.setOnCompletionListener(mediaPlayer -> stopEndCallSound());
                } catch (Exception e) {
                    stopEndCallSound();
                }
            }
        });
    }

    public void stopEndCallSound() {
        StringeeUtils.runOnUiThread(() -> {
            if (endCallPlayer != null) {
                endCallPlayer.stop();
                endCallPlayer.reset();
            }
        });
    }

    public void startRingtoneAndVibration() {
        StringeeUtils.runOnUiThread(() -> {
            am = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
            previousAudioModel = am.getMode();
            previousSpeaker = am.isSpeakerphoneOn();
            am.setMode(AudioManager.MODE_RINGTONE);
            am.setSpeakerphoneOn(true);
            boolean isHeadsetPlugged = isWiredHeadsetOn();
            boolean needRing = am.getRingerMode() == AudioManager.RINGER_MODE_NORMAL;
            boolean needVibrate = am.getRingerMode() != AudioManager.RINGER_MODE_SILENT;

            if (needRing) {
                if (incomingRingtone != null) {
                    if (incomingRingtone.isPlaying() || incomingRingtone.isLooping()) {
                        incomingRingtone.stop();
                        incomingRingtone.reset();
                    }
                    incomingRingtone.setOnPreparedListener(mediaPlayer -> incomingRingtone.start());
                    incomingRingtone.setLooping(true);
                    if (isHeadsetPlugged) {
                        incomingRingtone.setAudioStreamType(AudioManager.STREAM_VOICE_CALL);
                    } else {
                        incomingRingtone.setAudioStreamType(AudioManager.STREAM_RING);
                    }
                    try {
                        incomingRingtone.setDataSource(context, incomingRingtoneUri);
                        incomingRingtone.prepareAsync();
                    } catch (Exception e) {
                        incomingRingtone.stop();
                        incomingRingtone.reset();
                    }
                }
            }
            if (needVibrate && incomingVibrator != null) {
                if (VERSION.SDK_INT >= VERSION_CODES.O) {
                    incomingVibrator.vibrate(VibrationEffect.createWaveform(new long[]{0, 350, 500}, 0), new Builder().setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION).setUsage(AudioAttributes.USAGE_NOTIFICATION_RINGTONE).build());
                } else {
                    incomingVibrator.vibrate(new long[]{0, 350, 500}, 0);
                }
            }
        });
    }

    private boolean isWiredHeadsetOn() {
        if (VERSION.SDK_INT < VERSION_CODES.M) {
            return am.isWiredHeadsetOn();
        } else {
            final AudioDeviceInfo[] devices = am.getDevices(AudioManager.GET_DEVICES_OUTPUTS);
            for (AudioDeviceInfo device : devices) {
                final int type = device.getType();
                if (type == AudioDeviceInfo.TYPE_WIRED_HEADSET || type == AudioDeviceInfo.TYPE_WIRED_HEADPHONES || type == AudioDeviceInfo.TYPE_USB_DEVICE) {
                    return true;
                }
            }
        }
        return false;
    }

    public void stopRinging() {
        StringeeUtils.runOnUiThread(() -> {
            if (am != null) {
                am.setMode(previousAudioModel);
                am.setSpeakerphoneOn(previousSpeaker);
            }
            if (incomingRingtone != null) {
                incomingRingtone.stop();
                incomingRingtone.reset();
            }
            if (incomingVibrator != null) {
                incomingVibrator.cancel();
            }
        });
    }

    public void setSpeakerphoneOn(boolean on) {
        StringeeUtils.runOnUiThread(() -> {
            if (audioManager != null) {
                audioManager.setSpeakerphoneOn(on);
            }
        });
    }

    public void setBluetoothScoOn(boolean on) {
        StringeeUtils.runOnUiThread(() -> {
            if (audioManager != null) {
                audioManager.setBluetoothScoOn(on);
            }
        });
    }

    public void release() {
        StringeeUtils.runOnUiThread(() -> {
            if (instance != null) {
                instance = null;
            }
        });
    }
}
