package com.stringee.widget.common;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;

import androidx.core.app.NotificationCompat;
import androidx.core.app.Person;
import androidx.core.graphics.drawable.IconCompat;

import com.stringee.widget.R;
import com.stringee.widget.call.receiver.RejectCallService;
import com.stringee.widget.call.StringeeCallActivity;

import java.util.Random;

public class StringeeNotificationService {
    private static volatile StringeeNotificationService instance;
    private final Context context;
    private final NotificationManager notificationManager;
    private static final String CALL_CHANNEL_ID = "com.stringee.widget.call.notification";
    private static final String CALL_CHANNEL_NAME = "Stringee Widget Call Channel";
    private static final String CHANNEL_DESC = "Widget channel for notification";
    public static final int INCOMING_CALL_NOTIFICATION_ID = 10021987;

    public StringeeNotificationService(Context context) {
        this.context = context.getApplicationContext();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            notificationManager = context.getSystemService(NotificationManager.class);
        } else {
            notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        }
    }

    public static synchronized StringeeNotificationService getInstance(Context context) {
        if (instance == null) {
            synchronized (StringeeNotificationService.class) {
                if (instance == null && context != null) {
                    instance = new StringeeNotificationService(context.getApplicationContext());
                }
            }
        }
        return instance;
    }

    public void cancelNotification(int notificationId) {
        if (notificationManager != null) {
            notificationManager.cancel(notificationId);
        }
    }

    private String createCallNotificationChannel() {
        int channelIndex = StringeePrefUtils.getInstance(context).getInt(StringeeConstant.PREF_CALL_CHANNEL_ID_INDEX, 0);
        String channelId = CALL_CHANNEL_ID + channelIndex;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            //delete old channel
            NotificationChannel channel = notificationManager.getNotificationChannel(channelId);
            if (channel != null && !(channel.getImportance() == NotificationManager.IMPORTANCE_MAX || channel.getImportance() == NotificationManager.IMPORTANCE_HIGH)) {
                notificationManager.deleteNotificationChannel(channelId);
                channelIndex = channelIndex + 1;
                channelId = CALL_CHANNEL_ID + channelIndex;
            }
            //create new channel
            channel = new NotificationChannel(channelId, CALL_CHANNEL_NAME, NotificationManager.IMPORTANCE_HIGH);
            StringeePrefUtils.getInstance(context).putInt(StringeeConstant.PREF_CALL_CHANNEL_ID_INDEX, channelIndex);
            channel.setDescription(CHANNEL_DESC);
            channel.setSound(null, null);
            channel.enableVibration(true);
            notificationManager.createNotificationChannel(channel);
        }
        return channelId;
    }

    public void notifyIncomingCall(String from, String fromAlias) {
        String channelId = createCallNotificationChannel();

        int flag = PendingIntent.FLAG_UPDATE_CURRENT;
        if (VERSION.SDK_INT >= VERSION_CODES.S) {
            flag = PendingIntent.FLAG_IMMUTABLE;
        }

        Intent fullScreenIntent = new Intent(context, StringeeCallActivity.class);
        PendingIntent fullScreenPendingIntent = PendingIntent.getActivity(context, (int) (System.currentTimeMillis() & 0xfffffff), fullScreenIntent, flag);

        Intent rejectIntent = new Intent(context, RejectCallService.class);
        PendingIntent rejectPendingIntent = PendingIntent.getService(context, (int) (System.currentTimeMillis() & 0xfffffff), rejectIntent, flag);

        Intent answerIntent = new Intent(context, StringeeCallActivity.class);
        answerIntent.putExtra(StringeeConstant.PARAM_ANSWER, true);
        PendingIntent answerPendingIntent = PendingIntent.getActivity(context, (int) (System.currentTimeMillis() & 0xfffffff), answerIntent, flag);

        String name = fromAlias;
        if (StringeeUtils.isStringEmpty(name)) {
            name = from;
        }
        Bitmap avatar = StringeeUtils.getAvatar(context, fromAlias, new Random().nextInt(Integer.MAX_VALUE));

        Person.Builder personBuilder = new Person.Builder();
        personBuilder.setName(name);
        personBuilder.setImportant(true);
        personBuilder.setIcon(avatar != null ? IconCompat.createWithBitmap(avatar) : IconCompat.createWithResource(context, R.mipmap.icon));
        Person person = personBuilder.build();

        NotificationCompat.Builder notificationBuilder = new NotificationCompat.Builder(context, channelId);
        notificationBuilder.setStyle(NotificationCompat.CallStyle.forIncomingCall(person, rejectPendingIntent, answerPendingIntent).setIsVideo(false));
        notificationBuilder.addPerson(person);
        notificationBuilder.setLargeIcon(avatar != null ? avatar : BitmapFactory.decodeResource(context.getResources(), R.mipmap.icon));
        if (VERSION.SDK_INT >= VERSION_CODES.M) {
            notificationBuilder.setSmallIcon(IconCompat.createWithResource(context, R.mipmap.icon));
        } else {
            notificationBuilder.setSmallIcon(R.mipmap.icon);
        }
        notificationBuilder.setContentTitle(name);
        notificationBuilder.setContentText(from);
        notificationBuilder.setSubText(null);
        notificationBuilder.setContentIntent(fullScreenPendingIntent);
        notificationBuilder.setOngoing(true);
        notificationBuilder.setPriority(NotificationCompat.PRIORITY_MAX);
        notificationBuilder.setShowWhen(false);
        notificationBuilder.setVibrate(new long[0]);
        notificationBuilder.setCategory(NotificationCompat.CATEGORY_CALL);
        notificationBuilder.setFullScreenIntent(fullScreenPendingIntent, true);

        Notification notification = notificationBuilder.build();
        notificationManager.notify(INCOMING_CALL_NOTIFICATION_ID, notification);
    }
}
