package com.stringee.widget.common;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Base64;
import android.util.DisplayMetrics;
import android.util.Log;
import android.util.LruCache;
import android.view.Gravity;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.stringee.widget.R;
import com.stringee.widget.common.custom_view.text_drawable.TextDrawable;

import org.json.JSONObject;

import java.nio.charset.StandardCharsets;
import java.util.List;

public class StringeeUtils {

    public static void runOnUiThread(Runnable runnable) {
        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(runnable);
    }

    public static void reportMessage(Context context, int resId) {
        reportMessage(context, context.getString(resId));
    }

    public static void reportMessage(Context context, String message) {
        Toast toast = Toast.makeText(context, message, Toast.LENGTH_LONG);
        toast.setGravity(Gravity.CENTER, 0, 0);
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.R) {
            View toastView = toast.getView();
            if (toastView != null) {
                TextView v = toastView.findViewById(android.R.id.message);
                if (v != null) {
                    v.setGravity(Gravity.CENTER);
                }
            }
        }
        toast.show();
    }

    public static int dpToPx(Context context, int dp) {
        DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
        return Math.round(dp * (displayMetrics.xdpi / DisplayMetrics.DENSITY_DEFAULT));
    }

    public static boolean isStringEmpty(@Nullable String text) {
        if (text != null) {
            return text.trim().isEmpty();
        } else {
            return true;
        }
    }

    public static boolean isListEmpty(@Nullable List<?> list) {
        if (list != null) {
            return list.isEmpty();
        } else {
            return true;
        }
    }

    public static String getUserId(String connectToken) {
        if (StringeeUtils.isStringEmpty(connectToken)) {
            return null;
        }
        String userId = null;
        try {
            JSONObject jsonObject = decodeJWTPayload(connectToken);
            if (jsonObject != null) {
                userId = jsonObject.optString("userId");
            }
        } catch (Exception e) {
            reportException(StringeeUtils.class, e);
        }
        return userId;
    }

    public static JSONObject decodeJWTPayload(String jwt) {
        try {
            String[] parts = jwt.split("\\.");
            if (parts.length < 2) {
                return null;
            }

            // Lấy payload (phần thứ 2)
            StringBuilder payload = new StringBuilder(parts[1]);

            // Thêm padding nếu cần
            while (payload.length() % 4 != 0) {
                payload.append("=");
            }

            // Decode Base64 (JWT sử dụng Base64URL)
            byte[] decodedBytes =
                    Base64.decode(payload.toString(), Base64.URL_SAFE | Base64.NO_WRAP);
            String decodedString = new String(decodedBytes, StandardCharsets.UTF_8);

            // Parse JSON
            return new JSONObject(decodedString);
        } catch (Exception e) {
            reportException(StringeeUtils.class, e);
            return null;
        }
    }

    public static <T> void reportException(@NonNull Class<T> clazz, Exception exception) {
        Log.e(StringeeConstant.TAG + " exception", clazz.getName(), exception);
    }

    public static Bitmap getAvatar(Context context, String text, int index) {
        if (isStringEmpty(text)) {
            return null;
        }
        String character = getCharacterOfContact(text);
        return createBitmapByCharacter(context, character, index);
    }

    public static String getCharacterOfContact(String text) {
        String character = "";
        if (!isStringEmpty(text)) {
            text = convertVnToEn(text);
            text = text.replaceAll("[;/:*?\"<>|&']", "");
            String[] nameArray = text.trim().split("\\s+");
            if (nameArray.length > 1) {
                String char0 = nameArray[0];
                String char1 = nameArray[1];
                if (!isStringEmpty(char0) && !isStringEmpty(char1)) {
                    character = String.valueOf(char0.charAt(0)) + char1.charAt(0);
                } else if (!isStringEmpty(char0)) {
                    if (char0.trim().length() > 2) {
                        character = char0.substring(0, 2);
                    } else {
                        character = char0.trim();
                    }
                } else if (!isStringEmpty(char1)) {
                    if (char1.trim().length() > 2) {
                        character = char1.substring(0, 2);
                    } else {
                        character = char1.trim();
                    }
                } else {
                    if (text.trim().length() > 2) {
                        character = text.substring(0, 2);
                    } else {
                        character = text.trim();
                    }
                }
            } else if (nameArray.length > 0) {
                if (text.trim().length() > 2) {
                    character = text.substring(0, 2);
                } else {
                    character = text.trim();
                }
            }
        }
        return character.toUpperCase();
    }

    public static Bitmap createBitmapByCharacter(Context context, String character, int index) {
        if (StringeeCommon.memoryCache == null) {
            StringeeCommon.memoryCache =
                    new LruCache<String, Bitmap>((int) (Runtime.getRuntime().maxMemory() / 8)) {
                        @Override
                        protected int sizeOf(String key, Bitmap bitmap) {
                            return bitmap.getByteCount();
                        }
                    };
        }
        Bitmap bitmap = StringeeCommon.memoryCache.get(character);
        if (bitmap == null) {
            TextDrawable textDrawable = TextDrawable.builder()
                    .beginConfig()
                    .fontSize(40)
                    .toUpperCase()
                    .textColor(ContextCompat.getColor(context, R.color.white))
                    .endConfig()
                    .buildRound(character,
                            ContextCompat.getColor(context, AlphaNumberColorUtil.getColor(index)));

            bitmap = Bitmap.createBitmap(80, 80, Bitmap.Config.ARGB_8888);
            Canvas canvas = new Canvas(bitmap);
            textDrawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
            textDrawable.draw(canvas);

            long maxMemory = Runtime.getRuntime().maxMemory();
            long totalMemory = Runtime.getRuntime().totalMemory();
            long usedMemory = totalMemory - Runtime.getRuntime().freeMemory();

            if (usedMemory > 0.75 * maxMemory) {
                StringeeCommon.memoryCache.evictAll();
            }

            StringeeCommon.memoryCache.put(character, bitmap);
        }
        return bitmap;
    }

    public static String convertVnToEn(String s) {
        if (isStringEmpty(s)) {
            return "";
        }
        s = s.toLowerCase().trim();
        s = s.replaceAll("[àáạảãâầấậẩẫăằắặẳẵ]", "a")
                .replaceAll("[èéẹẻẽêềếệểễ]", "e")
                .replaceAll("[ìíịỉĩ]", "i")
                .replaceAll("[òóọỏõôồốộổỗơờớợởỡ]", "o")
                .replaceAll("[ùúụủũưừứựửữ]", "u")
                .replaceAll("[ỳýỵỷỹ]", "y")
                .replaceAll("đ", "d");
        return s;
    }
}
