package com.stringee.widget.common.custom_view.draggable_view;

import android.content.Context;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;

public class DraggableFrameLayout extends FrameLayout {

    private float lastTouchX;
    private float lastTouchY;

    private int leftBound;
    private int rightBound;
    private int topBound;
    private int bottomBound;

    private int screenWidth;
    private int screenHeight;
    private boolean allowDrag = true;

    public DraggableFrameLayout(Context context) {
        super(context);
        initBounds(context);
    }

    public DraggableFrameLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        initBounds(context);
    }

    public DraggableFrameLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initBounds(context);
    }

    public void setAllowDrag(boolean allowDrag) {
        this.allowDrag = allowDrag;
    }

    private void initBounds(Context context) {
        post(() -> {
            screenWidth = context.getResources().getDisplayMetrics().widthPixels;
            screenHeight = context.getResources().getDisplayMetrics().heightPixels;
            if (allowDrag) {
                MarginLayoutParams layoutParams = (MarginLayoutParams) this.getLayoutParams();

                View childView = getChildAt(0);
                leftBound = layoutParams.leftMargin;
                topBound = layoutParams.topMargin;
                rightBound = screenWidth - (childView != null ? childView.getWidth() : getWidth()) - layoutParams.rightMargin;
                bottomBound = screenHeight - (childView != null ? childView.getHeight() : getHeight()) - layoutParams.bottomMargin;
            }
        });
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        post(() -> {
            if (allowDrag) {
                MarginLayoutParams layoutParams = (MarginLayoutParams) this.getLayoutParams();

                leftBound = layoutParams.leftMargin;
                topBound = layoutParams.topMargin;
                View childView = getChildAt(0);
                rightBound = screenWidth - (childView != null ? childView.getWidth() : getWidth()) - layoutParams.rightMargin;
                bottomBound = screenHeight - (childView != null ? childView.getHeight() : getHeight()) - layoutParams.bottomMargin;
            }
        });
    }

    @Override
    public boolean performClick() {
        return super.performClick();
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (allowDrag) {
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    lastTouchX = event.getX();
                    lastTouchY = event.getY();
                    return true;
                case MotionEvent.ACTION_MOVE:
                    float dx = event.getX() - lastTouchX;
                    float dy = event.getY() - lastTouchY;

                    float newX = getX() + dx;
                    float newY = getY() + dy;

                    if (newX < leftBound) newX = leftBound;
                    if (newX > rightBound) newX = rightBound;
                    if (newY < topBound) newY = topBound;
                    if (newY > bottomBound) newY = bottomBound;

                    setX(newX);
                    setY(newY);

                    return true;
                case MotionEvent.ACTION_UP:
                    performClick();
                    return true;
                default:
                    return super.onTouchEvent(event);
            }
        } else {
            return super.onTouchEvent(event);
        }
    }
}