package org.codeability.sharing.plugins.api.search;

import java.util.List;
import java.util.stream.Collectors;


/**
 * transfers the queried meta data 
 * @author Michael Breu
 *
 */
public class SearchInputMetadataDTO {
    private String programmingLanguage;
    private String keyword;
    private List<String> naturalLanguage;
    private String license;
    private String author;
    private String learningResourceType;
    private String format;
    private String parentId;

    /**
     * just for JSON
     */
    public SearchInputMetadataDTO() {
    	// JSON
    }

    /**
     * a convenience constructor.
     * @param programmingLanguage the programmingLanguage
     * @param keyword the keywords
     * @param naturalLanguage the list of requested languages
     * @param license the license(s)
     * @param author the author
     * @param format the format
     */
    public SearchInputMetadataDTO(String programmingLanguage,
                                  String keyword,
                                  List<String> naturalLanguage,
                                  String license,
                                  String author,
                                  String format) {
        this.programmingLanguage = mapEmptyString(programmingLanguage);
        this.keyword = mapEmptyString(keyword);
        this.license = mapEmptyString(license);
        this.author = mapEmptyString(author);
        this.naturalLanguage = naturalLanguage;
        this.format = mapEmptyString(format);
    }

    /**
     * just a helper method to convert from list to string
     * @param str the array of strings
     * @return the concatenated strings
     */
    private static List<String> mapEmptyString(List<String> str) {
    	if(str==null) return null;
    	return str.stream().map(s -> "".equals(s) ? null : s).toList();
    }

    /**
     * just a helper method to deal with null strings
     * @param str the string
     * @return an empty string.
     */
    private static String mapEmptyString(String str) {
        return "".equals(str) ? null : str;
    }

    /**
     * 
     * @return the format
     */
    public String getFormat() {
        return format;
    }

    /**
     * sets the format
     * @param format the format
     */
    public void setFormat(String format) {
        this.format = format;
    }

    /**
     * checks for minimal meta data
     * @return the meta data
     */
    public boolean hasMetadataInput() {
        return !(programmingLanguage == null
            && keyword == null
            && author == null
            && license == null);
    }

    /**
     * the programming language
     * @return the programming language
     */
    public String getProgrammingLanguage() {
        return programmingLanguage;
    }

    /**
     * 
     * @param programmingLanguage the programming language
     */
    public void setProgrammingLanguage(String programmingLanguage) {
        this.programmingLanguage = mapEmptyString(programmingLanguage);
    }

    /**
     * a string containing the keywords
     * @return a string containing the keywords
     */
    public String getKeyword() {
        return keyword;
    }

    /**
     * 
     * @param keyword a string containing the keywords
     */
    public void setKeyword(String keyword) {
        this.keyword = mapEmptyString(keyword);
    }

    /**
     * 
     * @return the selected natural languages
     */
    public List<String> getNaturalLanguage() {
        return naturalLanguage;
    }

    /**
     * 
     * @return the selected natural languages
     */
    public String getNaturalLanguagesAsString() {
    	return naturalLanguage==null?"null":
    		naturalLanguage.stream().collect(Collectors.joining(", "));
    }

    /**
     * 
     * @param naturalLanguages the list of selected natural languages
     */
    public void setNaturalLanguage(List<String> naturalLanguages) {
        this.naturalLanguage = mapEmptyString(naturalLanguages);
    }

    /**
     * the license
     * @return the requested license
     */
    public String getLicense() {
        return license;
    }

    /**
     * 
     * @param license the requested license
     */
    public void setLicense(String license) {
        this.license = mapEmptyString(license);
    }

    /**
     * 
     * @return the author
     */
    public String getAuthor() {
        return author;
    }

    /**
     * setter for author
     * @param author the author
     */
    public void setAuthor(String author) {
        this.author = mapEmptyString(author);
    }

    public String getLearningResourceType() {
        return learningResourceType;
    }

    public void setLearningResourceType(final String learningResourceType) {
        this.learningResourceType = learningResourceType;
    }

    /**
	 * @return the parentId
	 */
	public String getParentId() {
		return parentId;
	}

	/**
	 * @param parentId the parentId to set
	 */
	public void setParentId(String parentId) {
		this.parentId = parentId;
	}

	/**
	 * to string
	 */
	@Override
    public String toString() {
        return "programmingLanguage: " + this.programmingLanguage
            + ", keyword: " + this.keyword
            + ", naturalLanguage: " + getNaturalLanguagesAsString()
            + ", license: " + this.license
            + ", author: " + this.author
            + ", format: " + this.format;
    }
}
