package org.codeability.sharing.plugins.api.search.util;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;

import java.io.IOException;
import java.text.ParseException;
import java.util.Objects;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.codeability.sharing.plugins.api.search.util.ExerciseId.ExerciseIdDeserializer;
import org.codeability.sharing.plugins.api.search.util.ExerciseId.ExerciseIdSerializer;

/**
 * represents an exercise id. Also provides some utility methods to parse and
 * unparse exercise Ids
 * 
 * @author Michael Breu
 *
 */
@JsonSerialize(using = ExerciseIdSerializer.class)
@JsonDeserialize(using = ExerciseIdDeserializer.class)
public class ExerciseId {

	public static class ExerciseIdDeserializer
			extends
				StdDeserializer<ExerciseId> {

		/**
		 * serialVersionUID
		 */
		private static final long serialVersionUID = 1L;

		protected ExerciseIdDeserializer() {
			super(ExerciseId.class);
		}

		@Override
		public ExerciseId deserialize(JsonParser p, DeserializationContext ctxt)
				throws IOException {
			JsonNode node = p.getCodec().readTree(p);
			String id = node.asText();
			try {
				return ExerciseId.fromString(id);
			} catch (ParseException e) {
				throw new IllegalArgumentException(
						"Cannot parse exerciseId " + id);
			}
		}

	}
	public static class ExerciseIdSerializer
			extends
				StdSerializer<ExerciseId>
	{

		/**
		 * serialVersionUID
		 */
		private static final long serialVersionUID = 1L;

		protected ExerciseIdSerializer() {
			super(ExerciseId.class, true /* ignored dummy */ );
		}

		@Override
		public void serialize(ExerciseId value, JsonGenerator gen,
				SerializerProvider provider) throws IOException {
			gen.writeString(value.toString());
		}

	}

	private String projectId; // should be numeric
	private String path; // a path with slashes (root slash omitted)
	private String gitUrl; // optional: a url to a git repository

	public ExerciseId() {

	}

	public ExerciseId(String projectId, String path) {
		this(projectId, path, null);
	}

	public ExerciseId(String projectId, String path, String gitUrl) {
		super();
		if (projectId == null && gitUrl == null) {
			throw new IllegalArgumentException(
					"project id and gitUrl must not be both null");
		}
		if (projectId != null) {
			this.projectId = projectId.trim();
			if (!StringUtils.isNumeric(projectId)) {
				// in theory this test is still to weak.
				throw new IllegalArgumentException(
						"project id must be numeric");
			}
		}

		this.path = (StringUtils.isEmpty(path)) ? null : path.trim();
		this.gitUrl = (StringUtils.isEmpty(gitUrl)) ? null : gitUrl.trim();
	}

	/**
	 * returns the root project Id.
	 * 
	 * @return the root project Id
	 */
	public ExerciseId getRootExerciseId() {
		return new ExerciseId(projectId, "", gitUrl);
	}

	/**
	 * @return the projectId
	 */
	public String getProjectId() {
		return projectId;
	}

	/**
	 * @return the path
	 */
	public String getPath() {
		return path;
	}

	public String getGitUrl() {
		return gitUrl;
	}

	/**
	 * adds an extra path to existing path
	 * 
	 * @param path
	 * @return
	 */
	public String extendPath(String extraPath) {
		if (path == null || path == "")
			return extraPath;
		if (extraPath == null || extraPath == "")
			return path;
		if (extraPath.startsWith("/"))
			return "/" + path + extraPath;
		else
			return path + "/" + extraPath;
	}

	// protected final static Pattern ExerciseIdPattern =
	// Pattern.compile("^(http[^[]+)?(\\[(\\d*)\\])?(.+)?$");
	protected final static Pattern ExerciseIdPattern = Pattern
			.compile("^(http[^\\[]+)?(\\[(\\d*)\\])?(.+)?$");
	protected final static Pattern ExerciseIdPatternOld = Pattern
			.compile("(\\d+)(:(.*))?");

	@Override
	public String toString() {
		if (!StringUtils.isEmpty(gitUrl)) {
			if (StringUtils.isEmpty(path))
				return gitUrl;
			else
				return gitUrl + "[]" + path;
		}
		if (StringUtils.isEmpty(path))
			return "[" + projectId + "]";
		return "[" + projectId + "]" + path;
	}

	public boolean isDescendantOf(ExerciseId parent) {
		if (parent.gitUrl != null && !parent.gitUrl.equals(this.gitUrl))
			return false;
		if (parent.gitUrl == null && this.gitUrl != null)
			return false;
		// we are in same repository
		if (!StringUtils.equals(parent.projectId, this.projectId))
			return false;
		if (parent.path == null)
			return true;
		if (this.path == null)
			return false;
		return this.path.startsWith(parent.path);
	}

	/**
	 * parses the external representation of an ExerciseId.
	 * 
	 * @param externalRepresentation
	 * @return
	 * @throws ParseException
	 */
	public static ExerciseId fromString(String externalRepresentation)
			throws ParseException {
		final Matcher ma = ExerciseIdPattern.matcher(externalRepresentation);
		if (!ma.matches()) {
			return fromStringOld(externalRepresentation);
		}
		final MatchResult matchResult = ma.toMatchResult();
		String gitUrl = matchResult.group(1);
		String projectId = matchResult.group(3);
		if (projectId == null) {
			return fromStringOld(externalRepresentation);
		}
		if ("".equals(projectId)) {
			projectId = null;
		}
		String path = matchResult.group(4);

		while (path != null && path.startsWith("/")) {
			path = path.substring(1);
		}

		return new ExerciseId(projectId, path, gitUrl);
	}

	/**
	 * parses the external representation of an ExerciseId (old format without
	 * brackets).
	 * 
	 * @param externalRepresentation
	 * @return
	 * @throws ParseException
	 */
	public static ExerciseId fromStringOld(String externalRepresentation)
			throws ParseException {
		final Matcher ma = ExerciseIdPatternOld.matcher(externalRepresentation);
		if (!ma.matches())
			throw new ParseException("Cannot parse " + externalRepresentation,
					0);
		final MatchResult matchResult = ma.toMatchResult();

		String projectId = matchResult.group(1);
		String path = matchResult.group(3);

		while (path != null && path.startsWith("/")) {
			path = path.substring(1);
		}

		return new ExerciseId(projectId, path);
	}

	@Override
	public int hashCode() {
		return Objects.hash(gitUrl, path, projectId);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ExerciseId other = (ExerciseId) obj;
		return Objects.equals(gitUrl, other.gitUrl)
				&& Objects.equals(path, other.path)
				&& Objects.equals(projectId, other.projectId);
	}

}
