package org.codeability.sharing.plugins.api.util;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * just a helper class to have a defined checksum calculator.
 */
public class SecretChecksumCalculator {

	private SecretChecksumCalculator() {
		// not instantiable
	}

	/**
	 * takes an array of strings (together with the secret key and calculates a
	 * checksum, returns it as an hex string.
	 * 
	 * @param params
	 *            array of url parameters
	 * @param secretKey
	 *            the shared secret key
	 * @return
	 */
	public static String calculateChecksum(final Map<String, String> params,
			final String secretKey) {
		try {
			SortedMap<String, String> sortedParams = new TreeMap<>();
			// we sort the param array, in order to be
			// indepentent of the sequence :-). Reduces
			// entropy slightly :-)
			sortedParams.putAll(params);
			// create MD5 MessageDigest-Instance
			MessageDigest md = MessageDigest.getInstance("MD5");

			// add secret key to calculation
			md.update(secretKey.getBytes());

			// add all strings to calculation
			for (String str : sortedParams.values()) {
				md.update(str.getBytes());
			}

			// get check sum
			byte[] checksumBytes = md.digest();

			// return as hexa string
			StringBuilder checksumHex = new StringBuilder();
			for (byte b : checksumBytes) {
				checksumHex.append(String.format("%02x", b));
			}

			return checksumHex.toString();
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException("Error: MD5-Algorithm not available?",
					e);
		}
	}

	/**
	 * checks the transferred checksum against deception.
	 * 
	 * @param params
	 *            array of url parameters
	 * @param secretKey
	 *            the shared secret key
	 * @param checkSum
	 *            the transferred check sum
	 * @return
	 */
	public static boolean checkChecksum(final Map<String, String> params,
			final String secretKey, final String checkSum) {
		String expected = calculateChecksum(params, secretKey);
		return (checkSum != null && checkSum.equals(expected));
	}

	/**
	 * extends a base url by parameters and a checksum of the parameters
	 * 
	 * @param base
	 *            base url
	 * @param params
	 *            parameter map
	 * @param secretKey
	 *            secret shared key
	 * @return the extended url
	 */
	public static String constructParamURL(String base,
			final Map<String, String> params, final String secretKey) {
		try {
			StringBuilder parameterizedURL = new StringBuilder(base);
			for (Map.Entry<String, String> entry : params.entrySet()) {
				// Encode the parameter name and value to ensure they're
				// URL-safe
				String encodedParamName = URLEncoder.encode(entry.getKey(),
						"UTF-8");
				String encodedParamValue;
				encodedParamValue = URLEncoder.encode(entry.getValue(),
						"UTF-8");

				extendParam(parameterizedURL, encodedParamName,
						encodedParamValue);
			}

			String checkSum = calculateChecksum(params, secretKey);
			extendParam(parameterizedURL, "checksum", checkSum);

			return parameterizedURL.toString();
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("Error: UTF-8 encoding not supported",
					e);
		}
	}

	/**
	 * just a helper method to add a parameter
	 * 
	 * @param parameterizedURL
	 * @param encodedParamName
	 * @param encodedParamValue
	 */
	private static void extendParam(StringBuilder parameterizedURL,
			String encodedParamName, String encodedParamValue) {
		// Check if the URL already contains parameters
		if (parameterizedURL.indexOf("?") >= 0) {
			// If yes, append the new parameter with "&"
			parameterizedURL.append("&");
		} else {
			parameterizedURL.append("?");
		}
		parameterizedURL.append(encodedParamName);
		parameterizedURL.append("=");
		parameterizedURL.append(encodedParamValue);
	}

}