// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.data.CommandArgPrefix;
import org.refcodes.data.CommandArgPrefixes;
import org.refcodes.runtime.OperatingSystem;
import org.refcodes.struct.Relation;

/**
 * The {@link AbstractOption} is an abstract implementation of an {@link Option}
 * providing the boiler plate when implementing the {@link Option} interface.
 *
 * @param <T> the generic type
 */
public abstract class AbstractOption<T> extends AbstractOperand<T> implements Option<T> {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _shortOption;
	private String _longOption;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link Option} with the given arguments.
	 * 
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aAlias The option argument's name to be used when constructing the
	 *        syntax.
	 * @param aDescription A description without any line breaks.
	 */
	public AbstractOption( String aShortOption, String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		super( aType, aAlias, aDescription );
		if ( aShortOption != null && (!aShortOption.startsWith( CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() ) || aShortOption.startsWith( CommandArgPrefix.POSIX_LONG_OPTION.getPrefix() )) ) {
			throw new IllegalArgumentException( "Your short-option \"" + aShortOption + "\" must start exactly with \"" + CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() + "\"." );
		}
		if ( aLongOption != null && !aLongOption.startsWith( CommandArgPrefix.POSIX_LONG_OPTION.getPrefix() ) ) {
			throw new IllegalArgumentException( "Your long-option \"" + aLongOption + "\" must start exactly with \"" + CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() + "\"." );
		}
		_shortOption = aShortOption;
		_longOption = aLongOption;
	}

	/**
	 * Constructs a {@link Option} with the given arguments.
	 * 
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aDescription A description without any line breaks.
	 */
	protected AbstractOption( String aShortOption, String aLongOption, Class<T> aType, String aDescription ) {
		super( aType, null, aDescription );
		if ( aShortOption == null && aLongOption == null ) {
			throw new IllegalArgumentException( "You must provide at least a short-option or a long-option!" );
		}
		if ( !aShortOption.startsWith( CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() ) || aShortOption.startsWith( CommandArgPrefix.POSIX_LONG_OPTION.getPrefix() ) ) {
			throw new IllegalArgumentException( "Your short-option \"" + aShortOption + "\" must start exactly with \"" + CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() + "\"." );
		}
		if ( !aLongOption.startsWith( CommandArgPrefix.POSIX_LONG_OPTION.getPrefix() ) ) {
			throw new IllegalArgumentException( "Your long-option \"" + aLongOption + "\" must start exactly with \"" + CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() + "\"." );
		}
		_shortOption = aShortOption;
		_longOption = aLongOption;
	}

	/**
	 * Instantiates a new option.
	 *
	 * @param aProperty The key (= alias) and the value for the operand.
	 * @param aType the type
	 */
	public AbstractOption( Relation<String, T> aProperty, Class<T> aType ) {
		super( aProperty, aType );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<T>> parseArgs( String[] aArgs, String[] aOptions ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Relation<String, String> theOptionArgument = CliUtility.getOptionArgument( this, aArgs, aOptions );
		if ( theOptionArgument != null ) {
			List<Operand<T>> theList = new ArrayList<Operand<T>>();
			theList.add( this );
			setArgs( new String[] {
					theOptionArgument.getKey(), theOptionArgument.getValue()
			} );
			setValue( toType( theOptionArgument.getValue() ) );
			return theList;
		}
		if ( CliUtility.contains( aArgs, getShortOption() ) ) {
			throw new ParseArgsException( aArgs, "Missing value; the short-option \"" + getShortOption() + "\" requires a value." );
		}
		if ( CliUtility.contains( aArgs, getLongOption() ) ) {
			throw new ParseArgsException( aArgs, "Missing value; the long-option \"" + getLongOption() + "\" requires a value." );
		}
		throw new UnknownArgsException( aArgs, "Neither the short-option \"" + getShortOption() + "\" nor the long-option \"" + getLongOption() + "\"  was found in the command line arguments; at least one of them must be specified." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getShortOption() {
		return _shortOption;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLongOption() {
		return _longOption;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptionEscCode, String aResetEscCode ) {
		aOptionEscCode = aOptionEscCode == null ? "" : aOptionEscCode;
		aResetEscCode = aResetEscCode == null ? "" : aResetEscCode;
		return aOptionEscCode + (getShortOption() != null ? getShortOption() : getLongOption()) + (getAlias() != null ? (" " + CliUtility.toParameterSpec( this )) : "") + aResetEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		String theSwitch = null;
		if ( getShortOption() != null ) {
			theSwitch = getShortOption();
		}
		else if ( getLongOption() != null ) {
			theSwitch = getLongOption();
		}
		if ( theSwitch != null ) {
			StringBuilder theBuilder = new StringBuilder();
			theBuilder.append( theSwitch );
			theBuilder.append( ":=" );
			if ( getValue() != null ) {
				if ( getValue() instanceof String ) {
					theBuilder.append( '"' );
				}
				theBuilder.append( getValue().toString() );
				if ( getValue() instanceof String ) {
					theBuilder.append( '"' );
				}
			}
			else {
				theBuilder.append( "null" );
			}
			return theBuilder.toString();
		}
		return "null";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the prefixes suitable for the environment. Prefixes on Windows
	 * might differ from those on POSIX conforming systems.
	 * 
	 * @return The prefixes to used for this runtime.
	 */
	protected String[] getOptionPrefixes() {
		OperatingSystem theOperatingSystem = OperatingSystem.toOperatingSystem();
		if ( theOperatingSystem == OperatingSystem.MAC ) {
			return CommandArgPrefixes.POSIX.toPrefixes();
		}
		else if ( theOperatingSystem == OperatingSystem.UNIX ) {
			return CommandArgPrefixes.POSIX.toPrefixes();
		}
		else if ( theOperatingSystem == OperatingSystem.WINDOWS ) {
			return CommandArgPrefixes.WINDOWS.toPrefixes();
		}
		return CommandArgPrefix.toPrefixes();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void setValue( T aValue ) {
		super.setValue( aValue );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void setArgs( String[] aArgs ) {
		super.setArgs( aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
