// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.List;

/**
 * This mixin provides builder additions (as of the builder pattern for chained
 * configuring method calls) for parsing command line arguments.
 */
public interface ArgsParserMixin<B extends ArgsParserMixin<B>> {

	/**
	 * Evaluates the provided command line arguments.
	 * 
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return This instance as of the builder pattern to chain further method
	 *         calls.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	B withEvalArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException;

	/**
	 * Evaluates the provided command line arguments .
	 * 
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return This instance as of the builder pattern to chain further method
	 *         calls.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	default B withEvalArgs( List<String> aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		return withEvalArgs( aArgs.toArray( new String[aArgs.size()] ) );
	}

}
