// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.refcodes.data.CommandArgPrefix;
import org.refcodes.struct.Relation;
import org.refcodes.struct.RelationImpl;

/**
 * This utility class provides method useful for the refcodes-cli artifact and
 * whose implementation has been motivated by the implementation of the
 * refcodes-cli artifact.
 */
public final class CliUtility {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new console utility.
	 */
	private CliUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines whether a given {@link String} is contained in the given
	 * {@link String} array.
	 * 
	 * @param aArgs The {@link String} array to be tested whether it contains
	 *        the given {@link String}.
	 * @param aArg The {@link String} to be tested whether it is contained in
	 *        the provided {@link String} array.
	 * 
	 * @return True in case the {@link String} is contained in the array, else
	 *         false.
	 */
	public static boolean contains( String[] aArgs, String aArg ) {
		List<String> theList = new ArrayList<String>( Arrays.asList( aArgs ) );
		return theList.contains( aArg );
	}

	/**
	 * Determines whether the provided {@link String} starts with one of the
	 * prefixes Identifying an {@link Option}. The prefixes are defined in the
	 * {@link CommandArgPrefix} enumeration and are usually the single
	 * hyphen-minus "-" and the double hyphen-minus "--". Usually option
	 * arguments are prefixed either with a
	 * 
	 * @param aArg The argument for which to determines whether it represents an
	 *        option argument or not.
	 * 
	 * @return True, in case the provided argument is an option argument.
	 */
	public static boolean isOptionArgument( String aArg ) {
		for ( int i = 0; i < CommandArgPrefix.toPrefixes().length; i++ ) {
			if ( aArg.startsWith( CommandArgPrefix.toPrefixes()[i] ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Takes the {@link Option}'s short-option and long-option and tries to
	 * determine that {@link Option}'s value in the provided command line
	 * arguments. Depending on whether the short-option or the long-option was
	 * detected with a value, the result contains the according option as the
	 * key with the detected value in the {@link Relation} instance. Null is
	 * returned when either no option was found or no value for one of the
	 * options.
	 * 
	 * @param aOption The option for which to get the value
	 * @param aArgs The command line arguments from which to determine the
	 *        {@link Option}'s value.
	 * @param aOptions The list of options (short and well as long) which are
	 *        reserved and cannot be used as value.
	 * 
	 * @return A key/value-pair containing the detected (short / long) option
	 *         alongside the detected value.
	 */
	public static Relation<String, String> getOptionArgument( Option<?> aOption, String[] aArgs, String[] aOptions ) {
		if ( aArgs.length < 2 ) {
			return null;
		}
		Relation<String, String> theAttribute = getOptionArgument( aArgs, aOption.getShortOption(), aOptions );
		if ( theAttribute != null ) {
			return theAttribute;
		}
		theAttribute = getOptionArgument( aArgs, aOption.getLongOption(), aOptions );
		if ( theAttribute != null ) {
			return theAttribute;
		}
		return null;
	}

	/**
	 * Takes all {@link Operand} instances found in the provided {@link List}s
	 * and adds all therein found argument arrays (as of
	 * {@link Operand#getArgs()}) to the result.
	 * 
	 * @param aOperands The lists containing the {@link Operand} instances whose
	 *        command line arguments are to be added to the result.
	 * 
	 * @return All the command line arguments detected in the provided
	 *         {@link Operand}s {@link List}s.
	 */
	@SafeVarargs
	public static String[] toArgs( List<? extends Operand<?>>... aOperands ) {
		if ( aOperands != null ) {
			List<String> theArgs = new ArrayList<String>();
			for ( List<? extends Operand<?>> eOperands : aOperands ) {
				if ( eOperands != null ) {
					for ( Operand<?> eOperand : eOperands ) {
						if ( eOperand.getArgs() != null ) {
							theArgs.addAll( Arrays.asList( eOperand.getArgs() ) );
						}
					}
				}
			}
			return theArgs.toArray( new String[theArgs.size()] );
		}
		return null;
	}

	/**
	 * Creates the difference between the provided set and the provided subset.
	 * 
	 * @param aSet The set to be used for the diff operation.
	 * @param aSubset The subset to be used for the diff operation.
	 * 
	 * @return The difference between the set and the subset.
	 */
	public static String[] toDiff( String[] aSet, String[] aSubset ) {
		List<String> theDiff = new ArrayList<String>( Arrays.asList( aSet ) );
		int index = 0;
		String eeElement;
		for ( String eElement : aSubset ) {
			out: {
				// Try to get subsequent elements |-->
				for ( int l = index; l < theDiff.size(); l++ ) {
					eeElement = theDiff.get( l );
					if ( eElement.equals( eeElement ) ) {
						theDiff.remove( l );
						index = l;
						break out;
					}
				}
				// Try to get subsequent elements <--|

				// No subsequent elelemt found, start from beginning |-->
				for ( int l = 0; l < theDiff.size(); l++ ) {
					eeElement = theDiff.get( l );
					if ( eElement.equals( eeElement ) ) {
						theDiff.remove( l );
						index = l;
						break out;
					}
				}
				// No subsequent elelemt found, start from beginning <--|
			}
		}
		return theDiff.toArray( new String[theDiff.size()] );
	}

	/**
	 * Creates the difference between the provided set and the provided
	 * {@link List}s therein found argument arrays subset (as of
	 * {@link Operand#getArgs()}).
	 * 
	 * @param aSet The set to be used for the diff operation.
	 * @param aSubset The subset to be used for the diff operation being the
	 *        lists containing the {@link Operand} instances whose command line
	 *        arguments are to be diffed.
	 * 
	 * @return The difference between the set and the subset.
	 */
	public static String[] toDiff( String[] aSet, List<? extends Operand<?>> aSubset ) {
		String[] theSubset = toArgs( aSubset );
		return toDiff( aSet, theSubset );
	}

	/**
	 * Creates the parameter specification from the provided {@link Operand}. In
	 * case we got a {@link Flag}, then there will be an empty parameter
	 * specification as the {@link Flag} implies a parameter. If the
	 * {@link Operand} is an option, then the parameter will be prefixed and
	 * suffixed different to an {@link Operand} type.
	 * 
	 * @param aOperand The operand from which to get the parameter
	 *        specification.
	 * 
	 * @return The parameter specification.
	 */
	public static String toParameterSpec( Operand<?> aOperand ) {
		if ( aOperand instanceof ArrayOperand<?> ) {
			ArrayOperand<?> theArrayOperand = (ArrayOperand<?>) aOperand;
			return toParameterSpec( aOperand, theArrayOperand.getMinLength(), theArrayOperand.getMaxLength() );
		}
		return toParameterSpec( aOperand, -1, -1 );
	}

	/**
	 * Creates the parameter specification from the provided {@link Operand}. In
	 * case we got a {@link Flag}, then there will be an empty parameter
	 * specification as the {@link Flag} implies a parameter. If the
	 * {@link Operand} is an option, then the parameter will be prefixed and
	 * suffixed different to an {@link Operand} type.
	 * 
	 * @param aOperand The operand from which to get the parameter
	 *        specification.
	 * @param aMin The minimum value for the operand.
	 * @param aMax The maximum value for the operand.
	 * 
	 * @return The parameter specification.
	 */
	public static String toParameterSpec( Operand<?> aOperand, int aMin, int aMax ) {
		if ( aOperand.getAlias() == null ) {
			return "";
		}
		StringBuilder theBuilder = new StringBuilder();
		if ( !(aOperand instanceof Flag) ) {
			if ( aOperand.getAlias() != null ) {
				if ( aOperand instanceof Option<?> ) {
					theBuilder.append( '<' );
				}
				theBuilder.append( aOperand.getAlias() );
				if ( aOperand.getType().isArray() ) {
					theBuilder.append( "[" );
				}
				else if ( aMin != -1 || aMax != -1 ) {
					theBuilder.append( "{" );
				}

				if ( aMin != -1 && aMin == aMax ) {
					theBuilder.append( aMin );
				}
				else {
					if ( aMin != -1 ) {
						theBuilder.append( aMin );
					}
					if ( aOperand.getType().isArray() || aMin != -1 || aMax != -1 ) {
						theBuilder.append( "..." );
					}
					if ( aMax != -1 ) {
						theBuilder.append( aMax );
					}
				}
				if ( aOperand.getType().isArray() ) {
					theBuilder.append( "]" );
				}
				else if ( aMin != -1 || aMax != -1 ) {
					theBuilder.append( "}" );
				}
				if ( aOperand instanceof Option<?> ) {
					theBuilder.append( '>' );
				}
			}
		}
		return theBuilder.toString();
	}

	/**
	 * Creates the options specification containing the short option (if any)
	 * and the long option ( if any) from the provided {@link Operand};.
	 *
	 * @param aOperand The operand from which to create the options
	 *        specification.
	 * 
	 * @return The options specification {@link String}.
	 */
	public static String toOptionsSpec( Operand<?> aOperand ) {
		Option<?> eOption;
		StringBuilder theBuilder = new StringBuilder();
		if ( aOperand instanceof Option ) {
			eOption = (Option<?>) aOperand;
			if ( eOption.getShortOption() != null ) {
				theBuilder.append( eOption.getShortOption() );
			}
			if ( eOption.getLongOption() != null ) {
				if ( theBuilder.length() > 0 ) {
					theBuilder.append( ' ' );
				}
				theBuilder.append( eOption.getLongOption() );
			}
		}
		return theBuilder.toString();
	}

	/**
	 * Creates a specification for the given {@link Operand} consisting of the
	 * options specification (if any) as of {@link #toOptionsSpec(Operand)} and
	 * the parameter specification (if any) as of
	 * {@link #toParameterSpec(Operand)}.
	 * 
	 * @param aOperand The {@link Operand} from which to create the
	 *        specification.
	 * 
	 * @return The specification.
	 */
	public static String toSpec( Operand<?> aOperand ) {
		StringBuilder theBuilder = new StringBuilder();
		theBuilder.append( toOptionsSpec( aOperand ) );
		String theParameterSpec = toParameterSpec( aOperand );
		if ( theParameterSpec != null && theParameterSpec.length() != 0 && theBuilder.length() > 0 ) {
			theBuilder.append( ' ' );
		}
		theBuilder.append( theParameterSpec );
		return theBuilder.toString();
	}

	/**
	 * Determines the options (short and long) found in the provided
	 * {@link Syntaxable} element and its children.
	 * 
	 * @param aSyntaxable The element with its children from which to determine
	 *        the options.
	 * 
	 * @return An array containing all found options.
	 */
	static String[] toOptions( Syntaxable aSyntaxable ) {
		Set<String> theOptions = new HashSet<>();
		String[] eOptions;
		if ( aSyntaxable instanceof ArgsSyntax ) {
			ArgsSyntax theArgsSyntax = (ArgsSyntax) aSyntaxable;
			for ( Operand<?> eOperand : theArgsSyntax.toOperands() ) {
				eOptions = toOptions( eOperand );
				if ( eOptions != null && eOptions.length != 0 ) {
					theOptions.addAll( Arrays.asList( eOptions ) );
				}
			}
		}
		if ( aSyntaxable instanceof Option<?> ) {
			Option<?> theOption = (Option<?>) aSyntaxable;
			if ( theOption.getShortOption() != null && theOption.getShortOption().length() != 0 ) {
				theOptions.add( theOption.getShortOption() );
			}
			if ( theOption.getLongOption() != null && theOption.getLongOption().length() != 0 ) {
				theOptions.add( theOption.getLongOption() );
			}
		}
		return theOptions.toArray( new String[theOptions.size()] );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the option argument.
	 *
	 * @param args the args
	 * @param aOption the e option
	 * @param aOptions The list of options (short and well as long) which are
	 *        reserved and cannot be used as value.
	 * 
	 * @return the option argument
	 */
	private static Relation<String, String> getOptionArgument( String[] args, String aOption, String[] aOptions ) {
		String eArg;
		String eOptArg;
		for ( int i = 0; i < args.length - 1; i++ ) {
			eArg = args[i];
			if ( eArg.equals( aOption ) ) {
				eOptArg = args[i + 1];
				for ( String eOption : aOptions ) {
					if ( eOptArg.equalsIgnoreCase( eOption ) ) {
						return null;
					}
				}
				return new RelationImpl<String, String>( aOption, eOptArg );
			}
		}
		return null;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
