// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import org.refcodes.mixin.AliasAccessor;
import org.refcodes.mixin.Clonable;
import org.refcodes.mixin.DescriptionAccessor;
import org.refcodes.mixin.TypeAccessor;
import org.refcodes.mixin.ValueAccessor;

/**
 * An {@link Operand} represents a value parsed from command line arguments. In
 * comparison to an {@link Option} an {@link Operand} just represents a plain
 * value without an identifier in front of it. An {@link Operand} has a state
 * which changes with each invocation of the
 * {@link #parseArgs(String[], String[])} method.
 * <p>
 * It is recommended to put your {@link Operand} instance(s) at the end of your
 * top {@link ArgsSyntax} to enforce it to be the last {@link Syntaxable}(s) when
 * parsing the command line arguments - this makes sure that any {@link Option}s
 * pick their option arguments so that the {@link Operand}(s) will correctly be
 * left over for parsing command line argument(s); the {@link Operand} will not
 * pick by mistake an option argument.
 *
 * @param <T> the generic type
 */

public interface Operand<T> extends ValueAccessor<T>, DescriptionAccessor, TypeAccessor<T>, Syntaxable, ArgsAccessor, Comparable<Operand<?>>, AliasAccessor, Cloneable, Clonable {
	/**
	 * {@inheritDoc}
	 */
	@Override
	T getValue();

	/**
	 * When being parsed via the {@link #parseArgs(String[], String[])} method,
	 * use this method to determine whether the {@link #getValue()} method
	 * contains parsed value.
	 * 
	 * @return The true in case there a value has been parsed via the
	 *         {@link #parseArgs(String[], String[])} method.
	 */
	default boolean hasValue() {
		return getValue() != null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	String[] getArgs();

	/**
	 * Retrieves the name (alias) of the parameter value which can be the name
	 * of the operand ({@link Operand}) or the option argument ({@link Option}),
	 * depending on the sub-type inheriting from this interface. The parameter
	 * name is merely used for constructing the command line arguments syntax
	 * {@link String} via {@link #toSyntax(SyntaxNotation, String, String)} and
	 * the command line arguments detail description when creating a command
	 * line tool's help output. Attention: The alias can also be used as a key
	 * when putting an {@link Operand}'s key/value-pair into a dictionary such
	 * as is done by the <code>RuntimeProperties</code> type provided by the
	 * <code>refcodes-properties-ext-runime</code> artifact!
	 * 
	 * @return The name of the parameter, e.g the name of the operand or the
	 *         name of the option argument.
	 */
	@Override
	String getAlias();

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	default <V> V toValue( String aAlias ) {
		if ( aAlias.equals( getAlias() ) && hasValue() ) {
			return (V) getValue();
		}
		return null;
	}
}
