// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

/**
 * Provides an accessor for a root condition property.
 */
public interface RootConditionAccessor {

	/**
	 * Retrieves the root condition from the root condition property.
	 * 
	 * @return The root condition stored by the root condition property.
	 */
	ArgsSyntax getRootArgsSyntax();

	/**
	 * Provides a mutator for a root condition property.
	 */
	public interface RootConditionMutator {

		/**
		 * Sets the root condition for the root condition property.
		 * 
		 * @param aRootArgsSyntax The root condition to be stored by the root
		 *        condition property.
		 */
		void setRootArgsSyntax( ArgsSyntax aRootArgsSyntax );

		/**
		 * Sets the root option for the root condition property. Convenience
		 * method wrapping an {@link Option} into a
		 * {@link OptionCondition} and passing it to the
		 * {@link #setRootArgsSyntax(ArgsSyntax)} method.
		 * 
		 * @param aRootOption The root option to be stored by the root condition
		 *        property.
		 */
		default void setRootOption( Option<?> aRootOption ) {
			setRootArgsSyntax( new OptionCondition( aRootOption ) );
		}
	}

	/**
	 * Provides a builder method for a root condition property returning the
	 * builder for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface RootConditionBuilder<B extends RootConditionBuilder<B>> {

		/**
		 * Sets the root condition for the root condition property.
		 * 
		 * @param aRootArgsSyntax The root condition to be stored by the root
		 *        condition property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withRootArgsSyntax( ArgsSyntax aRootArgsSyntax );

		/**
		 * Sets the root option for the root condition property. Convenience
		 * method wrapping an {@link Option} into a
		 * {@link OptionCondition} and passing it to the
		 * {@link #withRootArgsSyntax(ArgsSyntax)} method.
		 * 
		 * @param aRootOption The root condition to be stored by the root
		 *        condition property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		default B withRootOption( Option<?> aRootOption ) {
			return withRootArgsSyntax( new OptionCondition( aRootOption ) );
		}
	}

	/**
	 * Provides a root condition property.
	 */
	public interface RootConditionProperty extends RootConditionAccessor, RootConditionMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link ArgsSyntax}
		 * (setter) as of {@link #setRootArgsSyntax(ArgsSyntax)} and returns the
		 * very same value (getter).
		 * 
		 * @param aRootArgsSyntax The {@link ArgsSyntax} to set (via
		 *        {@link #setRootArgsSyntax(ArgsSyntax)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default ArgsSyntax letRootArgsSyntax( ArgsSyntax aRootArgsSyntax ) {
			setRootArgsSyntax( aRootArgsSyntax );
			return aRootArgsSyntax;
		}
	}
}
