// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

/**
 * <h2>Getting started:</h2> Consider you have a tool called "foo-bar" to be
 * invoked with the below allowed argument combinations (syntax):
 * <p>
 * <code>foo-bar [{ -a | -d }] -f &lt;file&gt;</code>
 * <p>
 * "foo-bar" can be invoked either with an optional "-a" or with an optional
 * "-d" switch, but not both of them at the same time, and a file "-f
 * &lt;file&gt;" must be provided, else the passed arguments are rejected as not
 * being valid.
 * <p>
 * Valid arguments would be:
 * <ul>
 * <li>foo-bar -f someFile
 * <li>foo-bar -d -f anyFile
 * <li>foo-bar -f otherFile -a
 * <li>foo-bar -a -f otherFile
 * </ul>
 * Invalid arguments would be:
 * <ul>
 * <li>foo-bar -f someFile -b
 * <li>foo-bar -a someFile -f
 * <li>foo-bar -a -d -f anyFile
 * <li>foo-bar -a -x -f otherFile
 * </ul>
 * This means that additional switches not supported are not valid. The parser
 * detects such situations and you can print out a help message in such cases.
 * <h2>Construct your parser:</h2> First build your syntax using
 * {@link org.refcodes.cli.Flag}es , {@link org.refcodes.cli.Option}s and
 * {@link org.refcodes.cli.ArgsSyntax}s. You actually build the syntax tree for
 * your command line tool's supported arguments:
 * <p>
 * <code>
 * Option&lt;String&gt; theFile = new StringOption( "-f", "--file", "file", "A file" );
 * Flag theAdd = new Flag( "-a", null, "Add the specified file" );
 * Flag theDelete = new Flag( "-d", null, "Delete the specified file" );
 * ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
 * Syntaxable theOptional = new AnyCondition( theXor );
 * ArgsSyntax theAnd = new AndCondition( theOptional, theFile );
 * 
 * ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
 * theArgsParser.printUsage();
 * // theArgsParser.printHelp();
 * </code>
 * <h2>Using syntactic sugar</h2> The
 * [`TinyRestfulServer`](https://bitbucket.org/refcodes/funcodes-tinyrestful/raw
 * /master/src/main/java/club/funcodes/tinyrestful/TinyRestfulServer.java) demo
 * application uses `syntactic sugar` for setting up the command line arguments
 * parser:
 * <p>
 * <code>
 * import static org.refcodes.cli.CliSugar.*;
 * ...
 * 	public static void main( String args[] ) {
 * 
 * 		Option&lt;Integer&gt; theWidth = intOption( "-w", "--width", "width", "Sets the console width" );
 * 		Option&lt;Integer&gt; thePort = intOption( "-p", "--port", "port", "Sets the port for the server" );
 * 		Option&lt;Integer&gt; theMaxConns = intOption( "-c", "--connections", "connections", "Sets the number of max. connections" );
 * 		Option&lt;String&gt; theUsername = stringOption( "-u", "--user", "username", "The username for HTTP Basic-Authentication" );
 * 		Option&lt;String&gt; theSecret = stringOption( "-s", "--secret", "secret", "The password for HTTP Basic-Authentication" );
 * 
 * 		Flag theHelp = helpSwitch( "Shows this help" );
 * 		ArgsSyntax theRootCondition = xor( 
 * 			and( 
 * 				thePort, optional( theMaxConns ), optional( and( theUsername, theSecret ) ), optional( theWidth )
 * 			),
 * 			theHelp
 * 		);
 * 		ArgsParser theArgsParser = new ArgsParserImpl( theRootCondition );
 * 		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
 * 		theArgsParser.withName( "TinyRestful" ).withTitle( "TINYRESTFUL" ).withCopyrightNote( "Copyright (c) by FUNCODES.CLUB, Munich, Germany." ).withLicenseNote( "Licensed under GNU General Public License, v3.0 and Apache License, v2.0" );
 * 		theArgsParser.withBannerFont( new FontImpl( FontFamily.DIALOG, FontStyle.BOLD, 14 ) ).withBannerFontPalette( AsciiColorPalette.MAX_LEVEL_GRAY.getPalette() );
 * 		theArgsParser.setDescription( "Tiny evil RESTful server. TinyRestfulServer makes heavy use of the REFCODES.ORG artifacts found together with the FUNCODES.CLUB sources at http://bitbucket.org/refcodes." );
 * 		List&lt;? extends Operand&lt;?&gt;&gt; theResult = theArgsParser.evalArgs( args );
 * 		...
 * 	}
 * ...
 * </code>
 * <p>
 * Most obvious is the missing <code>new</code> statement for instantiating the
 * parts of your command line parser as this is done by the statically imported
 * methods. ### Under the hood As seen above, you pass your root
 * {@link org.refcodes.cli.ArgsSyntax} to the {@link org.refcodes.cli.ArgsParser}
 * which then already can print out your command line tool's usage string:
 * <h2>Test (invoke) your parser:</h2> In real live you would pass your
 * main-method's args[] array to the parser. Now just for a test-run, pass a
 * {@link java.lang.String} array to your parser and let it parse it:
 * <p>
 * <code>
 * String[] args = new String[] {
 *	"-f", "someFile", "-d"
 * };
 * List&lt;? extends Operand&lt;?&gt;&gt; theResult = theArgsParser.evalArgs( args );
 * File theConfigFile = new File( theFile.getValue() );
 * </code>
 * <p>
 * Now the leaves of your syntax tree are filled with the argument's values
 * according to the syntax you have been setting up: Your
 * {@link org.refcodes.cli.StringOption} instance aliased "theFile" now contains
 * the value "someFile".
 * <p>
 * The "theResult" contains the parsed arguments for you to use in your business
 * logic.
 * <p>
 * In case of argument rejection, a sub-type of the
 * {@link org.refcodes.cli.ArgsMismatchException} is being thrown; actually one
 * of the exceptions {@link org.refcodes.cli.UnknownArgsException},
 * {@link org.refcodes.cli.AmbiguousArgsException},
 * {@link org.refcodes.cli.SuperfluousArgsException} or
 * {@link org.refcodes.cli.ParseArgsException} is being thrown, according to the
 * cause of the rejection. So you can either catch the
 * {@link org.refcodes.cli.ArgsMismatchException} or, if you need more details
 * on the cause, the other sub-exceptions.
 * <h2>{@link org.refcodes.cli.Syntaxable}:</h2> {@link Syntaxable} defines the
 * methods at least required when building a command line arguments syntax tree
 * for traversing the syntax tree; either for parsing command line arguments or
 * for constructing the command line arguments syntax.
 * <p>
 * By providing various implementations of the {@link Syntaxable}'s subclasses
 * such as {@link Operand}, {@link Option} or {@link ArgsSyntax}, a command line
 * arguments syntax tree can be constructed. This syntax tree can be use to
 * create a human readable (verbose) command line arguments syntax and to parse
 * an array of command line arguments for determining the {@link Operand}s', the
 * {@link Flag}es' or the {@link Option}s' values.
 * <h2>{@link org.refcodes.cli.Operand}:</h2> An {@link Operand} represents a
 * value parsed from command line arguments. An {@link Operand} has a state
 * which changes with each invocation of the
 * {@link ArgsParser#evalArgs(String[])} method.
 * <p>
 * It is recommended to put your {@link Operand} instance(s) at the end of your
 * top {@link ArgsSyntax} to enforce it to be the last {@link Syntaxable}(s) when
 * parsing the command line arguments - this makes sure that any {@link Option}s
 * pick their option arguments so that the {@link Operand}(s) will correctly be
 * left over for parsing command line argument(s); the {@link Operand} will not
 * pick by mistake an {@link org.refcodes.cli.Option} argument.
 * <h2>{@link org.refcodes.cli.Option}:</h2> An {@link Option} represents a
 * command line option with the according option's value. An {@link Option} can
 * be seen as a key/value(s) pair defined in the command line arguments parsed
 * via the {@link ArgsParser#evalArgs(String[])} method.
 * <p>
 * An {@link Option} has a state which changes with each invocation of the
 * {@link ArgsParser#evalArgs(String[])} method.
 * <h2>{@link org.refcodes.cli.Flag}:</h2> A {@link Flag} is an {@link Option}
 * with a {@link Boolean} state. Usually switches are just set or omitted in the
 * command line arguments with no value provided; former representing a
 * <code>true</code> status and latter representing a <code>false</code> status.
 * <h2>{@link org.refcodes.cli.Operation}:</h2> The {@link Operation} is an
 * argument representing a function or a method and is either provided or not
 * provided as of {@link Operation#isEnabled()}. It must neither be prefixed
 * with "-" nor with "--" in contrast to the {@link Option} or the {@link Flag}
 * type.
 * <h2>{@link org.refcodes.cli.ArgsSyntax}:</h2> The {@link ArgsSyntax} interface
 * represents a node in the command line arguments syntax tree; simply extending
 * the {@link Syntaxable} interface and adding the functionality of providing
 * access to the added {@link Operand}s (leafs). In future extensions, a
 * {@link ArgsSyntax} might provide access to the child {@link Syntaxable}
 * elements contained in a {@link ArgsSyntax} instance. As of the current
 * findings, access to the children of the {@link ArgsSyntax} node is not
 * required and would make the interface unnecessarily complicated.
 * <p>
 * See also:
 * <ul>
 * <li>{@link org.refcodes.cli.StringOption}</li>
 * <li>{@link org.refcodes.cli.Flag}</li>
 * <li>{@link org.refcodes.cli.StringOperand}</li>
 * <li>{@link org.refcodes.cli.XorCondition}</li>
 * <li>{@link org.refcodes.cli.AndCondition}</li>
 * <li>{@link org.refcodes.cli.AnyCondition}</li>
 * </ul>
 */
package org.refcodes.cli;
