// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.cli.CliSugar.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.jupiter.api.Test;
import org.refcodes.data.AsciiColorPalette;
import org.refcodes.runtime.RuntimeUtility;
import org.refcodes.struct.Property;
import org.refcodes.struct.PropertyImpl;
import org.refcodes.textual.FontFamily;
import org.refcodes.textual.FontImpl;
import org.refcodes.textual.FontStyle;
import org.refcodes.textual.VerboseTextBuilder;

public class ArgsParserTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TESTS = Boolean.getBoolean( "log.tests" );

	private static String[][] ARGS = new String[][] {
			{
					"--name", "FILE", "--boolean", "--active"
			}, {
					"--name", "FILE", "--boolean", "--alias", "ALIAS"
			}, {
					"--name", "FILE", "--boolean", "--alias", "ALIAS", "--active"
			}, {
					"--name", "FILE", "--boolean", "--alias", "ALIAS", "NULL"
			}, {
					"NULL", "--name", "FILE", "--boolean", "--alias", "ALIAS"
			}, {
					"NULL", "--name", "FILE", "--boolean", "--alias", "ALIAS", "NIL"
			}, {
					"NULL", "--name", "NAME0", "--name", "NAME1", "--boolean", "--alias", "ALIAS", "NIL"
			}, {
					"NULL", "/name", "NAME0", "/name", "NAME1", "/boolean", "/alias", "ALIAS", "NIL"
			}, {
					"NULL", "-name", "NAME0", "-name", "NAME1", "-boolean", "-alias", "ALIAS", "NIL"
			}, {
					"NULL", "/name", "NAME0", "-name", "NAME1", "--boolean", "/alias", "ALIAS", "NIL"
			}
	};

	private static String[][] PROPERTIES = new String[][] {
			{
					"name=FILE", "boolean=true", "active=true"
			}, {
					"name=FILE", "boolean=true", "alias=ALIAS"
			}, {
					"name=FILE", "boolean=true", "alias=ALIAS", "active=true"
			}, {
					"name=FILE", "boolean=true", "alias=ALIAS", "null=NULL"
			}, {
					"null=NULL", "name=FILE", "boolean=true", "alias=ALIAS"
			}, {
					"name=FILE", "boolean=true", "alias=ALIAS", "null#0=NULL", "null#1=NIL"
			}, {
					"name#0=NAME0", "name#1=NAME1", "boolean=true", "alias=ALIAS", "null#0=NULL", "null#1=NIL"
			}, {
					"name#0=NAME0", "name#1=NAME1", "boolean=true", "alias=ALIAS", "null#0=NULL", "null#1=NIL"
			}, {
					"name#0=NAME0", "name#1=NAME1", "boolean=true", "alias=ALIAS", "null#0=NULL", "null#1=NIL"
			}, {
					"name#0=NAME0", "name#1=NAME1", "boolean=true", "alias=ALIAS", "null#0=NULL", "null#1=NIL"
			}
	};

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tests parsing a simple command line arguments syntax.
	 */
	@Test
	public void testParser1() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFile = new StringOption( "-f", "--file", "file", "The file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFile );
		String[] args = new String[] {
				"-f", "someFile", "-a"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		@SuppressWarnings("unused")
		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
		assertTrue( theAdd.getValue() );
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a simple command line arguments syntax with an optional
	 * xor-statement.
	 */
	@Test
	public void testParser2() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFile = new StringOption( "-f", "--file", "file", "The file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		Syntaxable theOptional = new AnyCondition( theXor );
		ArgsSyntax theAnd = new AndCondition( theOptional, theFile );
		String[] args = new String[] {
				"-f", "someFile", "-d"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		if ( IS_LOG_TESTS ) System.out.println( theAnd.toUsage() );
		@SuppressWarnings("unused")
		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
		assertFalse( theAdd.getValue() );
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a simple command line arguments syntax with an operand.
	 */
	@Test
	public void testParser3() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFile = new StringOption( "-f", "--file", "file", "The file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFile, theOperand );
		String[] args = new String[] {
				"-f", "someFile", "-a", "anOperand"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		@SuppressWarnings("unused")
		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
		assertTrue( theAdd.getValue() );
		assertEquals( "anOperand", theOperand.getValue() );
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a simple command line arguments syntax with a missing
	 * operand.
	 */
	@Test
	public void testParser4() throws AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFile = new StringOption( "-f", "--file", "file", "The file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFile, theOperand );
		String[] args = new String[] {
				"-f", "someFile", "-a"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		try {
			theArgsParser.evalArgs( args );
			fail( "Expecting an exception" );
		}
		catch ( UnknownArgsException e ) {}
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a simple command line arguments syntax with a superfluous
	 * operand.
	 */
	@Test
	public void testParser5() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException {
		Option<String> theFile = new StringOption( "-f", "--file", "file", "The file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFile, theOperand );
		String[] args = new String[] {
				"-f", "someFile", "-a", "anOperand", "isSuperfluous"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		try {
			theArgsParser.evalArgs( args );
			fail( "Expecting an exception" );
		}
		catch ( SuperfluousArgsException e ) {
			assertEquals( 1, e.getArgs().length );
			assertEquals( "isSuperfluous", e.getArgs()[0] );
		}
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a more complex command line arguments syntax.
	 */
	@Test
	public void testParser6() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFromFile = new StringOption( "-f", "--from", "from_file", "The source file to be processed" );
		Option<String> theToFile = new StringOption( "-t", "--to", "to_file", "The destination file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFromFile, theToFile, theOperand );
		String[] args = new String[] {
				"-f", "fromFile", "-t", "toFile", "-a", "anOperand"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		@SuppressWarnings("unused")
		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
		assertTrue( theAdd.getValue() );
		assertEquals( "anOperand", theOperand.getValue() );
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a more complex command line arguments syntax with mixed up
	 * arguments order.
	 */
	@Test
	public void testParser7() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFromFile = new StringOption( "-f", "--from", "from_file", "The source file to be processed" );
		Option<String> theToFile = new StringOption( "-t", "--to", "to_file", "The destination file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed" );
		ArgsSyntax theXor = new XorCondition( theAdd, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFromFile, theToFile, theOperand );
		String[] args = new String[] {
				"-t", "toFile", "anOperand", "-f", "fromFile", "-a",
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		@SuppressWarnings("unused")
		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
		assertTrue( theAdd.getValue() );
		assertEquals( "anOperand", theOperand.getValue() );
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests parsing a more complex command line arguments syntax with mixed up
	 * arguments order printing the identified operands.
	 */
	@Test
	public void testParser8() {
		Option<String> theFromFile = new StringOption( "-f", "--from", "from_file", "The source file to be processed" );
		Option<String> theToFile = new StringOption( "-t", "--to", "to_file", "The destination file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed" );
		ArgsSyntax theXor = new XorCondition( theDelete, theAdd );
		ArgsSyntax theAnd = new AndCondition( theOperand, theXor, theFromFile, theToFile );
		List<? extends Operand<?>> theList = theAnd.toOperands();
		Collections.sort( theList );
		doLogOperands( theList );
	}

	/**
	 * Tests parsing an even more complex command line arguments syntax.
	 */
	@Test
	public void testParser9() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFromFile = new StringOption( "-f", "--from", "from_file", "The source file to be processed, please provide a relative path as absolute paths are not permitted because absolute pahts may prevent path jail brakeing." );
		Option<String> theToFile = new StringOption( "-t", "--to", "to_file", "The destination file to be processed" );
		Flag theAdd = new Flag( "-a", null, null, "Add the specified file" );
		Flag theCreate = new Flag( "-c", null, null, "Creates the specified file" );
		ArgsSyntax theOrArgsSyntax = new OrCondition( theAdd, theCreate );
		Flag theDelete = new Flag( "-d", null, null, "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed, it can be a complex artihmetic expression or a plain placeholder representing an environment variable." );
		ArgsSyntax theXor = new XorCondition( theOrArgsSyntax, theDelete );
		ArgsSyntax theAnd = new AndCondition( theXor, theFromFile, theToFile, theOperand );
		String[] args = new String[] {
				"-f", "fromFile", "-t", "toFile", "-a", "anOperand"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
		@SuppressWarnings("unused")
		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
		assertTrue( theAdd.getValue() );
		assertEquals( "anOperand", theOperand.getValue() );
		doLogArgs( theArgsParser, args );
	}

	/**
	 * Tests looking up values by their alias.
	 */
	@Test
	public void testAliasSyntaxTreeLookup() throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		Option<String> theFromFile = new StringOption( "-f", "--from", "from", "The source file to be processed, please provide a relative path as absolute paths are not permitted because absolute pahts may prevent path jail brakeing." );
		Option<String> theToFile = new StringOption( "-t", "--to", "to", "The destination file to be processed" );
		Flag theAddFile = new Flag( "-a", "--add", "add", "Add the specified file" );
		Flag theCreateFile = new Flag( "-c", "--create", "create", "Creates the specified file" );
		ArgsSyntax theOrArgsSyntax = new OrCondition( theAddFile, theCreateFile );
		Flag theDeleteFile = new Flag( "-d", "--delete", "delete", "Delete the specified file" );
		Operand<String> theOperand = new StringOperand( "operand", "The operand to be processed, it can be a complex artihmetic expression or a plain placeholder representing an environment variable." );
		ArgsSyntax theXor = new XorCondition( theOrArgsSyntax, theDeleteFile );
		ArgsSyntax theAnd = new AndCondition( theXor, theFromFile, theToFile, theOperand );
		String[] args = new String[] {
				"-f", "fromFile", "-t", "toFile", "-a", "anOperand"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theAnd );

		if ( IS_LOG_TESTS ) System.out.println( theAnd.toUsage() );

		theArgsParser.evalArgs( args );
		String theFrom = theAnd.toValue( "from" );
		String theTo = theAnd.toValue( "to" );
		boolean theAdd = theAnd.toValue( "add" );
		boolean theCreate = theAnd.toValue( "create" );
		boolean theDelete = theAnd.toValue( "delete" );

		if ( IS_LOG_TESTS ) System.out.println( "From := " + theFrom );
		if ( IS_LOG_TESTS ) System.out.println( "To := " + theTo );
		if ( IS_LOG_TESTS ) System.out.println( "Add := " + theAdd );
		if ( IS_LOG_TESTS ) System.out.println( "Create := " + theCreate );
		if ( IS_LOG_TESTS ) System.out.println( "Delete := " + theDelete );

		assertEquals( "fromFile", theFrom );
		assertEquals( "toFile", theTo );
		assertTrue( theAdd );
		assertFalse( theCreate );
		assertFalse( theDelete );
	}

	@Test
	public void testArgsProperties() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		for ( int i = 0; i < ARGS.length; i++ ) {
			testArgs( i );
		}
	}

	enum TrigonometricFunction {
		SINE, COSINE
	}

	@Test
	public void testProductionError() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {
				"-l", "1", "-s", "100", "-f", "1"
		};
		Option<Double> theFrequencyOpt = doubleOption( "-f", "--frequency", "FREQUENCY_HZ", "The frequency (Hz) to use when generating the values." );
		Option<Integer> theSamplingRateOpt = intOption( "-s", "--sampling-rate", "SAMPLING_RATE", "The sample rate (per second) for the generated values (defaults to DEFAULT_SAMPLING_RATE_PER_SEC samples/second)." );
		Option<Double> theLengthOpt = doubleOption( "-l", "--length", "LENGTH_SEC", "The length (in seconds) for the generated values (defaults to " + "DFAULT_LENGTH_SEC" + " seconds)." );

		// @formatter:off
		ArgsSyntax theRoot = and( theFrequencyOpt, any( theLengthOpt, theSamplingRateOpt ) );
		// @formatter:on

		ArgsParser theArgsParser = new ArgsParserImpl( theRoot );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.withName( "Waves" ).withTitle( "~waves~" ).withCopyrightNote( "Copyright (c) by FUNCODES.CLUB, Munich, Germany." ).withLicenseNote( "Licensed under GNU General Public License, v3.0 and Apache License, v2.0" );
		theArgsParser.withBannerFont( new FontImpl( FontFamily.DIALOG, FontStyle.BOLD ) ).withBannerFontPalette( AsciiColorPalette.MAX_LEVEL_GRAY.getPalette() );
		theArgsParser.withDescription( "Generate (sound) wave tables for given frequencies and amplitudes. Export them as CSV for further processing. Makes heavy use of the  REFCODES.ORG artifacts found together with the FUNCODES.CLUB sources at <http://bitbucket.org/refcodes>." );
		theArgsParser.evalArgs( args );
	}

	@Test
	public void testXorEdgeCase1() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		StringOption theTextArg = stringOption( "-t", "--text", "TEXT_PROPERTY", "The text message which to process." );
		Flag theEncryptFlag = flag( "-e", "--encrypt", "ENCRYPT_PROPERTY", "Encrypts the given message." );
		ArgsSyntax theArgsSyntax = xor( and( theEncryptFlag, theTextArg ), new AllCondition( theEncryptFlag ) );
		String[] args = {
				"-t", "text", "-e"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theArgsSyntax );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.evalArgs( args );
		assertEquals( "text", theTextArg.getValue() );
	}

	@Test
	public void testXorEdgeCase2() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		StringOption theTextArg = stringOption( "-t", "--text", "TEXT_PROPERTY", "The text message which to process." );
		Flag theEncryptFlag = flag( "-e", "--encrypt", "ENCRYPT_PROPERTY", "Encrypts the given message." );
		ArgsSyntax theArgsSyntax = cases( and( theEncryptFlag, theTextArg ), theEncryptFlag );
		String[] args = {
				"-t", "text", "-e"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theArgsSyntax );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.evalArgs( args );
		assertEquals( "text", theTextArg.getValue() );
	}

	@Test
	public void testArrayOption1() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {
				"-t", "text1", "-t", "text2", "-t", "text3"
		};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ), 1, 3 );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( theArray );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.evalArgs( args );
		if ( IS_LOG_TESTS ) {
			System.out.println( Arrays.toString( theArray.getValue() ) );
		}
		for ( int i = 0; i < theArray.getValue().length; i++ ) {
			assertEquals( args[i * 2 + 1], theArray.getValue()[i] );
		}
	}

	@Test
	public void testArrayOption2() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {
				"-t", "text1", "-t", "text2"
		};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ), 3 );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( theArray );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		try {
			theArgsParser.evalArgs( args );
			fail( "Expected an <UnknownArgsException>!" );
		}
		catch ( UnknownArgsException expected ) {
			// Expected!
		}
	}

	@Test
	public void testArrayOption3() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {
				"-t", "text1", "-t", "text2", "-t", "text3", "-t", "text4"
		};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ), 1, 3 );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( theArray );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		try {
			theArgsParser.evalArgs( args );
			fail( "Expected an <SuperfluousArgsException>!" );
		}
		catch ( SuperfluousArgsException expected ) {
			// Expected!
		}
	}

	@Test
	public void testArrayOption4() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ), 1, 3 );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( theArray );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		try {
			theArgsParser.evalArgs( args );
			fail( "Expected an <UnknownArgsException>!" );
		}
		catch ( UnknownArgsException expected ) {
			// Expected!
		}
	}

	@Test
	public void testArrayOption5() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ) );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( new AnyCondition( theArray ) );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.evalArgs( args );
		if ( IS_LOG_TESTS ) {
			System.out.println( Arrays.toString( theArray.getValue() ) );
		}
		assertFalse( theArray.hasValue() );
		assertNull( theArray.getValue() );
	}

	@Test
	public void testArrayOption6() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {
				"-t", "text1"
		};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ) );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( theArray );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.evalArgs( args );
		if ( IS_LOG_TESTS ) {
			System.out.println( Arrays.toString( theArray.getValue() ) );
		}
		for ( int i = 0; i < theArray.getValue().length; i++ ) {
			assertEquals( args[i * 2 + 1], theArray.getValue()[i] );
		}
	}

	@Test
	public void testArrayOption7() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		String[] args = {
				"-t", "text1", "-t", "text2", "-t", "text3"
		};
		ArrayOption<String> theArray = new ArrayOption<>( new StringOption( "-t", "--text", "text", "A text" ) );
		if ( IS_LOG_TESTS ) {
			System.out.println( theArray.toUsage() );
		}
		ArgsParser theArgsParser = new ArgsParserImpl( new AnyCondition( theArray ) );
		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
		theArgsParser.evalArgs( args );
		if ( IS_LOG_TESTS ) {
			System.out.println( Arrays.toString( theArray.getValue() ) );
		}
		for ( int i = 0; i < theArray.getValue().length; i++ ) {
			assertEquals( args[i * 2 + 1], theArray.getValue()[i] );
		}
		assertEquals( args.length / 2, theArray.getValue().length );
	}

	@Test
	public void testOperation1() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		Operation ls = new Operation( "ls", "List all entries" );
		Operation cd = new Operation( "cd", "Change direcory" );
		Flag verbose = new VerboseFlag();
		ArgsSyntax theRoot = new XorCondition( new AndCondition( ls, verbose ), cd );
		String[] theArgs = {
				"ls", "--verbose"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theRoot );
		theArgsParser.evalArgs( theArgs );
		assertTrue( ls.isEnabled() );
		assertTrue( verbose.isEnabled() );
		assertFalse( cd.isEnabled() );

		theRoot.reset();

		theArgs = new String[] {
				"cd"
		};
		theArgsParser.evalArgs( theArgs );
		assertFalse( ls.isEnabled() );
		assertFalse( verbose.isEnabled() );
		assertTrue( cd.isEnabled() );
	}

	@Test
	public void testNone1() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		Operation ls = new Operation( "ls", "List all entries" );
		Operation cd = new Operation( "cd", "Change direcory" );
		NoneOperand none = new NoneOperand( "none", "No arguments at all" );
		Flag verbose = new VerboseFlag();
		ArgsSyntax theRoot = new XorCondition( none, new AndCondition( ls, verbose ), cd );
		String[] theArgs = {
				"ls", "--verbose"
		};
		ArgsParser theArgsParser = new ArgsParserImpl( theRoot );
		theArgsParser.evalArgs( theArgs );
		assertTrue( ls.isEnabled() );
		assertTrue( verbose.isEnabled() );
		assertFalse( cd.isEnabled() );

		theRoot.reset();

		theArgs = new String[] {
				"cd"
		};
		theArgsParser.evalArgs( theArgs );
		assertFalse( ls.isEnabled() );
		assertFalse( verbose.isEnabled() );
		assertTrue( cd.isEnabled() );
		assertFalse( none.isEnabled() );
	}

	@Test
	public void testNone2() throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		Operation ls = new Operation( "ls", "List all entries" );
		Operation cd = new Operation( "cd", "Change direcory" );
		NoneOperand none = new NoneOperand( "none", "No arguments at all" );
		Flag verbose = new VerboseFlag();
		ArgsSyntax theRoot = new XorCondition( none, new AndCondition( ls, verbose ), cd );
		String[] theArgs = {};
		ArgsParser theArgsParser = new ArgsParserImpl( theRoot );
		theArgsParser.evalArgs( theArgs );
		assertFalse( ls.isEnabled() );
		assertFalse( verbose.isEnabled() );
		assertFalse( cd.isEnabled() );
		assertTrue( none.isEnabled() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void testArgs( int i ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		ArgsParser theArgsParser = new ArgsParserImpl();
		Map<String, String> theProperties = new HashMap<>();
		List<? extends Operand<?>> theOperands = theArgsParser.evalArgs( ARGS[i] );
		for ( Operand<?> eOperand : theOperands ) {
			theProperties.put( eOperand.getAlias(), "" + eOperand.getValue() );
		}
		if ( IS_LOG_TESTS ) System.out.println( "Line <" + i + ">:" );
		for ( String eKey : theProperties.keySet() ) {
			if ( IS_LOG_TESTS ) System.out.println( eKey + " := " + theProperties.get( eKey ) );
		}
		String[] theExcepcted = PROPERTIES[i];
		for ( int j = 0; j < theExcepcted.length; j++ ) {
			Property eProperty = new PropertyImpl( theExcepcted[j] );
			if ( eProperty.getKey().equals( "null" ) ) {
				eProperty = new PropertyImpl( null, eProperty.getValue() );
			}

			assertEquals( eProperty.getValue(), theProperties.get( eProperty.getKey() ), "Line <" + i + "> (" + eProperty.toString() + ")" );
		}
		if ( IS_LOG_TESTS ) System.out.println( "--------------------------------------------------------------------------------" );
	}

	/**
	 * Logs the provided arguments and the static hierarchy as well as inner
	 * state.
	 *
	 * @param aArgsParser the args parser
	 * @param aArgs The arguments being applied to that condition.
	 */
	private void doLogArgs( ArgsParser aArgsParser, String[] aArgs ) {
		ArgsSyntax theArgsSyntax = aArgsParser.getRootArgsSyntax();
		if ( IS_LOG_TESTS ) System.out.println( RuntimeUtility.getCallerStackTraceElement().getMethodName() );
		aArgsParser.withSyntaxNotation( SyntaxNotation.GNU_POSIX ).printHelp();
		if ( IS_LOG_TESTS ) System.out.println( "[" + SyntaxNotation.REFCODES + " SYNTAX NOTATION]  " + theArgsSyntax.toUsage( SyntaxNotation.REFCODES ) );
		if ( IS_LOG_TESTS ) System.out.println( "[" + SyntaxNotation.GNU_POSIX + " SYNTAX NOTATION] " + theArgsSyntax.toUsage( SyntaxNotation.GNU_POSIX ) );
		if ( IS_LOG_TESTS ) System.out.println( "[COMMAND LINE ARGS]         " + new VerboseTextBuilder().withElements( aArgs ).toString() );
		if ( IS_LOG_TESTS ) System.out.println( "[INTERNAL STATUS]           " + theArgsSyntax.toString() );
		if ( IS_LOG_TESTS ) System.out.println( "--------------------------------------------------------------------------------" );
	}

	/**
	 * Logs the provided arguments and the static hierarchy as well as inner
	 * state.
	 *
	 * @param aOperands the operands
	 */
	private void doLogOperands( List<? extends Operand<?>> aOperands ) {
		if ( IS_LOG_TESTS ) System.out.println( RuntimeUtility.getCallerStackTraceElement().getMethodName() );
		for ( Operand<?> eOperand : aOperands ) {
			if ( IS_LOG_TESTS ) System.out.println( "[OPERAND] " + eOperand.toSyntax( SyntaxNotation.REFCODES ) );
		}
		if ( IS_LOG_TESTS ) System.out.println( "--------------------------------------------------------------------------------" );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
