// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.List;

import org.refcodes.textual.VerboseTextBuilder;

/**
 * The {@link AllCondition} enforces that the encapsulated {@link Syntaxable}
 * consumes all arguments passed via invoking its
 * {@link Syntaxable#parseArgs(String[])}, else a
 * {@link SuperfluousArgsException} is thrown. This enables the construction of
 * a syntax which otherwise would not be unambiguous when encapsulated within an
 * {@link XorCondition}.
 */
public class AllCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String ALL_ARGS = "∀";
	private static final String LEFT_TAG = "( ";
	private static final String RIGHT_TAG = " )";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link AllCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instance to be nested.
	 *
	 * @param aArg The {@link Syntaxable} ({@link ArgsSyntax}) instance to be
	 *        nested
	 */
	public AllCondition( Syntaxable aArg ) {
		super( aArg );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		List<? extends Operand<?>> theResult = getFirst().parseArgs( aArgs, aOptions );
		String[] theRemainderArgs = CliUtility.toDiff( aArgs, theResult );
		if ( theRemainderArgs != null && theRemainderArgs.length != 0 ) {
			throw new SuperfluousArgsException( theRemainderArgs, "There were command line arguments (" + VerboseTextBuilder.asString( theRemainderArgs ) + ") " );
		}
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		if ( getChildren() != null && getChildren().size() > 0 ) {
			StringBuilder theBuilder = new StringBuilder();
			for ( Syntaxable eChild : getChildren() ) {
				if ( theBuilder.length() != 0 ) {
					theBuilder.append( " " );
				}
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( ALL_ARGS );
					theBuilder.append( LEFT_TAG );
				}
				theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( RIGHT_TAG );
				}
			}
			return theBuilder.toString();
		}
		return "";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		if ( getChildren() != null && getChildren().size() > 0 ) {
			if ( getChildren().size() == 1 ) {
				return getChildren().get( 0 ).toSynopsis( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode );
			}
			StringBuilder theBuilder = new StringBuilder();
			for ( Syntaxable eChild : getChildren() ) {
				if ( theBuilder.length() != 0 ) {
					theBuilder.append( " " );
				}
				theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
			}
			return theBuilder.toString();
		}
		return "";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( ALL_ARGS );
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( " " );
			}
			theBuilder.append( eChild.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		getChildren().get( 0 ).reset();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toState();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
