// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.io.PrintStream;
import java.util.Collection;
import java.util.List;

import org.refcodes.component.Resetable;
import org.refcodes.mixin.DescriptionAccessor.DescriptionBuilder;
import org.refcodes.mixin.DescriptionAccessor.DescriptionProperty;
import org.refcodes.mixin.EscapeCodesStatusAccessor.EscapeCodeStatusBuilder;
import org.refcodes.mixin.EscapeCodesStatusAccessor.EscapeCodeStatusProperty;
import org.refcodes.mixin.NameAccessor.NameBuilder;
import org.refcodes.mixin.NameAccessor.NameProperty;
import org.refcodes.mixin.ResetEscapeCodeAccessor.ResetEscapeCodeBuilder;
import org.refcodes.mixin.ResetEscapeCodeAccessor.ResetEscapeCodeProperty;
import org.refcodes.mixin.TitleAccessor.TitleBuilder;
import org.refcodes.mixin.TitleAccessor.TitleProperty;
import org.refcodes.textual.Font;
import org.refcodes.textual.TextBoxGrid;
import org.refcodes.textual.TextBoxGridAccessor.TextBoxGridBuilder;
import org.refcodes.textual.TextBoxGridAccessor.TextBoxGridProperty;

/**
 * The {@link ArgsParser} provides means for parsing command line arguments and
 * constructing a command line utility's help output.
 */
public interface ArgsParser extends ArgsParserMixin<ArgsParser>, TitleProperty, TitleBuilder<ArgsParser>, NameProperty, NameBuilder<ArgsParser>, DescriptionProperty, DescriptionBuilder<ArgsParser>, Resetable, RootConditionAccessor, EscapeCodeStatusProperty, EscapeCodeStatusBuilder<ArgsParser>, ResetEscapeCodeProperty, ResetEscapeCodeBuilder<ArgsParser>, TextBoxGridProperty, TextBoxGridBuilder<ArgsParser> {

	/**
	 * Prints the given line to standard error with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 * 
	 * @param aLine The line to be printed.
	 */
	void errorLn( String aLine );

	/**
	 * Same as {@link #evalArgs(String[])} with the difference that the elements
	 * representing the arguments are passed as a list instead of an array.
	 *
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return The list of evaluated command line arguments being instances of
	 *         the {@link Operand} interfaces or its sub-types.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	default List<? extends Operand<?>> evalArgs( List<String> aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		return evalArgs( aArgs.toArray( new String[aArgs.size()] ) );
	}

	/**
	 * Evaluates the provided command line arguments and determines the
	 * according values by evaluating the root {@link ArgsSyntax}. In case of
	 * parsing failure, an according exception is thrown. ATTENTION: This method
	 * tests(!) for superfluous command line arguments being passed; e.g.
	 * command line arguments not being evaluated by any of the
	 * {@link Syntaxable} instance being traversed starting at the root
	 * {@link ArgsSyntax} will be reported. Business logic therefore should
	 * invoke this root node's {@link #evalArgs(String[])} method instead of a
	 * {@link ArgsSyntax}'s {@link ArgsSyntax#parseArgs(String[], String[])}
	 * method; as ignoring superfluous command line arguments will cause
	 * unexpected behavior from the point of view of the invoker.
	 * 
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return The list of evaluated command line arguments being instances of
	 *         the {@link Operand} interfaces or its sub-types.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	List<? extends Operand<?>> evalArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException;

	/**
	 * Retrieves the copyright.
	 * 
	 * @return The copyright note.
	 */
	String getCopyrightNote();

	/**
	 * Retrieves the description.
	 * 
	 * @return The description.
	 */
	String getDescription();

	/**
	 * Retrieves the license.
	 * 
	 * @return The license note.
	 */
	String getLicenseNote();

	/**
	 * Retrieves the command name's (CLI executable) Escape-Code from the
	 * command name Escape-Code property.
	 * 
	 * @return The command name's Escape-Code stored by the command name
	 *         Escape-Code property.
	 */
	String getCommandEscapeCode();

	/**
	 * Retrieves the parameter's description Escape-Code from the parameter's
	 * description Escape-Code property.
	 * 
	 * @return The description Escape-Code stored by the description Escape-Code
	 *         property.
	 */
	String getDescriptionEscapeCode();

	/**
	 * Retrieves the parameter Escape-Code from the parameter Escape-Code
	 * property.
	 * 
	 * @return The parameter Escape-Code stored by the parameter Escape-Code
	 *         property.
	 */
	String getParamEscapeCode();

	/**
	 * Retrieves the option Escape-Code from the option Escape-Code property.
	 * 
	 * @return The option Escape-Code stored by the option Escape-Code property.
	 */
	String getOptionEscapeCode();

	/**
	 * Retrieves the banner's content Escape-Code from the banner Escape-Code
	 * property.
	 * 
	 * @return The banner Escape-Code stored by the banner Escape-Code property.
	 */
	String getBannerEscapeCode();

	/**
	 * Retrieves the line separator Escape-Code from the line separator
	 * Escape-Code property.
	 * 
	 * @return The line separator Escape-Code stored by the line separator
	 *         Escape-Code property.
	 */
	String getLineSeparatorEscapeCode();

	/**
	 * Gets the character to be used when printing a separator line with the
	 * {@link #printSeparatorLn()} method.
	 * 
	 * @return aSeparatorChar The character used by the
	 *         {@link #printSeparatorLn()} method when printing out the line of
	 *         characters..
	 */
	char getSeparatorLnChar();

	/**
	 * Retrieves the banner's border Escape-Code from the banner border
	 * Escape-Code property.
	 * 
	 * @return The banner border Escape-Code stored by the banner border
	 *         Escape-Code property.
	 */
	String getBannerBorderEscapeCode();

	/**
	 * The root condition is the starting point node of a {@link Syntaxable}
	 * hierarchy to be traversed when determining the syntax for command line
	 * arguments or when evaluating the command line arguments. {@inheritDoc}
	 * 
	 * @return The root {@link ArgsSyntax} in which's syntax this parser is
	 *         based.
	 */
	@Override
	ArgsSyntax getRootArgsSyntax();

	/**
	 * Retrieves the {@link Example} instances as addeed using
	 * {@link #addExample(String, Operand...)}.
	 * 
	 * @return The according {@link Example} elements.
	 */
	List<Example> getExamples();

	/**
	 * Adds a usage example by providing a description of the example and the
	 * command line arguments required by the example.
	 * 
	 * @param aDescription The description of the example.
	 * @param aOperands The command line arguments used by the example.
	 */
	default void addExample( String aDescription, Operand<?>... aOperands ) {
		addExample( new Example( aDescription, aOperands ) );
	}

	/**
	 * Adds a usage example by providing a description of the example and the
	 * command line arguments required by the example.
	 * 
	 * @param aExample The description as well as the command line arguments
	 *        used by the example.
	 */
	void addExample( Example aExample );

	/**
	 * Adds a usage example.
	 * 
	 * @param aExample The {@link Example} to add.
	 * 
	 * @return This instance as of the builder pattern.
	 */
	default ArgsParser withExample( Example aExample ) {
		addExample( aExample );
		return this;
	}

	/**
	 * Adds a usage example by providing a description of the example and the
	 * command line arguments required by the example.
	 * 
	 * @param aDescription The description of the example.
	 * @param aOperands The command line arguments used by the example.
	 * 
	 * @return This instance as of the builder pattern.
	 */
	default ArgsParser withExample( String aDescription, Operand<?>... aOperands ) {
		addExample( aDescription, aOperands );
		return this;
	}

	/**
	 * Adds all the bunch of {@link Example} elements provided.
	 * 
	 * @param aExamples The example usages to be added.
	 * 
	 * @return This instance as of the builder pattern.
	 */
	default ArgsParser withExamples( Collection<Example> aExamples ) {
		for ( Example eExample : aExamples ) {
			addExample( eExample );
		}
		return this;
	}

	/**
	 * Adds all the bunch of {@link Example} elements provided.
	 * 
	 * @param aExamples The example usages to be added.
	 * 
	 * @return This instance as of the builder pattern.
	 */
	default ArgsParser withExamples( Example[] aExamples ) {
		for ( Example eExample : aExamples ) {
			addExample( eExample );
		}
		return this;
	}

	/**
	 * Prints the banner; the banner most probably is an ASCII_HEADER_ASCII_BODY
	 * art text block which's look depends strongly on the taste of the author
	 * implementing this interface.
	 */
	void printBanner();

	/**
	 * Prints the help as of {@link #printHelp()} without the banner.
	 */
	default void printBody() {
		if ( getLicenseNote() != null ) {
			printLicenseNote();
			printSeparatorLn();
		}
		printSynopsis();
		printSeparatorLn();
		if ( getDescription() != null ) {
			printDescription();
			printSeparatorLn();
		}
		if ( getRootArgsSyntax() != null ) {
			printOptions();
			printSeparatorLn();
		}
		if ( getExamples() != null && getExamples().size() != 0 ) {
			printExamples();
			printSeparatorLn();
		}
		if ( getCopyrightNote() != null ) {
			printCopyrightNote();
			printSeparatorLn();
		}
	}

	/**
	 * Prints the copyright note as specified by the
	 * {@link #withCopyrightNote(String)} method; with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printCopyrightNote();

	/**
	 * Prints the description as set by the {@link #withDescription(String)}
	 * method with regards to the console width as specified by the
	 * {@link #withConsoleWidth(int)} method.
	 */
	void printDescription();

	/**
	 * Prints example usages as as added by the
	 * {@link #addExample(String, Operand...)} method with regards to the
	 * console width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printExamples();

	/**
	 * Prints the help to the standard output specified by the
	 * {@link #withStandardOut(PrintStream)} method. This method can make use of
	 * the more atomic methods {@link #printBanner()}, {@link #printSynopsis()},
	 * {@link #printDescription()}, {@link #printOptions()} or
	 * {@link #printSeparatorLn()} to print a help text which's look depends
	 * strongly on the taste of the author implementing this interface. In case
	 * you dislike the implementing author's taste, feel free to overwrite this
	 * method and compose your own help text from the building blocks such as
	 * {@link #printBanner()}, {@link #printSynopsis()},
	 * {@link #printDescription()}, {@link #printOptions()} or
	 * {@link #printSeparatorLn()}
	 */
	default void printHelp() {
		printBanner();
		printBody();
	}

	/**
	 * Prints the license note as specified by the
	 * {@link #withLicenseNote(String)} method; with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printLicenseNote();

	/**
	 * Prints an empty line / a line break.
	 */
	void printLn();

	/**
	 * Prints the given line to standard out with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 * 
	 * @param aLine The line to be printed.
	 */
	void printLn( String aLine );

	/**
	 * Prints the {@link Option}s (short- and the long-options), the
	 * {@link Flag}es and the {@link Operand} and their description with regards
	 * to the console width as specified by the {@link #withConsoleWidth(int)}
	 * method.
	 */
	void printOptions();

	/**
	 * Prints a separator line using the separator character as specified by the
	 * {@link #withSeparatorLnChar(char)} method; with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printSeparatorLn();

	/**
	 * Prints the syntax as retrieved by the root {@link ArgsSyntax} element as
	 * of {@link ArgsSyntax#toSynopsis(SyntaxNotation)} with regards to the
	 * {@link SyntaxNotation} set by the
	 * {@link #withSyntaxNotation(SyntaxNotation)} method.
	 */
	void printSynopsis();

	/**
	 * Sets the banner font.
	 *
	 * @param aBannerFont the new banner font
	 */
	void setBannerFont( Font aBannerFont );

	/**
	 * Sets the banner font palette.
	 *
	 * @param aColorPalette the new banner font palette
	 */
	void setBannerFontPalette( char[] aColorPalette );

	/**
	 * Set the console with. A setting of "-1" makes the instance use the
	 * <code>SystemUtility.getTerminalWidth()</code> value, i.e. the console
	 * width is set automatically to be the width of your terminal.
	 * 
	 * @param aConsoleWidth The width to set or -1 to let the parser
	 *        automatically determine the console width.
	 */
	void setConsoleWidth( int aConsoleWidth );

	/**
	 * Set the copyright note used by the {@link #printHelp()} method when
	 * writing out the copyright claim and make other adjustments with the
	 * result (with regards to the Builder-Pattern).
	 * 
	 * @param aCopyrightNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 */
	void setCopyrightNote( String aCopyrightNote );

	/**
	 * Set the error out {@link PrintStream} and make other adjustments with the
	 * result (with regards to the Builder-Pattern).
	 * 
	 * @param aErrorOut The error out {@link PrintStream} to set.
	 */
	void setErrorOut( PrintStream aErrorOut );

	/**
	 * Set the license note used by the {@link #printHelp()} method when writing
	 * out the licensing conditions.
	 * 
	 * @param aLicenseNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 */
	void setLicenseNote( String aLicenseNote );

	/**
	 * Set the console's line break. A setting of null makes the instance use
	 * the <code>SystemUtility.getLineBreak()</code> value.
	 *
	 * @param aLineBreak the new line break
	 */
	void setLineBreak( String aLineBreak );

	/**
	 * Set the maximum console width to use in case the console width is greater
	 * than the maximum you want. This is most useful when the console width is
	 * determined automatically to be the width of your terminal.
	 *
	 * @param aMaxConsoleWidth the new max console width
	 */
	void setMaxConsoleWidth( int aMaxConsoleWidth );

	/**
	 * Sets the description Escape-Code for the description Escape-Code
	 * property.
	 * 
	 * @param aDescriptionEscCode The description Escape-Code to be stored by
	 *        the description Escape-Code property.
	 */
	void setDescriptionEscapeCode( String aDescriptionEscCode );

	/**
	 * Sets the command name's (CLI executable) Escape-Code for the command name
	 * Escape-Code property.
	 * 
	 * @param aCommandEscCode The command name's Escape-Code to stored by the
	 *        command name Escape-Code property.
	 */
	void setCommandEscapeCode( String aCommandEscCode );

	/**
	 * Sets the parameter Escape-Code for the parameter Escape-Code property.
	 * 
	 * @param aParamEscCode The parameter Escape-Code to be stored by the
	 *        parameter Escape-Code property.
	 */
	void setParamEscapeCode( String aParamEscCode );

	/**
	 * Sets the option Escape-Code for the option Escape-Code property.
	 * 
	 * @param aOptEscCode The option Escape-Code to be stored by the option
	 *        Escape-Code property.
	 */
	void setOptionEscapeCode( String aOptEscCode );

	/**
	 * Sets the banner's content Escape-Code for the banner Escape-Code
	 * property.
	 * 
	 * @param aBannerEscCode The banner Escape-Code to be stored by the banner
	 *        Escape-Code property.
	 */
	void setBannerEscapeCode( String aBannerEscCode );

	/**
	 * Sets the line separator Escape-Code for the line separator Escape-Code
	 * property.
	 * 
	 * @param aLineSeparatorEscCode The line separator Escape-Code to be stored
	 *        by the line separator Escape-Code property.
	 */
	void setLineSeparatorEscapeCode( String aLineSeparatorEscCode );

	/**
	 * Sets the banner's border Escape-Code for the banner border Escape-Code
	 * property.
	 * 
	 * @param aBannerBorderEscCode The banner border Escape-Code to be stored by
	 *        the banner border Escape-Code property.
	 */
	void setBannerBorderEscapeCode( String aBannerBorderEscCode );

	/**
	 * Set the character to be used when printing a separator line with the
	 * {@link #printSeparatorLn()} method.
	 * 
	 * @param aSeparatorLnChar The character used by the
	 *        {@link #printSeparatorLn()} method when printing out the line of
	 *        characters..
	 */
	void setSeparatorLnChar( char aSeparatorLnChar );

	/**
	 * Set the standard out {@link PrintStream} and make other adjustments with
	 * the result (with regards to the Builder-Pattern).
	 * 
	 * @param aStandardOut The standard out {@link PrintStream} to set.
	 */
	void setStandardOut( PrintStream aStandardOut );

	/**
	 * Set the {@link SyntaxNotation}.
	 * 
	 * @param aSyntaxNotation The {@link SyntaxNotation} to set.
	 */
	void setSyntaxNotation( SyntaxNotation aSyntaxNotation );

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ArgsParser withTextBoxGrid( TextBoxGrid aTextBoxGrid ) {
		setTextBoxGrid( aTextBoxGrid );
		return this;
	}

	/**
	 * With banner font.
	 *
	 * @param aBannerFont the banner font
	 * 
	 * @return the args parser
	 */
	default ArgsParser withBannerFont( Font aBannerFont ) {
		setBannerFont( aBannerFont );
		return this;
	}

	/**
	 * With banner font palette.
	 *
	 * @param aColorPalette the color palette
	 * 
	 * @return the args parser
	 */
	default ArgsParser withBannerFontPalette( char[] aColorPalette ) {
		setBannerFontPalette( aColorPalette );
		return this;
	}

	/**
	 * Set the console with. A setting of "-1" makes the instance use the
	 * <code>SystemUtility.getTerminalWidth()</code> value.
	 * 
	 * @param aConsoleWidth The width to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withConsoleWidth( int aConsoleWidth ) {
		setConsoleWidth( aConsoleWidth );
		return this;
	}

	/**
	 * Set the copyright note used by the {@link #printHelp()} method when
	 * writing out the copyright claim and make other adjustments with the
	 * result (with regards to the Builder-Pattern).
	 * 
	 * @param aCopyrightNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withCopyrightNote( String aCopyrightNote ) {
		setCopyrightNote( aCopyrightNote );
		return this;
	}

	/**
	 * With description.
	 *
	 * @param aDescription the description
	 * 
	 * @return the args parser
	 */
	@Override
	default ArgsParser withDescription( String aDescription ) {
		setDescription( aDescription );
		return this;
	}

	/**
	 * Set the error out {@link PrintStream} and make other adjustments with the
	 * result (with regards to the Builder-Pattern).
	 * 
	 * @param aErrorOut The error out {@link PrintStream} to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withErrorOut( PrintStream aErrorOut ) {
		setErrorOut( aErrorOut );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ArgsParser withEscapeCodesEnabled( boolean isEscCodeEnabled ) {
		setEscapeCodesEnabled( isEscCodeEnabled );
		return this;
	}

	/**
	 * Same as {@link #evalArgs(List)} with the difference that this class's
	 * instance being invoked is returned as of the builder pattern.
	 *
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return This invoked instance as of the builder pattern to chain method
	 *         calls.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	default ArgsParser withEvalArgs( List<String> aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		evalArgs( aArgs.toArray( new String[aArgs.size()] ) );
		return this;
	}

	/**
	 * Set the license note used by the {@link #printHelp()} method when writing
	 * out the licensing conditions and make other adjustments with the result
	 * (with regards to the Builder-Pattern).
	 * 
	 * @param aLicenseNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withLicenseNote( String aLicenseNote ) {
		setLicenseNote( aLicenseNote );
		return this;
	}

	/**
	 * Set the console's line break. A setting of null makes the instance use
	 * the <code>SystemUtility.getLineBreak()</code> value.
	 *
	 * @param aLineBreak the line break
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withLineBreak( String aLineBreak ) {
		setLineBreak( aLineBreak );
		return this;
	}

	/**
	 * Set the maximum console width to use in case the console width is greater
	 * than the maximum you want. This is most useful when the console width is
	 * determined automatically to be the width of your terminal.
	 *
	 * @param aMaxConsoleWidth the max console width
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withMaxConsoleWidth( int aMaxConsoleWidth ) {
		setMaxConsoleWidth( aMaxConsoleWidth );
		return this;
	}

	/**
	 * With name.
	 *
	 * @param aName the name
	 * 
	 * @return the args parser
	 */
	@Override
	default ArgsParser withName( String aName ) {
		setName( aName );
		return this;
	}

	/**
	 * Sets the description Escape-Code for the description Escape-Code
	 * property.
	 * 
	 * @param aDescriptionEscCode The parameter's description Escape-Code to be
	 *        stored by the parameter's description Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */

	default ArgsParser withDescriptionEscapeCode( String aDescriptionEscCode ) {
		setDescriptionEscapeCode( aDescriptionEscCode );
		return this;
	}

	/**
	 * Sets the command name's (CLI executable) Escape-Code for the command name
	 * Escape-Code property.
	 * 
	 * @param aCommandEscCode The command name's Escape-Code to stored by the
	 *        command name Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	default ArgsParser withCommandEscapeCode( String aCommandEscCode ) {
		setCommandEscapeCode( aCommandEscCode );
		return this;
	}

	/**
	 * Sets the parameter Escape-Code for the parameter Escape-Code property.
	 * 
	 * @param aParamEscCode The parameter Escape-Code to be stored by the
	 *        parameter Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	default ArgsParser withParamEscapeCode( String aParamEscCode ) {
		setParamEscapeCode( aParamEscCode );
		return this;
	}

	/**
	 * Sets the option Escape-Code for the option Escape-Code property.
	 * 
	 * @param aOptEscCode The option Escape-Code to be stored by the option
	 *        Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	default ArgsParser withOptionEscapeCode( String aOptEscCode ) {
		setParamEscapeCode( aOptEscCode );
		return this;
	}

	/**
	 * Sets the banner's content Escape-Code for the banner Escape-Code
	 * property.
	 * 
	 * @param aBannerEscCode The banner Escape-Code to be stored by the banner
	 *        Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	default ArgsParser withBannerEscapeCode( String aBannerEscCode ) {
		setBannerEscapeCode( aBannerEscCode );
		return this;
	}

	/**
	 * Sets the line separator Escape-Code for the line separator Escape-Code
	 * property.
	 * 
	 * @param aLineSeparatorEscCode The line separator Escape-Code to be stored
	 *        by the line separator Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	default ArgsParser withLineSeparatorEscapeCode( String aLineSeparatorEscCode ) {
		setLineSeparatorEscapeCode( aLineSeparatorEscCode );
		return this;
	}

	/**
	 * Sets the banner's border Escape-Code for the banner border Escape-Code
	 * property.
	 * 
	 * @param aBannerBorderEscCode The banner border Escape-Code to be stored by
	 *        the banner border Escape-Code property.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	default ArgsParser withBannerBorderEscapeCode( String aBannerBorderEscCode ) {
		setBannerEscapeCode( aBannerBorderEscCode );
		return this;
	}

	/**
	 * Same as {@link #evalArgs(String[])} with the difference that this class's
	 * instance being invoked is returned as of the builder pattern.
	 *
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return This invoked instance as of the builder pattern to chain method
	 *         calls.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	@Override
	default ArgsParser withEvalArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		evalArgs( aArgs );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ArgsParser withResetEscapeCode( String aResetEscCode ) {
		setResetEscapeCode( aResetEscCode );
		return this;
	}

	/**
	 * Set the character to be used when printing a separator line with the
	 * {@link #printSeparatorLn()} method.
	 * 
	 * @param aSeparatorLnChar The character used by the
	 *        {@link #printSeparatorLn()} method when printing out the line of
	 *        characters..
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withSeparatorLnChar( char aSeparatorLnChar ) {
		setSeparatorLnChar( aSeparatorLnChar );
		return this;
	}

	/**
	 * Set the standard out {@link PrintStream} and make other adjustments with
	 * the result (with regards to the Builder-Pattern).
	 * 
	 * @param aStandardOut The standard out {@link PrintStream} to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withStandardOut( PrintStream aStandardOut ) {
		setStandardOut( aStandardOut );
		return this;
	}

	/**
	 * Set the {@link SyntaxNotation} and make other adjustments with the result
	 * (with regards to the Builder-Pattern).
	 * 
	 * @param aSyntaxNotation The {@link SyntaxNotation} to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withSyntaxNotation( SyntaxNotation aSyntaxNotation ) {
		setSyntaxNotation( aSyntaxNotation );
		return this;
	}

	/**
	 * With title.
	 *
	 * @param aTitle the title
	 * 
	 * @return the args parser
	 */
	@Override
	default ArgsParser withTitle( String aTitle ) {
		setTitle( aTitle );
		return this;
	}
}
