// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.io.File;

import org.refcodes.struct.Relation;

/**
 * Declarative syntactic sugar which may be statically imported in order to
 * allow declarative definitions for the command line {@link Flag},
 * {@link ArgsSyntax}, {@link Option} and {@link Operand} elements.
 */
public class CliSugar {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sugar for creating an {@link Example} array from a varargs argument.
	 * 
	 * @param aExamples the {@link Example} varargs argument.
	 * 
	 * @return The according array representation.
	 */
	public static Example[] examples( Example... aExamples ) {
		return aExamples;
	}

	/**
	 * Sugar for creating an {@link Example} from a description and the
	 * according {@link Operand} elements.
	 * 
	 * @param aDescription The explanatory text of the example.
	 * @param aOperands The operands required for the {@link Example}.
	 * 
	 * @return The according {@link Example} instance.
	 */
	public static Example example( String aDescription, Operand<?>... aOperands ) {
		return new Example( aDescription, aOperands );
	}

	/**
	 * Constructs a {@link Operation} with the given arguments.
	 * 
	 * @param aOperation The operation to declare.
	 * @param aDescription A description without any line breaks.
	 * 
	 * @return The accordingly created {@link Operation}.
	 */
	public static Operation operation( String aOperation, String aDescription ) {
		return new Operation( aOperation, aDescription );
	}

	/**
	 * Constructs a {@link Operation} with the given arguments.
	 * 
	 * @param aOperation The operation to declare.
	 * @param aAlias The operation's name to be used when constructing the
	 *        syntax.
	 * @param aDescription A description without any line breaks.
	 * 
	 * @return The accordingly created {@link Operation}.
	 */
	public static Operation operation( String aOperation, String aAlias, String aDescription ) {
		return new Operation( aOperation, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link AndCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested.
	 * 
	 * @return The according AND condition.
	 * 
	 * @see AndCondition
	 */
	public static ArgsSyntax and( Syntaxable... aArgs ) {
		return new AndCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link AllCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instance to be nested.
	 *
	 * @param aArg The {@link Syntaxable} ({@link ArgsSyntax}) instance to be
	 *        nested.
	 * 
	 * @return The according ALL condition.
	 * 
	 * @see AllCondition
	 */
	public static ArgsSyntax all( Syntaxable aArg ) {
		return new AllCondition( aArg );
	}

	/**
	 * Instantiates a new {@link OrCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested.
	 * 
	 * @return The according OR condition.
	 * 
	 * @see OrCondition
	 */
	public static ArgsSyntax or( Syntaxable... aArgs ) {
		return new OrCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested.
	 * 
	 * @return The according XOR condition.
	 * 
	 * @see XorCondition
	 */
	public static ArgsSyntax xor( Syntaxable... aArgs ) {
		return new XorCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link CasesCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested.
	 * 
	 * @return The according CasesCondition (special XOR) condition.
	 * 
	 * @see CasesCondition
	 */
	public static ArgsSyntax cases( Syntaxable... aArgs ) {
		return new CasesCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link AnyCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested. Any of the nested
	 * {@link ArgsSyntax} conditions may match for the {@link AnyCondition} to
	 * match, e.g. all of the nested conditions are optional.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested
	 * 
	 * @return The according {@link AbstractCondition}.
	 * 
	 * @see AnyCondition
	 */
	public static ArgsSyntax any( Syntaxable... aArgs ) {
		return new AnyCondition( aArgs );
	}

	/**
	 * Semantically identical synonym for the {@link #optional(Syntaxable...)}
	 * declaration.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested
	 * 
	 * @return The according {@link AbstractCondition}.
	 * 
	 * @see AnyCondition
	 */
	public static ArgsSyntax optional( Syntaxable... aArgs ) {
		return new AnyCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aAlias The identifier to be used when printing the syntax via the
	 *        {@link Syntaxable#toSyntax(SyntaxNotation, String, String, String)}
	 *        method.
	 * @param aDescription A description without any line breaks.
	 * 
	 * @return the none operand
	 * 
	 * @see NoneOperand
	 */
	public static NoneOperand none( String aAlias, String aDescription ) {
		return new NoneOperand( aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aDescription A description without any line breaks.
	 * 
	 * @return the none operand
	 * 
	 * @see NoneOperand
	 */
	public static NoneOperand none( String aDescription ) {
		return new NoneOperand( aDescription );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aProperty The key (= alias) and the value for the operand.
	 * 
	 * @return the none operand
	 * 
	 * @see NoneOperand
	 */
	public static NoneOperand none( Relation<String, Boolean> aProperty ) {
		return new NoneOperand( aProperty );
	}

	/**
	 * Instantiates a new enum option.
	 *
	 * @param <T> the generic type
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aType the type
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see EnumOption
	 */
	public static <T extends Enum<T>> EnumOption<T> enumOption( String aShortOption, String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		return new EnumOption<T>( aShortOption, aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * Instantiates a new enum option.
	 *
	 * @param <T> the generic type
	 * @param aLongOption the long option
	 * @param aType the type
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see EnumOption
	 */
	public static <T extends Enum<T>> EnumOption<T> enumOption( String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		return new EnumOption<T>( aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * As "switch" is a reserved word in Java, we use "flag" :-(.
	 *
	 * @param aLongOption the long option
	 * @param aAlias The alias of the flag
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see Flag
	 */
	public static Flag flag( String aLongOption, String aAlias, String aDescription ) {
		return new Flag( aLongOption, aAlias, aDescription );
	}

	/**
	 * As "switch" is a reserved word in Java, we use "flag" :-(.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias The alias of the flag
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see Flag
	 */
	public static Flag flag( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new Flag( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Constructs the predefined daemon {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see DaemonFlag
	 */
	public static DaemonFlag daemonFlag( String aDescription ) {
		return new DaemonFlag( aDescription );
	}

	/**
	 * Constructs the predefined daemon {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see DaemonFlag
	 */
	public static DaemonFlag daemonFlag() {
		return new DaemonFlag();
	}

	/**
	 * Constructs the predefined force {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see ForceFlag
	 */
	public static ForceFlag forceFlag( String aDescription ) {
		return new ForceFlag( aDescription );
	}

	/**
	 * Constructs the predefined force {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see ForceFlag
	 */
	public static ForceFlag forceFlag() {
		return new ForceFlag();
	}

	/**
	 * Constructs the predefined force {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see ForceFlag
	 */
	public static ForceFlag forceFlag( boolean hasShortOption ) {
		return new ForceFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined help {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see HelpFlag
	 */
	public static HelpFlag helpFlag( String aDescription ) {
		return new HelpFlag( aDescription );
	}

	/**
	 * Constructs the predefined help {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see HelpFlag
	 */
	public static HelpFlag helpFlag() {
		return new HelpFlag();
	}

	/**
	 * Constructs the predefined help {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see HelpFlag
	 */
	public static HelpFlag helpFlag( boolean hasShortOption ) {
		return new HelpFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined init {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see InitFlag
	 */
	public static InitFlag initFlag( String aDescription ) {
		return new InitFlag( aDescription );
	}

	/**
	 * Constructs the predefined init {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see InitFlag
	 */
	public static InitFlag initFlag() {
		return new InitFlag();
	}

	/**
	 * Constructs the predefined clean {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see CleanFlag
	 */
	public static CleanFlag cleanFlag( String aDescription ) {
		return new CleanFlag( aDescription );
	}

	/**
	 * Constructs the predefined clean {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see CleanFlag
	 */
	public static CleanFlag cleanFlag() {
		return new CleanFlag();
	}

	/**
	 * Constructs the predefined system info {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see SysInfoFlag
	 */
	public static SysInfoFlag sysInfoFlag( String aDescription ) {
		return new SysInfoFlag( aDescription );
	}

	/**
	 * Constructs the predefined system info {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see SysInfoFlag
	 */
	public static SysInfoFlag sysInfoFlag() {
		return new SysInfoFlag();
	}

	/**
	 * Constructs the predefined quiet {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see QuietFlag
	 */
	public static QuietFlag quietFlag( String aDescription ) {
		return new QuietFlag( aDescription );
	}

	/**
	 * Constructs the predefined quiet {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see QuietFlag
	 */
	public static QuietFlag quietFlag() {
		return new QuietFlag();
	}

	/**
	 * Constructs the predefined quiet {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see QuietFlag
	 */
	public static QuietFlag quietFlag( boolean hasShortOption ) {
		return new QuietFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined verbose {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see VerboseFlag
	 */
	public static VerboseFlag verboseFlag( String aDescription ) {
		return new VerboseFlag( aDescription );
	}

	/**
	 * Constructs the predefined verbose {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see VerboseFlag
	 */
	public static VerboseFlag verboseFlag() {
		return new VerboseFlag();
	}

	/**
	 * Constructs the predefined verbose {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see VerboseFlag
	 */
	public static VerboseFlag verboseFlag( boolean hasShortOption ) {
		return new VerboseFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined debug {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The according {@link Flag}.
	 * 
	 * @see DebugFlag
	 */
	public static DebugFlag debugFlag( String aDescription ) {
		return new DebugFlag( aDescription );
	}

	/**
	 * Constructs the predefined debug {@link Flag}.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see DebugFlag
	 */
	public static DebugFlag debugFlag() {
		return new DebugFlag();
	}

	/**
	 * Constructs the predefined debug {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The according {@link Flag}.
	 * 
	 * @see DebugFlag
	 */
	public static DebugFlag debugFlag( boolean hasShortOption ) {
		return new DebugFlag( hasShortOption );
	}

	/**
	 * Int option.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see IntOption
	 */
	public static IntOption intOption( String aLongOption, String aAlias, String aDescription ) {
		return new IntOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Int option.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see IntOption
	 */
	public static IntOption intOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new IntOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Long option.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see LongOption
	 */
	public static LongOption longOption( String aLongOption, String aAlias, String aDescription ) {
		return new LongOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Long option.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see LongOption
	 */
	public static LongOption longOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new LongOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Float option.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see LongOption
	 */
	public static FloatOption floatOption( String aLongOption, String aAlias, String aDescription ) {
		return new FloatOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Float option.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see LongOption
	 */
	public static FloatOption floatOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new FloatOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Double option.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see LongOption
	 */
	public static DoubleOption doubleOption( String aLongOption, String aAlias, String aDescription ) {
		return new DoubleOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Double option.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see LongOption
	 */
	public static DoubleOption doubleOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new DoubleOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * String option.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see StringOption
	 */
	public static StringOption stringOption( String aLongOption, String aAlias, String aDescription ) {
		return new StringOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * String option.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see StringOption
	 */
	public static StringOption stringOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new StringOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Character option.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see StringOption
	 */
	public static CharOption charOption( String aLongOption, String aAlias, String aDescription ) {
		return new CharOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Character option.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see StringOption
	 */
	public static CharOption charOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new CharOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Creates a {@link ConfigOption} representing value specifying a
	 * configuration resource (file).
	 *
	 * @return the according {@link ConfigOption}.
	 * 
	 * @see ConfigOption
	 */
	public static ConfigOption configOption() {
		return new ConfigOption();
	}

	/**
	 * Creates a {@link ConfigOption} representing a value specifying a
	 * configuration resource (file).
	 * 
	 * @param aDescription The description to use when printing out the help
	 *        text. to use when printing out the help text.
	 *
	 * @return the according {@link ConfigOption}.
	 * 
	 * @see ConfigOption
	 */
	public static ConfigOption configOption( String aDescription ) {
		return new ConfigOption( aDescription );
	}

	/**
	 * Creates a {@link FileOption} for getting a {@link File} instance from a
	 * path.
	 *
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see FileOption
	 */
	public static FileOption fileOption( String aLongOption, String aAlias, String aDescription ) {
		return new FileOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Creates a {@link FileOption} for getting a {@link File} instance from a
	 * path.
	 *
	 * @param aShortOption the short option
	 * @param aLongOption the long option
	 * @param aAlias the option arg name
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the option
	 * 
	 * @see FileOption
	 */
	public static FileOption fileOption( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new FileOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * String operand.
	 *
	 * @param aIdentifier the identifier
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return the operand
	 * 
	 * @see StringOperand
	 */
	public static StringOperand stringOperand( String aIdentifier, String aDescription ) {
		return new StringOperand( aIdentifier, aDescription );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Option}. This way any {@link Option} can also be used as an array
	 * {@link Option}, e.g. it can be provided multiple times in the command
	 * line arguments.
	 * 
	 * @param <T> The type of the {@link Option} for which to create an
	 *        {@link ArrayOption}.
	 * @param aOption The {@link Option} which's array counterpart is to be
	 *        defined.
	 * 
	 * @return The according {@link ArrayOption}.
	 */
	public static <T> ArrayOption<T> asArray( Option<T> aOption ) {
		return new ArrayOption<T>( aOption );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Option}. This way any {@link Option} can also be used as an array
	 * {@link Option}, e.g. it can by provided multiple times in the command
	 * line arguments.
	 * 
	 * @param <T> The type of the {@link Option} for which to create an
	 *        {@link ArrayOption}.
	 * @param aOption The {@link Option} which's array counterpart is to be
	 *        defined.
	 * @param aLength The number of array elements, or -1 if there is no limit.
	 * 
	 * @return The according {@link ArrayOption}.
	 */
	public static <T> ArrayOption<T> asArray( Option<T> aOption, int aLength ) {
		return new ArrayOption<T>( aOption, aLength );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Option}. This way any {@link Option} can also be used as an array
	 * {@link Option}, e.g. it can by provided multiple times in the command
	 * line arguments.
	 * 
	 * @param <T> The type of the {@link Option} for which to create an
	 *        {@link ArrayOption}.
	 * @param aOption The {@link Option} which's array counterpart is to be
	 *        defined.
	 * @param aMinLength The minimum number of array elements, or -1 if there is
	 *        no limit.
	 * @param aMaxLength The maximum number of array elements, or -1 if there is
	 *        no limit.
	 * 
	 * @return The according {@link ArrayOption}.
	 */
	public static <T> ArrayOption<T> asArray( Option<T> aOption, int aMinLength, int aMaxLength ) {
		return new ArrayOption<T>( aOption, aMinLength, aMaxLength );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Operand}. This way any {@link Operand} can also be used as an
	 * array {@link Operand}, e.g. it can be provided multiple times in the
	 * command line arguments.
	 * 
	 * @param <T> The type of the {@link Operand} for which to create an
	 *        {@link ArrayOperand}.
	 * @param aOperand The {@link Operand} which's array counterpart is to be
	 *        defined.
	 * 
	 * @return The according {@link ArrayOperand}.
	 */
	public static <T> ArrayOperand<T> asArray( Operand<T> aOperand ) {
		return new ArrayOperand<T>( aOperand );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Operand}. This way any {@link Operand} can also be used as an
	 * array {@link Operand}, e.g. it can by provided multiple times in the
	 * command line arguments.
	 * 
	 * @param <T> The type of the {@link Operand} for which to create an
	 *        {@link ArrayOperand}.
	 * @param aOperand The {@link Operand} which's array counterpart is to be
	 *        defined.
	 * @param aLength The number of array elements, or -1 if there is no limit.
	 * 
	 * @return The according {@link ArrayOperand}.
	 */
	public static <T> ArrayOperand<T> asArray( Operand<T> aOperand, int aLength ) {
		return new ArrayOperand<T>( aOperand, aLength );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Operand}. This way any {@link Operand} can also be used as an
	 * array {@link Operand}, e.g. it can by provided multiple times in the
	 * command line arguments.
	 * 
	 * @param <T> The type of the {@link Operand} for which to create an
	 *        {@link ArrayOperand}.
	 * @param aOperand The {@link Operand} which's array counterpart is to be
	 *        defined.
	 * @param aMinLength The minimum number of array elements, or -1 if there is
	 *        no limit.
	 * @param aMaxLength The maximum number of array elements, or -1 if there is
	 *        no limit.
	 * 
	 * @return The according {@link ArrayOperand}.
	 */
	public static <T> ArrayOperand<T> asArray( Operand<T> aOperand, int aMinLength, int aMaxLength ) {
		return new ArrayOperand<T>( aOperand, aMinLength, aMaxLength );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
