// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.mixin.EnabledAccessor;
import org.refcodes.struct.Relation;

/**
 * The {@link NoneOperand} represents an empty set of arguments, e.g. no command
 * line argument is being passed. Usually only makes sense near the root of a
 * {@link Syntaxable} tree (e.g. inside the main {@link XorCondition}) to test
 * ALL arguments passed to an application and one invocation may have neither
 * optional (as of {@link AnyCondition}) nor mandatory arguments allowed.
 */
public class NoneOperand extends AbstractOperand<Boolean> implements Operand<Boolean>, EnabledAccessor {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String EMPTY_SET = "∅";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aDescription A description without any line breaks.
	 */
	public NoneOperand( String aDescription ) {
		super( Boolean.class, EMPTY_SET, aDescription );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aAlias The identifier to be used when printing the syntax via the
	 *        {@link #toSyntax(SyntaxNotation, String, String, String)} method.
	 * @param aDescription A description without any line breaks.
	 */
	public NoneOperand( String aAlias, String aDescription ) {
		super( Boolean.class, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aProperty The key (= alias) and the value for the operand.
	 */
	public NoneOperand( Relation<String, Boolean> aProperty ) {
		super( aProperty, Boolean.class );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<Boolean>> parseArgs( String[] aArgs, String[] aOptions ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException, SuperfluousArgsException {
		boolean isEmpty = aArgs == null || aArgs.length == 0;
		setValue( isEmpty );
		if ( !isEmpty ) {
			throw new UnknownArgsException( aArgs, "No arguments are expected but <" + aArgs.length + "> arguments have been provided!" );
		}
		List<Operand<Boolean>> theResult = new ArrayList<>();
		theResult.add( this );
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		aParamEscCode = aParamEscCode == null ? "" : aParamEscCode;
		aResetEscCode = aResetEscCode == null ? "" : aResetEscCode;
		return aParamEscCode + EMPTY_SET + aResetEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		return EMPTY_SET;
	}

	/**
	 * Determines whether none arguments have been passed.
	 * 
	 * @return True in case none arguments have been passed.
	 */
	@Override
	public boolean isEnabled() {
		Boolean theValue = getValue();
		if ( theValue != null ) {
			return theValue;
		}
		return false;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To value.
	 *
	 * @param aArg the arg
	 * 
	 * @return the string
	 */
	@Override
	protected Boolean toType( String aArg ) {
		return (aArg == null || aArg.length() == 0);
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
