// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import org.refcodes.data.CommandArgPrefix;

/**
 * An {@link Option} represents a command line option with the according
 * option's value. An {@link Option} can be seen as a key/value(s) pair defined
 * in the command line arguments parsed via the
 * {@link #parseArgs(String[], String[])} method.
 * <p>
 * An {@link Option} has a state which changes with each invocation of the
 * {@link #parseArgs(String[], String[])} method.
 *
 * @param <T> the generic type
 */
public interface Option<T> extends Operand<T> /* , Relation<String, T> */ {

	/**
	 * Returns the key for this Option. In case we have a long option, then the
	 * key will be the long option (as of {@link #getLongOption()}). Else the
	 * key will be the short option (as of {@link #getShortOption()}).
	 * Reasoning: As the Option becomes of type {@link Relation}, we can add it
	 * to a {@link MutableDictionary} via
	 * {@link MutableDictionary#put(Relation)}.
	 *
	 * @return the short option
	 */
	// @Override
	// default String getKey() {
	// return getLongOption() != null ? getLongOption() : getShortOption();
	// }

	/**
	 * Returns the short-option representing an {@link Option} instance. A short
	 * option usually is being prefixed with a single hyphen-minus "-" as
	 * defined in {@link CommandArgPrefix#POSIX_SHORT_OPTION}.
	 * 
	 * @return The short-option {@link String}.
	 */
	String getShortOption();

	/**
	 * Returns the long-option representing an {@link Option} instance. A long
	 * option usually is being prefixed with a double hyphen-minus "--" as
	 * defined in {@link CommandArgPrefix#POSIX_LONG_OPTION}.
	 * 
	 * @return The long-option {@link String}.
	 */
	String getLongOption();

	/**
	 * Retrieves an array of the options being configured for the implementing
	 * instance, being the long option (when set) and the short option (when
	 * set).
	 * 
	 * @return The array of options being the long option (if not null) and the
	 *         short option (if not null).
	 */
	default String[] getOptions() {
		int i = 0;
		if ( getLongOption() != null ) i++;
		if ( getShortOption() != null ) i++;
		String[] theResult = new String[i];
		i = 0;
		if ( getShortOption() != null ) {
			theResult[i] = getShortOption();
			i++;
		}
		if ( getLongOption() != null ) theResult[i] = getLongOption();
		return theResult;
	}
}
