// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

/**
 * A no-operation {@link ArgsSyntax} used to wrap an {@link Option} in places
 * where a {@link ArgsSyntax} is expected.
 */
public class OptionCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String RIGHT_TAG = " )";
	private static final String LEFT_TAG = "( ";

	private static final char GUN_POSIX_NOP = '/';
	private static final String LOGICAL_NOP = " / ";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link OptionCondition} with the {@link Option}
	 * ({@link Syntaxable}) instance to be wrapped.
	 *
	 * @param aArg The {@link Option} ({@link Syntaxable}) instance to be
	 *        wrapped
	 */
	public OptionCondition( Option<?> aArg ) {
		super( aArg );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException {
		List<Operand<?>> theResult = new ArrayList<Operand<?>>();
		List<Operand<?>> eOperands;
		for ( Syntaxable eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions );
			}
			catch ( UnknownArgsException exc ) {
				throw new UnknownArgsException( exc.getArgs(), "At least one mandatory syntax did not match the provided command line arguments, though all mandatories must match.", exc );
			}
			catch ( AmbiguousArgsException exc ) {
				throw new AmbiguousArgsException( exc.getArgs(), "At least one mandatory syntax did not match the provided command line arguments, though all mandatories must match.", exc );
			}
			if ( eOperands != null ) {
				theResult.addAll( eOperands );
			}
			aArgs = CliUtility.toDiff( aArgs, eOperands );
		}
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() != 0 ) {
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( LOGICAL_NOP );
				}
				else {
					theBuilder.append( GUN_POSIX_NOP );
				}
			}
			theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( LOGICAL_NOP );
			}
			theBuilder.append( eChild.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
