// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * The {@link AbstractCondition} is an abstract implementation of the
 * {@link ArgsSyntax} interface providing the boiler plate when implementing the
 * {@link ArgsSyntax} interface as done by the {@link AbstractCondition}'s
 * sub-classes.
 */
abstract public class AbstractCondition extends AbstractSyntaxable implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private List<Syntaxable> _children = new ArrayList<Syntaxable>();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new abstract condition.
	 *
	 * @param aElements the elements
	 */
	public AbstractCondition( Syntaxable... aElements ) {
		_children.addAll( Arrays.asList( aElements ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		StringBuilder theBuilder = new StringBuilder( toSynopsis( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
		if ( theBuilder.length() > 0 ) {
			if ( getChildren() != null && getChildren().size() > 1 ) {
				theBuilder.insert( 0, "( " );
				theBuilder.append( " )" );
			}
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		for ( Syntaxable eSyntaxable : _children ) {
			eSyntaxable.reset();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		String theState = toState();
		// Remove annoying encapsulating parenthesis:
		if ( theState != null && theState.length() >= 4 && theState.startsWith( "( " ) && theState.endsWith( " )" ) ) {
			theState = theState.substring( 2 ).substring( 0, theState.length() - 4 );
		}
		return theState;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<?>> toOperands() {
		List<Operand<?>> theList = new ArrayList<Operand<?>>();
		for ( Synopsis eSyntaxable : getChildren() ) {
			if ( eSyntaxable instanceof Operand<?> ) {
				theList.add( (Operand<?>) eSyntaxable );
			}
			if ( eSyntaxable instanceof ArgsSyntax ) {
				theList.addAll( ((ArgsSyntax) eSyntaxable).toOperands() );
			}
		}
		return theList;
	}

	/**
	 * Gets the children.
	 *
	 * @return the children
	 */
	protected List<Syntaxable> getChildren() {
		return _children;
	}

	/**
	 * Gets the first child. It is up to the developer to make sure that there
	 * actually is(!) a child!
	 *
	 * @return the first child.
	 */
	protected Syntaxable getFirst() {
		return _children.get( 0 );
	}

	/**
	 * Adds the child.
	 *
	 * @param aArgumentizer the argumentizer
	 */
	protected void addChild( Syntaxable aArgumentizer ) {
		_children.add( aArgumentizer );
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <V> V toValue( String aAlias ) {
		Object eValue;
		for ( Syntaxable eElement : _children ) {
			eValue = eElement.toValue( aAlias );
			if ( eValue != null ) {
				return (V) eValue;
			}
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends Operand<?>> T toOperand( String aAlias, Class<T> aType ) {
		Operand<?> eOperand;
		ArgsSyntax eCondition;
		boolean eTypeMatch, eAliasMatch;
		for ( Syntaxable eSyntaxable : getChildren() ) {

			if ( eSyntaxable instanceof Operand ) {
				eOperand = (Operand<?>) eSyntaxable;
				eTypeMatch = true;
				eAliasMatch = true;
				if ( aAlias != null && aAlias.length() != 0 ) {
					eAliasMatch = aAlias.equals( eOperand.getAlias() );
				}
				if ( aType != null ) {
					eTypeMatch = aType.isAssignableFrom( eOperand.getClass() );
				}
				if ( eTypeMatch && eAliasMatch ) {
					return (T) eOperand;
				}
			}
			else if ( eSyntaxable instanceof ArgsSyntax ) {
				eCondition = (ArgsSyntax) eSyntaxable;
				eOperand = eCondition.toOperand( aAlias, aType );
				if ( eOperand != null ) {
					return (T) eOperand;
				}
			}
		}
		return null;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
