// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import org.refcodes.data.ExitCode;
import org.refcodes.exception.AbstractException;
import org.refcodes.mixin.StatusCodeAccessor;

/**
 * Base exception for the console artifact.
 *
 * @author steiner
 */
public abstract class CliException extends AbstractException {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new console exception.
	 *
	 * @param aMessage the message
	 * @param aErrorCode the error code
	 */
	public CliException( String aMessage, String aErrorCode ) {
		super( aMessage, aErrorCode );
	}

	/**
	 * Instantiates a new console exception.
	 *
	 * @param aMessage the message
	 * @param aCause the cause
	 * @param aErrorCode the error code
	 */
	public CliException( String aMessage, Throwable aCause, String aErrorCode ) {
		super( aMessage, aCause, aErrorCode );
	}

	/**
	 * Instantiates a new console exception.
	 *
	 * @param message the message
	 * @param cause the cause
	 */
	public CliException( String message, Throwable cause ) {
		super( message, cause );
	}

	/**
	 * Instantiates a new console exception.
	 *
	 * @param message the message
	 */
	public CliException( String message ) {
		super( message );
	}

	/**
	 * Instantiates a new console exception.
	 *
	 * @param aCause the cause
	 * @param aErrorCode the error code
	 */
	public CliException( Throwable aCause, String aErrorCode ) {
		super( aCause, aErrorCode );
	}

	/**
	 * Instantiates a new console exception.
	 *
	 * @param cause the cause
	 */
	public CliException( Throwable cause ) {
		super( cause );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class ConsoleCliException.
	 */
	public static abstract class ConsoleCliException extends CliException implements StatusCodeAccessor<ExitCode> {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////////

		private ExitCode _exitCode;

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new console cli exception.
		 *
		 * @param aExitCode the exit code
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public ConsoleCliException( ExitCode aExitCode, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_exitCode = aExitCode;
		}

		/**
		 * Instantiates a new console cli exception.
		 *
		 * @param aExitCode the exit code
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ConsoleCliException( ExitCode aExitCode, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_exitCode = aExitCode;
		}

		/**
		 * Instantiates a new console cli exception.
		 *
		 * @param aExitCode the exit code
		 * @param message the message
		 * @param cause the cause
		 */
		public ConsoleCliException( ExitCode aExitCode, String message, Throwable cause ) {
			super( message, cause );
			_exitCode = aExitCode;
		}

		/**
		 * Instantiates a new console cli exception.
		 *
		 * @param aExitCode the exit code
		 * @param message the message
		 */
		public ConsoleCliException( ExitCode aExitCode, String message ) {
			super( message );
			_exitCode = aExitCode;
		}

		/**
		 * Instantiates a new console cli exception.
		 *
		 * @param aExitCode the exit code
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ConsoleCliException( ExitCode aExitCode, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_exitCode = aExitCode;
		}

		/**
		 * Instantiates a new console cli exception.
		 *
		 * @param aExitCode the exit code
		 * @param cause the cause
		 */
		public ConsoleCliException( ExitCode aExitCode, Throwable cause ) {
			super( cause );
			_exitCode = aExitCode;
		}

		// /////////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Gets the status code.
		 *
		 * @return the status code
		 */
		@Override
		public ExitCode getStatusCode() {
			return _exitCode;
		}
	}

	/**
	 * This abstract exception is the base exception for all command line
	 * argument related exceptions.
	 * 
	 * @author steiner
	 */
	public static abstract class ConsoleArgsException extends CliException implements ArgsAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private String[] _args;

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new console args exception.
		 *
		 * @param aArgs the args
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public ConsoleArgsException( String[] aArgs, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_args = aArgs;
		}

		/**
		 * Instantiates a new console args exception.
		 *
		 * @param aArgs the args
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ConsoleArgsException( String[] aArgs, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_args = aArgs;
		}

		/**
		 * Instantiates a new console args exception.
		 *
		 * @param aArgs the args
		 * @param message the message
		 * @param cause the cause
		 */
		public ConsoleArgsException( String[] aArgs, String message, Throwable cause ) {
			super( message, cause );
			_args = aArgs;
		}

		/**
		 * Instantiates a new console args exception.
		 *
		 * @param aArgs the args
		 * @param message the message
		 */
		public ConsoleArgsException( String[] aArgs, String message ) {
			super( message );
			_args = aArgs;
		}

		/**
		 * Instantiates a new console args exception.
		 *
		 * @param aArgs the args
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ConsoleArgsException( String[] aArgs, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_args = aArgs;
		}

		/**
		 * Instantiates a new console args exception.
		 *
		 * @param aArgs the args
		 * @param cause the cause
		 */
		public ConsoleArgsException( String[] aArgs, Throwable cause ) {
			super( cause );
			_args = aArgs;
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Gets the args.
		 *
		 * @return the args
		 */
		@Override
		public String[] getArgs() {
			return _args;
		}
	}
}
