// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

/**
 * Any of the nested {@link ArgsSyntax} conditions may match for the
 * {@link AnyCondition} to match, e.g. all of the nested conditions are
 * optional. The {@link AnyCondition} can nested by an {@link AndCondition} and
 * encapsulates a {@link ArgsSyntax} which may be provided but does not
 * necessarily needs to be provided in order for the {@link AndCondition} to
 * successfully parse the nested {@link ArgsSyntax} items including the
 * {@link AnyCondition}.
 */
public class AnyCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String LEFT_TAG = "[ ";
	private static final String RIGHT_TAG = " ]";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link AnyCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested
	 */
	public AnyCondition( Syntaxable... aArgs ) {
		super( "Allows that any nested syntaxables optionally matches from the arguments.", aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException {
		List<Operand<?>> theResult = new ArrayList<Operand<?>>();
		List<Operand<?>> eOperands = null;
		for ( Syntaxable eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions );
				if ( eOperands != null ) {
					theResult.addAll( eOperands );
				}
				aArgs = CliUtility.toDiff( aArgs, eOperands );
			}
			catch ( UnknownArgsException ignore ) {
				/* optional! */
			}
			// catch ( AmbiguousArgsException exc ) { /* optional! */ }
		}
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		if ( getChildren() != null && getChildren().size() > 0 ) {
			StringBuilder theBuilder = new StringBuilder();
			for ( Syntaxable eChild : getChildren() ) {
				if ( theBuilder.length() != 0 ) {
					theBuilder.append( " " );
				}
				theBuilder.append( LEFT_TAG );
				theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
				theBuilder.append( RIGHT_TAG );
			}
			return theBuilder.toString();
		}
		return "";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( " " );
			}
			theBuilder.append( eChild.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		getChildren().get( 0 ).reset();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toState();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
