// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.data.CommandArgPrefix;
import org.refcodes.mixin.EnabledAccessor;

/**
 * The {@link Operation} is an argument representing a function or a method
 * ("command") and is either provided or not provided as of
 * {@link #isEnabled()}. It must neither be prefixed with "-" nor with "--" in
 * contrast to the {@link Option} or the {@link Flag} type.
 */
public class Operation extends AbstractOperand<Boolean> implements EnabledAccessor {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _operation;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link Operation} with the given arguments.
	 * 
	 * @param aOperation The operation to declare.
	 * @param aDescription A description without any line breaks.
	 */
	public Operation( String aOperation, String aDescription ) {
		this( aOperation, aOperation, aDescription );
	}

	/**
	 * Constructs a {@link Operation} with the given arguments.
	 * 
	 * @param aOperation The operation to declare.
	 * @param aAlias The operation's name to be used when constructing the
	 *        syntax.
	 * @param aDescription A description without any line breaks.
	 */
	public Operation( String aOperation, String aAlias, String aDescription ) {
		super( Boolean.class, aAlias, aDescription );
		if ( aOperation == null ) {
			throw new IllegalArgumentException( "The operation argument must not be null!" );
		}
		if ( aOperation.startsWith( CommandArgPrefix.POSIX_LONG_OPTION.getPrefix() ) | aOperation.startsWith( CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() ) ) {
			throw new IllegalArgumentException( "Your operation \"" + aOperation + "\" must not start with \"" + CommandArgPrefix.POSIX_SHORT_OPTION.getPrefix() + "\" or \"" + CommandArgPrefix.POSIX_LONG_OPTION.getPrefix() + "\" as if it were an option." );
		}
		_operation = aOperation;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns true if the switch has been set (enabled).
	 * 
	 * @return True in case the {@link Flag} has been provided (set), else
	 *         false.
	 */
	@Override
	public boolean isEnabled() {
		Boolean theValue = getValue();
		if ( theValue != null ) {
			return theValue;
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<Boolean>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException {
		boolean hasOperation;
		out: {
			if ( aArgs != null ) {
				for ( int i = 0; i < aArgs.length; i++ ) {
					if ( getOperation().equals( aArgs[i] ) ) {
						hasOperation = true;
						break out;
					}
				}

			}
			hasOperation = false;
		}
		if ( hasOperation ) {
			List<Operand<Boolean>> theList = new ArrayList<Operand<Boolean>>();
			theList.add( this );
			setArgs( new String[] { getOperation() } );
			setValue( true );
			return theList;
		}
		throw new UnknownArgsException( aArgs, "The operation \"" + getOperation() + "\" was found in the command line arguments." );
	}

	/**
	 * Returns the operation's name.
	 * 
	 * @return The according operation's name.
	 */
	public String getOperation() {
		return _operation;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		aOptEscCode = aOptEscCode == null ? "" : aOptEscCode;
		aParamEscCode = aParamEscCode == null ? "" : aParamEscCode;
		aResetEscCode = aResetEscCode == null ? "" : aResetEscCode;
		return aOptEscCode + getOperation() + (getAlias() != null ? (" " + CliUtility.toParameterSpec( this, aParamEscCode, aResetEscCode )) : "") + aResetEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		super.reset();
		super.setValue( false );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Boolean toType( String aArg ) throws ParseArgsException {
		throw new UnsupportedOperationException( "*** NOT REQUIRED BY THIS IMPLEMENTATION ***" );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
