// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.refcodes.textual.VerboseTextBuilder;

/**
 * An {@link XorCondition} represents a list of {@link ArgsSyntax}
 * ({@link Syntaxable}) instances of which only one is allowed to be parsed
 * successfully when the {@link ArgsSyntax#parseArgs(String[], String[])}
 * methods are invoked. The command line arguments syntax
 * <code>{ -a ^ -b ^ -c }</code> specifies that only "-a", only "-b" or only
 * "-c" must be set. In case more then one is set or none, then the
 * {@link XorCondition} will terminate the
 * {@link #parseArgs(String[], String[])} method with an exception.
 */
public class XorCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String OTHER_RIGHT_TAG = " }";
	private static final String OTHER_LEFT_TAG = "{ ";
	private static final String RIGHT_TAG = " )";
	private static final String LEFT_TAG = "( ";
	private static final String GUN_POSIX_XOR = " | ";
	private static final String LOGICAL_XOR = " ^ ";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested
	 */
	public XorCondition( Syntaxable... aArgs ) {
		super( "Enforces that exactly one of the nested syntaxables must math from the arguments.", aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException {

		// In case if one child only |-->
		//	if ( getChildren().size() == 1 ) {
		//		return getChildren().get( 0 ).parseArgs( aArgs );
		//	}
		// In case if one child only <--|

		List<ArgsSyntax> theOptionals = new ArrayList<ArgsSyntax>();
		List<Operand<?>> theResult = null;
		Synopsisable theResultSyntaxable = null;
		List<Operand<?>> eOperands = null;
		int theMatchCount = getChildren().size();
		for ( Syntaxable eChild : getChildren() ) {
			if ( eChild instanceof AnyCondition ) theOptionals.add( (AnyCondition) eChild );
			try {
				if ( theResult != null && eChild instanceof Operand && theResultSyntaxable instanceof Option ) {
					eOperands = (List<Operand<?>>) eChild.parseArgs( CliUtility.toDiff( aArgs, theResult ), aOptions );
				}
				else {
					eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions );
				}
				if ( eOperands != null && eOperands.isEmpty() ) eOperands = null;

			}
			catch ( ArgsSyntaxException ignore ) {
				theMatchCount--;
			}

			// Same operands in different children |-->
			theResult = removeDuplicates( theResult, eOperands );
			// Same operands in different children <--|

			if ( theResult != null && eOperands != null ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match: " + new VerboseTextBuilder().withElements( CliUtility.toArgs( theResult, eOperands ) ).toString() );
			}
			if ( eOperands != null && !eOperands.isEmpty() && theResult == null ) {
				theResult = eOperands;
				theResultSyntaxable = eChild;
			}
			eOperands = null;
		}

		if ( theMatchCount == 0 ) {
			throw new UnknownArgsException( aArgs, "Not one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match!" );
		}
		if ( theMatchCount > 1 ) {
			if ( theOptionals.size() > 1 && (aArgs != null && aArgs.length > 0) ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive optional syntax matched the provided command line arguments, causing exclusive contextual ambiguity: " + new VerboseTextBuilder().withElements( toSpec( theOptionals ) ).toString() );
			}
			theMatchCount -= theOptionals.size();
			if ( theMatchCount > 1 ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match:" );
			}
		}
		if ( theResult != null ) {
			return theResult;
		}
		if ( theOptionals.size() == 0 ) {
			return Collections.EMPTY_LIST;
		}

		throw new UnknownArgsException( aArgs, "Not one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match!" );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		StringBuilder theBuilder = new StringBuilder( toSynopsis( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
		if ( theBuilder.length() > 0 && getChildren() != null && getChildren().size() > 1 ) {
			if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
				theBuilder.insert( 0, LEFT_TAG );
			}
			else {
				theBuilder.insert( 0, OTHER_LEFT_TAG );
			}
			if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
				theBuilder.append( RIGHT_TAG );
			}
			else {
				theBuilder.append( OTHER_RIGHT_TAG );
			}
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() != 0 ) {
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( LOGICAL_XOR );
				}
				else {
					theBuilder.append( GUN_POSIX_XOR );
				}
			}
			theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( LOGICAL_XOR );
			}
			theBuilder.append( eChild.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static List<String> toSpec( List<ArgsSyntax> aArgsSyntaxs ) {
		List<String> theParameters = new ArrayList<>();
		for ( ArgsSyntax eArgsSyntax : aArgsSyntaxs ) {
			for ( Operand<?> eOperand : eArgsSyntax.toOperands() ) {
				theParameters.add( CliUtility.toSpec( eOperand ) );
			}
		}
		return theParameters;
	}

	private static List<Operand<?>> removeDuplicates( List<Operand<?>> aElements, List<Operand<?>> aDuplicates ) {
		if ( aElements != null && aDuplicates != null ) {
			for ( Operand<?> eOperand : aDuplicates ) {
				if ( aElements.contains( eOperand ) ) {
					aElements.remove( eOperand );
				}
			}
			if ( aElements.isEmpty() ) {
				aElements = null;
			}
		}
		return aElements;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
