// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * The {@link AbstractCondition} is an abstract implementation of the
 * {@link Condition} interface providing the boiler plate when implementing the
 * {@link Condition} interface as done by the {@link AbstractCondition}'s
 * sub-classes.
 */
abstract public class AbstractCondition extends AbstractConstituent implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private List<Constituent> _children = new ArrayList<Constituent>();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new abstract condition.
	 * 
	 * @param aDescription The description of this {@link Condition}.
	 * @param aElements the elements
	 */
	public AbstractCondition( String aDescription, Constituent... aElements ) {
		super( aDescription );
		_children.addAll( Arrays.asList( aElements ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( CliContext aCliCtx ) {
		String theSyntax = toSynopsis( aCliCtx );
		if ( theSyntax.length() > 0 && getChildren() != null && getChildren().size() > 1 ) {
			theSyntax = aCliCtx.getSyntaxMetrics().getBeginListSymbol() + " " + theSyntax + " " + aCliCtx.getSyntaxMetrics().getEndListSymbol();
		}
		return theSyntax.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		for ( Constituent eSyntaxable : _children ) {
			eSyntaxable.reset();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toSchema().toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CliSchema toSchema() {
		CliSchema[] theSchemas = null;
		if ( _children != null && _children.size() != 0 ) {
			theSchemas = new CliSchema[_children.size()];
			for ( int i = 0; i < theSchemas.length; i++ ) {
				theSchemas[i] = _children.get( i ).toSchema();
			}
		}
		return new CliSchema( getClass(), _description, theSchemas );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<?>> toOperands() {
		List<Operand<?>> theList = new ArrayList<Operand<?>>();
		for ( Synopsisable eSyntaxable : getChildren() ) {
			if ( eSyntaxable instanceof Operand<?> ) {
				theList.add( (Operand<?>) eSyntaxable );
			}
			if ( eSyntaxable instanceof Condition ) {
				theList.addAll( ((Condition) eSyntaxable).toOperands() );
			}
		}
		return theList;
	}

	/**
	 * Gets the children.
	 *
	 * @return the children
	 */
	protected List<Constituent> getChildren() {
		return _children;
	}

	/**
	 * Gets the first child. It is up to the developer to make sure that there
	 * actually is(!) a child!
	 *
	 * @return the first child.
	 */
	protected Constituent getFirst() {
		return _children.get( 0 );
	}

	/**
	 * Adds the child.
	 *
	 * @param aArgumentizer the argumentizer
	 */
	protected void addChild( Constituent aArgumentizer ) {
		_children.add( aArgumentizer );
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <V> V toValue( String aAlias ) {
		Object eValue;
		for ( Constituent eElement : _children ) {
			eValue = eElement.toValue( aAlias );
			if ( eValue != null ) {
				return (V) eValue;
			}
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends Operand<?>> T toOperand( String aAlias, Class<T> aType ) {
		Operand<?> eOperand;
		Condition eCondition;
		boolean eTypeMatch, eAliasMatch;
		for ( Constituent eSyntaxable : getChildren() ) {

			if ( eSyntaxable instanceof Operand ) {
				eOperand = (Operand<?>) eSyntaxable;
				eTypeMatch = true;
				eAliasMatch = true;
				if ( aAlias != null && aAlias.length() != 0 ) {
					eAliasMatch = aAlias.equals( eOperand.getAlias() );
				}
				if ( aType != null ) {
					eTypeMatch = aType.isAssignableFrom( eOperand.getClass() );
				}
				if ( eTypeMatch && eAliasMatch ) {
					return (T) eOperand;
				}
			}
			else if ( eSyntaxable instanceof Condition ) {
				eCondition = (Condition) eSyntaxable;
				eOperand = eCondition.toOperand( aAlias, aType );
				if ( eOperand != null ) {
					return (T) eOperand;
				}
			}
		}
		return null;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
