// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.data.ArgsPrefix;
import org.refcodes.struct.Relation;
import org.refcodes.textual.VerboseTextBuilder;

/**
 * The {@link AbstractOperand} is an abstract implementation of an
 * {@link Operand} providing the boiler plate when implementing the
 * {@link Operand} interface.
 *
 * @param <T> the generic type
 */
public abstract class AbstractOperand<T> extends AbstractConstituent implements Operand<T> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Class<T> _type;
	private String _alias;
	private String[] _args = null;
	protected T _value = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link Operand} with the given arguments.
	 * 
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aAlias The {@link Operand}'s name, used for syntax creation.
	 * @param aDescription The description to be used (without any line breaks).
	 */
	public AbstractOperand( Class<T> aType, String aAlias, String aDescription ) {
		super( aDescription );
		_type = aType;
		_alias = aAlias;

	}

	/**
	 * Instantiates a new operand.
	 * 
	 * @param aProperty The key (= alias) and the value for the operand.
	 * @param aType the type
	 */
	public AbstractOperand( Relation<String, T> aProperty, Class<T> aType ) {
		_alias = aProperty.getKey();
		_value = aProperty.getValue();
		_type = aType;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<T>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliContext ) throws ArgsSyntaxException {
		int first = -1;
		for ( int i = aArgs.length - 1; i >= 0; i-- ) {
			if ( !aCliContext.isOption( aArgs[i] ) ) {
				if ( first == -1 || first + -1 == i ) {
					first = i;
				}
			}
		}
		if ( first != -1 ) {
			List<Operand<T>> theList = new ArrayList<Operand<T>>();
			theList.add( this );
			setValue( toType( aArgs[first] ) );
			setParsedArgs( new String[] { aArgs[first] } );
			return theList;
		}

		throw new UnknownArgsException( aArgs, "Unable to parse any args (not being prefixed with " + new VerboseTextBuilder().withElements( ArgsPrefix.toPrefixes() ).toString() + ")." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( CliContext aCliCtx ) {
		return aCliCtx.toArgumentSpec( this );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getAlias() {
		return _alias;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Class<T> getType() {
		return _type;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public T getValue() {
		return _value;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] toParsedArgs() {
		return _args;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		_args = null;
		_value = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toSchema().toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int compareTo( Operand<?> obj ) {
		if ( obj instanceof Option<?> aOption ) {
			if ( this instanceof Option<?> theOption ) {
				String otherOption = aOption.getShortOption() != null ? aOption.getShortOption().toString() : aOption.getLongOption();
				String thisOption = theOption.getShortOption() != null ? theOption.getShortOption().toString() : theOption.getLongOption();
				return thisOption.compareTo( otherOption );
			}
			if ( this instanceof Operand<?> ) {
				return 1;
			}
		}
		if ( obj instanceof Operand<?> ) {
			if ( this instanceof Option<?> ) {
				return -11;
			}
		}
		if ( getAlias() != null && obj.getAlias() != null ) {
			return getAlias().compareTo( obj.getAlias() );
		}
		return toSyntax().compareTo( obj.toSyntax() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CliSchema toSchema() {
		return new CliSchema( _alias, getClass(), _value, _description );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sets the value for the {@link Operand} as parsed by the
	 * {@link #parseArgs(String[], String[], CliContext)} method.
	 * 
	 * @param aValue The value to be set for this {@link Operand}.
	 */
	protected void setValue( T aValue ) {
		_value = aValue;
	}

	/**
	 * Sets the command line argument(s) representing the {@link Operand} and
	 * its value as parsed by the
	 * {@link #parseArgs(String[], String[], CliContext)} method.
	 * 
	 * @param aArgs The command line arguments representing this {@link Operand}
	 *        with its value.
	 */
	protected void setParsedArgs( String[] aArgs ) {
		_args = aArgs;
	}

	/**
	 * Double dispatch hook to be implemented by subclasses of the
	 * {@link AbstractOperand} for converting a command line argument to the
	 * required {@link Operand}'s type. In case conversion failed, then an
	 * according exception is to be thrown.
	 * 
	 * @param aArg The command line argument to be converted to an instance of
	 *        the given type T.
	 * 
	 * @return An instance of type T from the provided command line argument.
	 * 
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type.
	 */
	abstract protected T toType( String aArg ) throws ParseArgsException;

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
