// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.refcodes.struct.Relation;
import org.refcodes.struct.RelationImpl;

/**
 * The {@link AbstractOption} is an abstract implementation of an {@link Option}
 * providing the boiler plate when implementing the {@link Option} interface.
 *
 * @param <T> the generic type
 */
public abstract class AbstractOption<T> extends AbstractOperand<T> implements Option<T> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String SHORT_OPTION_KEY = "SHORT_OPTION";
	private static final String LONG_OPTION_KEY = "LONG_OPTION";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Character _shortOption;
	private String _longOption;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs an {@link AbstractOption} with the given arguments.
	 *
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aDescription The description to be used (without any line breaks).
	 */
	public AbstractOption( String aLongOption, Class<T> aType, String aDescription ) {
		this( null, aLongOption, aType, null, aDescription );
	}

	/**
	 * Constructs an {@link AbstractOption} with the given arguments.
	 *
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aAlias The option argument's name to be used when constructing the
	 *        syntax.
	 * @param aDescription The description to be used (without any line breaks).
	 */
	public AbstractOption( String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		this( null, aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * Constructs an {@link AbstractOption} with the given arguments.
	 * 
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aDescription The description to be used (without any line breaks).
	 */
	protected AbstractOption( Character aShortOption, String aLongOption, Class<T> aType, String aDescription ) {
		this( aShortOption, aLongOption, aType, null, aDescription );
	}

	/**
	 * Constructs an {@link AbstractOption} with the given arguments.
	 * 
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aAlias The option argument's name to be used when constructing the
	 *        syntax.
	 * @param aDescription The description to be used (without any line breaks).
	 */
	public AbstractOption( Character aShortOption, String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		super( aType, aAlias != null ? aAlias : (aLongOption != null ? aLongOption : (aShortOption != null ? aShortOption.toString() : null)), aDescription );
		if ( aShortOption == null && (aLongOption == null || aLongOption.length() == 0) ) {
			throw new IllegalArgumentException( "At least the short option <" + (aShortOption != null ? "'" + aShortOption + "'" : aShortOption) + "> must not be null or the long option <" + (aLongOption != null ? "\"" + aLongOption + "\"" : aLongOption) + "> must not be empty!" );
		}
		_shortOption = aShortOption;
		_longOption = aLongOption;
	}

	/**
	 * Constructs an {@link AbstractOption} with the given arguments.
	 *
	 * @param aProperty The key (= alias) and the value for the operand.
	 * @param aType the type
	 */
	public AbstractOption( Relation<String, T> aProperty, Class<T> aType ) {
		super( aProperty, aType );
		_shortOption = aProperty.getKey().length() == 1 ? aProperty.getKey().charAt( 0 ) : null;
		_longOption = aProperty.getKey().length() > 1 ? aProperty.getKey() : null;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<T>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {
		Relation<String, String> theOptionArgument = toOptionArgument( this, aArgs, aOptions, aCliCtx );
		if ( theOptionArgument != null ) {
			List<Operand<T>> theList = new ArrayList<Operand<T>>();
			theList.add( this );
			setParsedArgs( new String[] { theOptionArgument.getKey(), theOptionArgument.getValue() } );
			setValue( toType( theOptionArgument.getValue() ) );
			return theList;
		}
		if ( contains( aArgs, aCliCtx.toShortOption( this ) ) ) {
			throw new ParseArgsException( aArgs, "Missing value; the short-option \"" + aCliCtx.toShortOption( this ) + "\" requires a value." );
		}
		if ( contains( aArgs, aCliCtx.toLongOption( this ) ) ) {
			throw new ParseArgsException( aArgs, "Missing value; the long-option \"" + aCliCtx.toLongOption( this ) + "\" requires a value." );
		}
		throw new UnknownArgsException( aArgs, "Neither the short-option \"" + aCliCtx.toShortOption( this ) + "\" nor the long-option \"" + aCliCtx.toLongOption( this ) + "\"  was found in the command line arguments, at least one of them must be specified." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Character getShortOption() {
		return _shortOption;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLongOption() {
		return _longOption;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( CliContext aCliCtx ) {
		return aCliCtx.toOptionEscapeCode() + aCliCtx.toOption( this ) + aCliCtx.toResetEscapeCode() + (getAlias() != null ? (" " + aCliCtx.toArgumentSpec( this )) : "") + aCliCtx.toResetEscapeCode();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CliSchema toSchema() {
		CliSchema theSchema = super.toSchema();
		theSchema.put( SHORT_OPTION_KEY, _shortOption );
		theSchema.put( LONG_OPTION_KEY, _longOption );
		return theSchema;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void setValue( T aValue ) {
		super.setValue( aValue );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void setParsedArgs( String[] aArgs ) {
		super.setParsedArgs( aArgs );
	}

	/**
	 * Determines whether a given {@link String} is contained in the given
	 * {@link String} array.
	 * 
	 * @param aArgs The {@link String} array to be tested whether it contains
	 *        the given {@link String}.
	 * @param aArg The {@link String} to be tested whether it is contained in
	 *        the provided {@link String} array.
	 * 
	 * @return True in case the {@link String} is contained in the array, else
	 *         false.
	 */
	protected static boolean contains( String[] aArgs, String aArg ) {
		List<String> theList = new ArrayList<String>( Arrays.asList( aArgs ) );
		return theList.contains( aArg );
	}

	/**
	 * Takes the {@link Option}'s short-option and long-option and tries to
	 * determine that {@link Option}'s value in the provided command line
	 * arguments. Depending on whether the short-option or the long-option was
	 * detected with a value, the result contains the according option as the
	 * key with the detected value in the {@link Relation} instance. Null is
	 * returned when either no option was found or no value for one of the
	 * options.
	 * 
	 * @param aOption The option for which to get the value
	 * @param aArgs The command line arguments from which to determine the
	 *        {@link Option}'s value.
	 * @param aOptions The list of options (short and well as long) which are
	 *        reserved and cannot be used as value.
	 * @param aCliCtx The {@link CliContext} to run this method with.
	 * 
	 * @return A key/value-pair containing the detected (short / long) option
	 *         alongside the detected value.
	 */
	protected static Relation<String, String> toOptionArgument( Option<?> aOption, String[] aArgs, String[] aOptions, CliContext aCliCtx ) {
		if ( aArgs.length < 2 ) {
			return null;
		}
		Relation<String, String> theAttribute = toOptionArgument( aArgs, aCliCtx.toShortOption( aOption ), aOptions );
		if ( theAttribute != null ) {
			return theAttribute;
		}
		theAttribute = toOptionArgument( aArgs, aCliCtx.toLongOption( aOption ), aOptions );
		if ( theAttribute != null ) {
			return theAttribute;
		}
		return null;
	}

	/**
	 * Takes the {@link Option}'s short-option and long-option and tries to
	 * determine that {@link Option}'s value in the provided command line
	 * arguments. Depending on whether the short-option or the long-option was
	 * detected with a value, the result contains the according option as the
	 * key with the detected value in the {@link Relation} instance. Null is
	 * returned when either no option was found or no value for one of the
	 * options.
	 *
	 * @param aArgs The command line arguments from which to determine the
	 *        {@link Option}'s value.
	 * @param aOption The option for which to get the value
	 * @param aOptions The list of options (short and well as long) which are
	 *        reserved and cannot be used as value.
	 * 
	 * @return A key/value-pair containing the detected (short / long) option
	 *         alongside the detected value.
	 */
	protected static Relation<String, String> toOptionArgument( String[] aArgs, String aOption, String[] aOptions ) {
		String eArg;
		String eOptArg;
		for ( int i = 0; i < aArgs.length - 1; i++ ) {
			eArg = aArgs[i];
			if ( eArg.equals( aOption ) ) {
				eOptArg = aArgs[i + 1];
				for ( String eOption : aOptions ) {
					if ( eOptArg.equalsIgnoreCase( eOption ) ) {
						return null;
					}
				}
				return new RelationImpl<String, String>( aOption, eOptArg );
			}
		}
		return null;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
