// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.List;

import org.refcodes.textual.VerboseTextBuilder;

/**
 * The {@link AllCondition} enforces that the encapsulated {@link Constituent}
 * consumes all arguments passed via invoking its
 * {@link Constituent#parseArgs(String[], CliContext)}, else a
 * {@link SuperfluousArgsException} is thrown. This enables the construction of
 * a syntax which otherwise would not be unambiguous when encapsulated within an
 * {@link XorCondition}.
 */
public class AllCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link AllCondition} with the {@link Constituent}
	 * ({@link Condition}) instance to be nested.
	 *
	 * @param aArg The {@link Constituent} ({@link Condition}) instance to be
	 *        nested
	 */
	public AllCondition( Constituent aArg ) {
		super( "All (ALL) arguments passed are to be consumed by the nested syntaxables.", aArg );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {
		List<? extends Operand<?>> theResult = getFirst().parseArgs( aArgs, aOptions, aCliCtx );
		String[] theRemainderArgs = CliUtility.toArgsDiff( aArgs, theResult );
		if ( theRemainderArgs != null && theRemainderArgs.length != 0 ) {
			throw new SuperfluousArgsException( theRemainderArgs, "There were command line arguments (" + VerboseTextBuilder.asString( theRemainderArgs ) + ") " );
		}
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( CliContext aCliCtx ) {
		if ( getChildren() != null && getChildren().size() > 0 ) {
			String theSyntax = "";
			for ( Constituent eChild : getChildren() ) {
				if ( theSyntax.length() != 0 ) {
					theSyntax += " ";
				}
				if ( aCliCtx.getSyntaxMetrics().getAllSymbol() != null && aCliCtx.getSyntaxMetrics().getAllSymbol().length() != 0 ) {
					theSyntax += aCliCtx.getSyntaxMetrics().getAllSymbol() + aCliCtx.getSyntaxMetrics().getBeginListSymbol() + " ";
				}
				theSyntax += eChild.toSyntax( aCliCtx );
				if ( aCliCtx.getSyntaxMetrics().getAllSymbol() != null && aCliCtx.getSyntaxMetrics().getAllSymbol().length() != 0 ) {
					theSyntax += " " + aCliCtx.getSyntaxMetrics().getEndListSymbol();
				}
			}
			return theSyntax;
		}
		return "";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( CliContext aCliCtx ) {
		if ( getChildren() != null && getChildren().size() > 0 ) {
			if ( getChildren().size() == 1 ) {
				return getChildren().get( 0 ).toSynopsis( aCliCtx );
			}
			String theSynopsis = "";
			for ( Constituent eChild : getChildren() ) {
				if ( theSynopsis.length() != 0 ) {
					theSynopsis += " ";
				}
				theSynopsis += eChild.toSyntax( aCliCtx );
			}
			return theSynopsis;
		}
		return "";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		getChildren().get( 0 ).reset();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toSchema().toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
