// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

/**
 * An {@link AndCondition} contains (represents) a list of {@link Condition}
 * ({@link Constituent}) instances (nested by the {@link AndCondition}) of which
 * all are to be parsed successfully as of invoking the
 * {@link Condition#parseArgs(String[], String[], CliContext)} methods. The
 * command line arguments syntax <code>-a &amp; -b &amp; -c</code> specifies
 * that all "-a", "-b" and "-c" must be set. In case at least one is not set,
 * then the {@link AndCondition} will terminate the
 * {@link #parseArgs(String[], String[], CliContext)} method with an exception.
 * If one argument is optional, then consider nesting it in an
 * {@link AnyCondition}.
 */
public class AndCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link AndCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested.
	 */
	public AndCondition( Constituent... aArgs ) {
		super( "All (AND) nested syntaxables must match from the arguments.", aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {
		List<Operand<?>> theResult = new ArrayList<Operand<?>>();
		List<Operand<?>> eOperands;
		for ( Constituent eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions, aCliCtx );
			}
			catch ( UnknownArgsException exc ) {
				throw new UnknownArgsException( exc.toParsedArgs(), "At least one mandatory syntax did not match the provided command line arguments, though all mandatories must match.", exc );
			}
			catch ( AmbiguousArgsException exc ) {
				throw new AmbiguousArgsException( exc.toParsedArgs(), "At least one mandatory syntax did not match the provided command line arguments, though all mandatories must match.", exc );
			}
			if ( eOperands != null && eOperands.size() != 0 ) {
				theResult.addAll( eOperands );
			}
			aArgs = CliUtility.toArgsDiff( aArgs, eOperands );
		}
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( CliContext aCliCtx ) {
		return toSynopsis( aCliCtx );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( CliContext aCliCtx ) {
		String theSynopsis = "";
		for ( Constituent eChild : getChildren() ) {
			if ( theSynopsis.length() != 0 ) {
				theSynopsis += aCliCtx.getSyntaxMetrics().getAndSymbol() != null && aCliCtx.getSyntaxMetrics().getAndSymbol().length() != 0 ? " " + aCliCtx.getSyntaxMetrics().getAndSymbol() + " " : " ";
			}
			theSynopsis += eChild.toSyntax( aCliCtx );
		}
		return theSynopsis;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
