// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

/**
 * Any of the nested {@link Condition} conditions may match for the
 * {@link AnyCondition} to match, e.g. all of the nested conditions are
 * optional. The {@link AnyCondition} can nested by an {@link AndCondition} and
 * encapsulates a {@link Condition} which may be provided but does not
 * necessarily needs to be provided in order for the {@link AndCondition} to
 * successfully parse the nested {@link Condition} items including the
 * {@link AnyCondition}.
 */
public class AnyCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link AnyCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested
	 */
	public AnyCondition( Constituent... aArgs ) {
		super( "Any (OPTIONAL) nested syntaxables optionally matches from the arguments.", aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {
		List<Operand<?>> theResult = new ArrayList<Operand<?>>();
		List<Operand<?>> eOperands = null;
		for ( Constituent eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions, aCliCtx );
				if ( eOperands != null ) {
					theResult.addAll( eOperands );
				}
				aArgs = CliUtility.toArgsDiff( aArgs, eOperands );
			}
			catch ( UnknownArgsException ignore ) {
				/* optional! */
			}
			// catch ( AmbiguousArgsException exc ) { /* optional! */ }
		}
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( CliContext aCliCtx ) {
		if ( getChildren() != null && getChildren().size() > 0 ) {
			String theSyntax = "";
			for ( Constituent eChild : getChildren() ) {
				if ( theSyntax.length() != 0 ) {
					theSyntax += " ";
				}
				theSyntax += aCliCtx.getSyntaxMetrics().getAnySymbol() != null && aCliCtx.getSyntaxMetrics().getAnySymbol().length() != 0 ? aCliCtx.getSyntaxMetrics().getAnySymbol() : "";
				theSyntax += aCliCtx.getSyntaxMetrics().getBeginOptionalSymbol() != null && aCliCtx.getSyntaxMetrics().getBeginOptionalSymbol().length() != 0 ? aCliCtx.getSyntaxMetrics().getBeginOptionalSymbol() + " " : "";
				theSyntax += eChild.toSyntax( aCliCtx );
				theSyntax += aCliCtx.getSyntaxMetrics().getEndOptionalSymbol() != null && aCliCtx.getSyntaxMetrics().getEndOptionalSymbol().length() != 0 ? " " + aCliCtx.getSyntaxMetrics().getEndOptionalSymbol() : "";
			}
			return theSyntax;
		}
		return "";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		getChildren().get( 0 ).reset();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toSchema().toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
