// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

/**
 * Provides an accessor for a {@link ArgsParser} property.
 */
public interface ArgsParserAccessor {

	/**
	 * Retrieves the args-parser from the {@link ArgsParser} property.
	 * 
	 * @return The args-parser stored by the {@link ArgsParser} property.
	 */
	ArgsParser getArgsParser();

	/**
	 * Provides a mutator for a {@link ArgsParser} property.
	 */
	public interface ArgsParserMutator {

		/**
		 * Sets the args-parser for the {@link ArgsParser} property.
		 * 
		 * @param aProperties The args-parser to be stored by the
		 *        {@link ArgsParser} property.
		 */
		void setArgsParser( ArgsParser aProperties );
	}

	/**
	 * Provides a builder method for a {@link ArgsParser} property returning the
	 * builder for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface ArgsParserBuilder<B extends ArgsParserBuilder<B>> {

		/**
		 * Sets the args-parser for the {@link ArgsParser} property.
		 * 
		 * @param aProperties The args-parser to be stored by the
		 *        {@link ArgsParser} property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withArgsParser( ArgsParser aProperties );
	}

	/**
	 * Provides a {@link ArgsParser} property.
	 */
	public interface ArgsParserProperty extends ArgsParserAccessor, ArgsParserMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link ArgsParser}
		 * (setter) as of {@link #setArgsParser(ArgsParser)} and returns the
		 * very same value (getter).
		 * 
		 * @param aArgsParser The {@link ArgsParser} to set (via
		 *        {@link #setArgsParser(ArgsParser)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default ArgsParser letArgsParser( ArgsParser aArgsParser ) {
			setArgsParser( aArgsParser );
			return aArgsParser;
		}
	}
}
