// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.refcodes.data.AnsiEscapeCode;
import org.refcodes.data.ArgsPrefix;
import org.refcodes.data.AsciiColorPalette;
import org.refcodes.data.ConsoleDimension;
import org.refcodes.data.Delimiter;
import org.refcodes.data.License;
import org.refcodes.data.Literal;
import org.refcodes.data.SystemProperty;
import org.refcodes.graphical.BoxBorderMode;
import org.refcodes.runtime.RuntimeUtility;
import org.refcodes.runtime.Terminal;
import org.refcodes.struct.PropertyImpl;
import org.refcodes.struct.RelationImpl;
import org.refcodes.textual.AsciiArtBuilder;
import org.refcodes.textual.AsciiArtMode;
import org.refcodes.textual.ColumnWidthType;
import org.refcodes.textual.Font;
import org.refcodes.textual.FontFamily;
import org.refcodes.textual.FontStyle;
import org.refcodes.textual.HorizAlignTextBuilder;
import org.refcodes.textual.HorizAlignTextMode;
import org.refcodes.textual.PixmapRatioMode;
import org.refcodes.textual.SplitTextMode;
import org.refcodes.textual.TableBuilder;
import org.refcodes.textual.TableStyle;
import org.refcodes.textual.TextBlockBuilder;
import org.refcodes.textual.TextBorderBuilder;
import org.refcodes.textual.TextBoxGrid;
import org.refcodes.textual.TextBoxStyle;
import org.refcodes.textual.TextLineBuilder;
import org.refcodes.textual.VerboseTextBuilder;

/**
 * A straightforward implementation of the {@link ArgsParser} interface. The
 * constructor only provides means to set the required attributes as the
 * attributes to be adjusted optionally are already sufficiently preconfigured.
 * For adjusting them, a flavor of the Builder-Pattern is provided with which
 * you can easily chain the configuration of this instance; as them methods
 * return the instance of this class being configured. This helps to prevent the
 * telescoping constructor anti-pattern.
 * <p>
 * The {@link SyntaxNotation} is pre-set with the {@link SyntaxNotation#LOGICAL}
 * notation.
 * <p>
 * The console width id preconfigured with the console's width as determined by
 * the <code>SystemUtility.getTerminalWidth()</code>.
 * <p>
 * The standard out {@link PrintStream} is preconfigured with the
 * {@link System#out} {@link PrintStream}.
 * <p>
 * The newline characters to be used for line breaks is "\r\n" on Windows
 * machines and "\"n" on all other machines as of the
 * <code>SystemUtility.getLineBreak()</code>.
 *
 * @see "http://en.wikipedia.org/wiki/Builder_pattern"
 */

public class ArgsParserImpl implements ArgsParser {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static Logger LOGGER = Logger.getLogger( ArgsParserImpl.class.getName() );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Character _shortOptionPrefix = SyntaxNotation.DEFAULT.getShortOptionPrefix();
	private String _longOptionPrefix = SyntaxNotation.DEFAULT.getLongOptionPrefix();
	private List<Example> _examples = null;
	private Font _bannerFont = new Font( FontFamily.DIALOG, FontStyle.PLAIN, 12 );
	private char[] _bannerFontPalette = AsciiColorPalette.HALFTONE_GRAY.getPalette();
	private Constituent _argsSyntax;
	private SyntaxMetrics _syntaxMetrics = SyntaxNotation.DEFAULT;
	private int _consoleWidth = Terminal.toPreferredTerminalWidth();
	private int _maxConsoleWidth = -1;
	private String _lineBreak = Terminal.getLineBreak();
	private String _title = null;
	private String _name = "foobar";
	private String _description = "See the syntax declaration for usage, see the descriptions for the short- and the long-options. Option arguments are noted in angle brackets.";
	private String _licenseNote = License.LICENSE_NOTE.getText();
	private String _copyrightNote = License.COPYRIGHT_NOTE.getText();;
	private boolean _isEscapeCodesEnabled = Terminal.isAnsiTerminalPreferred();
	private String _resetEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.RESET );
	private String _commandEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.ITALIC );
	private String _argumentEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.FAINT, AnsiEscapeCode.ITALIC );
	private String _optionEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.BOLD );
	private String _descriptionEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR );
	private String _bannerEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.BOLD );
	private String _bannerBorderEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.FAINT );
	private String _lineSeparatorEscapeCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.FAINT );
	private TextBoxGrid _textBoxGrid = TextBoxStyle.toRuntimeTextBoxGrid();
	private char _separatorLnChar = _textBoxGrid.getInnerLine();
	private boolean _hasOverrideSeparatorLnChar = false;
	protected PrintStream _stdStream = RuntimeUtility.toSystemOut();
	protected PrintStream _errStream = RuntimeUtility.toSystemErr();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the {@link ArgsParser} instance without any restrictions to
	 * the parsed arguments. The constructor only provides means to set the
	 * required attributes as the attributes to be adjusted optionally are
	 * already sufficiently preconfigured. For adjusting them, a flavor of the
	 * Builder-Pattern is provided with which you can easily chain the
	 * configuration of this instance; as them methods return the instance of
	 * this class being configured.
	 */
	public ArgsParserImpl() {
		this( (Constituent) null );
	}

	/**
	 * Constructs the {@link ArgsParser} instance without any restrictions to
	 * the parsed arguments. The constructor only provides means to set the
	 * required attributes as the attributes to be adjusted optionally are
	 * already sufficiently preconfigured. For adjusting them, a flavor of the
	 * Builder-Pattern is provided with which you can easily chain the
	 * configuration of this instance; as them methods return the instance of
	 * this class being configured.
	 *
	 * @param aCliCtx The {@link CliContext} to be used for initializing.
	 */
	public ArgsParserImpl( CliContext aCliCtx ) {
		this( null, aCliCtx );
	}

	/**
	 * Constructs the {@link ArgsParser} instance with the given root
	 * {@link Condition} and the default {@link SyntaxNotation#LOGICAL}. The
	 * constructor only provides means to set the required attributes as the
	 * attributes to be adjusted optionally are already sufficiently
	 * preconfigured. For adjusting them, a flavor of the Builder-Pattern is
	 * provided with which you can easily chain the configuration of this
	 * instance; as them methods return the instance of this class being
	 * configured.
	 * 
	 * @param aArgsSyntax The args syntax root {@link Constituent} node being
	 *        the node from which parsing the command line arguments starts.
	 */
	public ArgsParserImpl( Constituent aArgsSyntax ) {
		_argsSyntax = aArgsSyntax;
	}

	/**
	 * Constructs the {@link ArgsParser} instance with the given root
	 * {@link Condition} and the default {@link SyntaxNotation#LOGICAL}. The
	 * constructor only provides means to set the required attributes as the
	 * attributes to be adjusted optionally are already sufficiently
	 * preconfigured. For adjusting them, a flavor of the Builder-Pattern is
	 * provided with which you can easily chain the configuration of this
	 * instance; as them methods return the instance of this class being
	 * configured.
	 *
	 * @param aArgsSyntax The args syntax root {@link Constituent} node being
	 *        the node from which parsing the command line arguments starts.
	 * @param aCliCtx The {@link CliContext} to be used for initializing.
	 */
	public ArgsParserImpl( Constituent aArgsSyntax, CliContext aCliCtx ) {
		_argsSyntax = aArgsSyntax;
		_syntaxMetrics = aCliCtx.getSyntaxMetrics();
		_argumentEscapeCode = aCliCtx.getArgumentEscapeCode();
		_optionEscapeCode = aCliCtx.getOptionEscapeCode();
		_resetEscapeCode = aCliCtx.getResetEscapeCode();
		_isEscapeCodesEnabled = aCliCtx.isEscapeCodesEnabled();
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addExample( Example aExampleUsage ) {
		if ( _examples == null ) {
			synchronized ( this ) {
				if ( _examples == null ) {
					_examples = new ArrayList<Example>();
				}
			}
		}
		_examples.add( aExampleUsage );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void errorLn( String aLine ) {
		if ( aLine == null ) aLine = "null";
		String[] theLines = new TextBlockBuilder().withText( aLine ).withColumnWidth( _consoleWidth ).withSplitTextMode( SplitTextMode.AT_SPACE ).toStrings();
		_errStream.print( fromTextBlock( theLines, toLineBreak() ) );
		_errStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<? extends Operand<?>> evalArgs( String[] aArgs ) throws ArgsSyntaxException {
		// Any command line filtering system property set? |-->
		String theFilterProperty = SystemProperty.ARGS_FILTER.getValue();
		if ( theFilterProperty != null && theFilterProperty.length() != 0 ) {
			ArgsFilter theArgsFilter = ArgsFilter.toArgsFilter( theFilterProperty );
			if ( theArgsFilter == null ) {
				LOGGER.log( Level.WARNING, "The provided system property <" + SystemProperty.ARGS_FILTER.getKey() + "> with value <" + SystemProperty.ARGS_FILTER.getValue() + "> does not resolve to a valid args filter! Valid args filter values are: " + VerboseTextBuilder.asString( ArgsFilter.values() ) );
			}
			else {
				aArgs = theArgsFilter.toFiltered( aArgs );
			}
		}
		// Any command line filtering system property set? <--|
		if ( _argsSyntax == null ) {
			return fromArgs( aArgs, getDelimiter() );
		}
		else {
			CliContext theCliCtx = toCliContext();
			aArgs = theCliCtx.toExpandOptions( aArgs );
			List<? extends Operand<?>> theOperands = _argsSyntax.parseArgs( aArgs, theCliCtx );
			String[] theSuperflousArgs = CliUtility.toArgsDiff( aArgs, theOperands );
			if ( theSuperflousArgs != null && theSuperflousArgs.length > 0 ) {
				throw new SuperfluousArgsException( theSuperflousArgs, "Superfluous command arguments " + new VerboseTextBuilder().withElements( theSuperflousArgs ).toString() + " were provided but cannot be evaluated or are not supported as of the given combination of arguments." );
			}
			return theOperands;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Constituent getArgsSyntax() {
		return _argsSyntax;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getArgumentEscapeCode() {
		return _argumentEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBannerBorderEscapeCode() {
		return _bannerBorderEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBannerEscapeCode() {
		return _bannerEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Font getBannerFont() {
		return _bannerFont;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public char[] getBannerFontPalette() {
		return _bannerFontPalette;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getCommandEscapeCode() {
		return _commandEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getConsoleWidth() {
		return _consoleWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getCopyright() {
		return _copyrightNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return _description;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescriptionEscapeCode() {
		return _descriptionEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Example[] getExamples() {
		return _examples != null ? _examples.toArray( new Example[_examples.size()] ) : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLicense() {
		return _licenseNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLineBreak() {
		return _lineBreak;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLineSeparatorEscapeCode() {
		return _lineSeparatorEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLongOptionPrefix() {
		return _longOptionPrefix;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getMaxConsoleWidth() {
		return _maxConsoleWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return _name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getOptionEscapeCode() {
		return _optionEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResetEscapeCode() {
		return _resetEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public char getSeparatorLnChar() {
		return _separatorLnChar;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Character getShortOptionPrefix() {
		return _shortOptionPrefix;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public SyntaxMetrics getSyntaxMetrics() {
		return _syntaxMetrics;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextBoxGrid getTextBoxGrid() {
		return _textBoxGrid;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTitle() {
		return _title;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEscapeCodesEnabled() {
		return _isEscapeCodesEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printBanner() {
		int theBannerWidth = _consoleWidth - 4;
		String[] theCanvas = new AsciiArtBuilder().withText( _title != null ? _title : _name ).withFont( _bannerFont ).withAsciiColors( _bannerFontPalette ).withColumnWidth( theBannerWidth ).withAsciiArtMode( AsciiArtMode.NORMAL ).withPixmapRatioMode( PixmapRatioMode.ONE_HALF ).toStrings();
		boolean hasBorder = hasBorder( theCanvas );
		if ( hasBorder ) {
			theBannerWidth = _consoleWidth - 2;
		}
		theCanvas = new AsciiArtBuilder().withText( _title != null ? _title : _name ).withFont( _bannerFont ).withAsciiColors( _bannerFontPalette ).withColumnWidth( theBannerWidth ).withAsciiArtMode( AsciiArtMode.NORMAL ).withPixmapRatioMode( PixmapRatioMode.ONE_HALF ).toStrings();
		theCanvas = new HorizAlignTextBuilder().withHorizAlignTextMode( HorizAlignTextMode.CENTER ).withText( theCanvas ).withColumnWidth( theBannerWidth ).withFillChar( ' ' ).toStrings();
		if ( !hasBorder ) {
			theCanvas = new TextBorderBuilder().withBoxBorderMode( BoxBorderMode.ALL ).withText( theCanvas ).withBorderWidth( 1 ).withBorderChar( ' ' ).toStrings();
		}
		theCanvas = new TextBorderBuilder().withTextBoxGrid( _textBoxGrid ).withText( theCanvas ).withBoxBorderMode( BoxBorderMode.ALL ).toStrings();

		if ( _isEscapeCodesEnabled ) {
			theCanvas[0] = _bannerBorderEscapeCode + theCanvas[0] + _resetEscapeCode;
			if ( theCanvas.length > 1 ) {
				for ( int i = 1; i < theCanvas.length - 1; i++ ) {
					theCanvas[i] = _bannerBorderEscapeCode + theCanvas[i].substring( 0, 1 ) + _resetEscapeCode + _bannerEscapeCode + theCanvas[i].substring( 1, theCanvas[i].length() - 1 ) + _resetEscapeCode + _bannerBorderEscapeCode + theCanvas[i].substring( theCanvas[i].length() - 1 ) + _resetEscapeCode;
				}
			}
			theCanvas[theCanvas.length - 1] = _bannerBorderEscapeCode + theCanvas[theCanvas.length - 1] + _resetEscapeCode;
		}
		_stdStream.print( fromTextBlock( theCanvas, toLineBreak() ) );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printCopyright() {
		_stdStream.println( _copyrightNote );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printDescription() {
		_stdStream.println( (_isEscapeCodesEnabled ? _descriptionEscapeCode : "") + _description + (_isEscapeCodesEnabled ? _resetEscapeCode : "") );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printExamples() {
		if ( _examples != null && _examples.size() != 0 ) {
			int theMaxLength = 0;
			String eDescription;
			for ( Example eUsage : _examples ) {
				eDescription = eUsage._description;
				if ( eDescription.length() > theMaxLength ) theMaxLength = eDescription.length();
			}
			if ( theMaxLength > _consoleWidth / 2 ) {
				theMaxLength = _consoleWidth / 2;
				if ( theMaxLength < ConsoleDimension.MIN_WIDTH.getValue() / 2 ) {
					theMaxLength = ConsoleDimension.MIN_WIDTH.getValue() / 2;
				}
			}

			TableBuilder theTable = new TableBuilder().withTableStyle( TableStyle.BLANK_HEADER_BLANK_BODY ).withRowWidth( _consoleWidth ).withPrintStream( _stdStream ).withEscapeCodesEnabled( _isEscapeCodesEnabled ).withLeftBorder( false ).withRightBorder( false ).withDividerLine( false );
			if ( _isEscapeCodesEnabled ) {
				theTable.setResetEscapeCode( _resetEscapeCode );
			}
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withColumnWidth( theMaxLength, ColumnWidthType.ABSOLUTE );
			if ( _isEscapeCodesEnabled ) {
				theTable.withRowColumnEscapeCode( _descriptionEscapeCode );
			}
			theTable.addColumn().withColumnWidth( 2, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( _name.length(), ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( 1, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.LEFT ).withColumnSplitTextMode( SplitTextMode.AT_SPACE );
			theTable.withLineBreak( toLineBreak() );

			String theCommand = (_isEscapeCodesEnabled ? _commandEscapeCode : "") + _name + (_isEscapeCodesEnabled ? _resetEscapeCode : "");
			for ( Example eUsage : _examples ) {
				String eArgs = "";
				String eOpt;
				for ( Operand<?> eOperand : eUsage.getOperands() ) {
					if ( eArgs.length() != 0 ) eArgs += " ";
					eOpt = toCliContext().toSpec( eOperand );
					eArgs += eOpt;
				}
				theTable.printRowContinue( eUsage.getDescription(), ": ", theCommand, " ", eArgs );
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printHeader() {
		int theBannerWidth = _consoleWidth - 4;
		String[] theCanvas = AsciiArtBuilder.asSimpleBanner( _title != null ? _title : _name, theBannerWidth, AsciiArtMode.NORMAL, _bannerFontPalette );
		boolean hasBorder = hasBorder( theCanvas );
		if ( hasBorder ) {
			theBannerWidth = _consoleWidth - 2;
		}
		theCanvas = AsciiArtBuilder.asSimpleBanner( _title != null ? _title : _name, theBannerWidth, AsciiArtMode.NORMAL, _bannerFontPalette );
		theCanvas = new HorizAlignTextBuilder().withHorizAlignTextMode( HorizAlignTextMode.CENTER ).withText( theCanvas ).withColumnWidth( theBannerWidth ).withFillChar( ' ' ).toStrings();
		if ( !hasBorder ) {
			theCanvas = new TextBorderBuilder().withBoxBorderMode( BoxBorderMode.ALL ).withText( theCanvas ).withBorderWidth( 1 ).withBorderChar( ' ' ).toStrings();
		}
		theCanvas = new TextBorderBuilder().withTextBoxGrid( _textBoxGrid ).withText( theCanvas ).withBoxBorderMode( BoxBorderMode.ALL ).toStrings();

		if ( _isEscapeCodesEnabled ) {
			theCanvas[0] = _bannerBorderEscapeCode + theCanvas[0] + _resetEscapeCode;
			if ( theCanvas.length > 1 ) {
				for ( int i = 1; i < theCanvas.length - 1; i++ ) {
					theCanvas[i] = _bannerBorderEscapeCode + theCanvas[i].substring( 0, 1 ) + _resetEscapeCode + _bannerEscapeCode + theCanvas[i].substring( 1, theCanvas[i].length() - 1 ) + _resetEscapeCode + _bannerBorderEscapeCode + theCanvas[i].substring( theCanvas[i].length() - 1 ) + _resetEscapeCode;
				}
			}
			theCanvas[theCanvas.length - 1] = _bannerBorderEscapeCode + theCanvas[theCanvas.length - 1] + _resetEscapeCode;
		}
		_stdStream.print( fromTextBlock( theCanvas, toLineBreak() ) );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printLicense() {
		_stdStream.println( _licenseNote );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printLn() {
		_stdStream.println();
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printLn( String aLine ) {
		String[] theLines = new TextBlockBuilder().withText( aLine ).withColumnWidth( _consoleWidth ).withSplitTextMode( SplitTextMode.AT_SPACE ).toStrings();
		_stdStream.print( fromTextBlock( theLines, toLineBreak() ) );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printOptions() {
		if ( _argsSyntax != null ) {
			List<? extends Operand<?>> theOperands;
			if ( _argsSyntax instanceof Condition theCondition ) {
				theOperands = theCondition.toOperands();
			}
			else if ( _argsSyntax instanceof Operand<?> theOperand ) {
				theOperands = Arrays.asList( theOperand );
			}
			else {
				throw new IllegalStateException( "The args syntax of type <" + _argsSyntax.getClass().getName() + "> is neither of type <" + Condition.class.getName() + "> nor <" + Operand.class.getName() + "> and therefor no operands are available for processing!" );
			}
			Map<String, String[]> theOptArgs = new HashMap<>();
			int theMaxLength = 0;
			String eOpt;
			int eEffectiveLength;
			for ( Operand<?> eOperand : theOperands ) {
				eOpt = toCliContext().toSpec( eOperand );
				eEffectiveLength = AnsiEscapeCode.toLength( eOpt, _isEscapeCodesEnabled );
				if ( eEffectiveLength > theMaxLength ) theMaxLength = eEffectiveLength;
				theOptArgs.put( eOpt, new String[] { eOpt, ":", " ", eOperand.getDescription() } );
			}
			TableBuilder theTable = new TableBuilder().withTableStyle( TableStyle.BLANK_HEADER_BLANK_BODY ).withRowWidth( _consoleWidth ).withPrintStream( _stdStream ).withEscapeCodesEnabled( _isEscapeCodesEnabled ).withLeftBorder( false ).withRightBorder( false ).withDividerLine( false );
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withColumnWidth( theMaxLength, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( 1, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( 1, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.LEFT ).withColumnSplitTextMode( SplitTextMode.AT_SPACE );
			if ( _isEscapeCodesEnabled ) {
				theTable.withRowColumnEscapeCode( _descriptionEscapeCode );
			}
			theTable.withLineBreak( toLineBreak() );
			List<String> theKeys = new ArrayList<>( theOptArgs.keySet() );
			Collections.sort( theKeys, new Comparator<String>() {
				@Override
				public int compare( String a, String b ) {
					if ( a != null ) while ( a.startsWith( "-" ) ) a = a.substring( 1 );
					if ( b != null ) while ( b.startsWith( "-" ) ) b = b.substring( 1 );
					return a.compareToIgnoreCase( b );
				}
			} );
			for ( String eKey : theKeys ) {
				theTable.printRowContinue( theOptArgs.get( eKey ) );
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printSeparatorLn() {
		String theLine = new TextLineBuilder().withColumnWidth( _consoleWidth ).withLineChar( _separatorLnChar ).toString() + toLineBreak();
		if ( _isEscapeCodesEnabled ) {
			theLine = _lineSeparatorEscapeCode + theLine + _resetEscapeCode;
		}
		_stdStream.print( theLine );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printSynopsis() {
		List<Constituent> theChildren;
		if ( _argsSyntax instanceof XorCondition ) {
			XorCondition theCases = (XorCondition) _argsSyntax;
			theChildren = theCases.getChildren();
		}
		else {
			theChildren = new ArrayList<>();
			theChildren.add( _argsSyntax );
		}
		TableBuilder theTable = new TableBuilder().withTableStyle( TableStyle.BLANK_HEADER_BLANK_BODY ).withRowWidth( _consoleWidth ).withPrintStream( _stdStream ).withEscapeCodesEnabled( _isEscapeCodesEnabled ).withLeftBorder( false ).withRightBorder( false ).withDividerLine( true );
		if ( _isEscapeCodesEnabled ) {
			theTable.setResetEscapeCode( _resetEscapeCode );
		}
		theTable.addColumn().withColumnWidth( _name.length(), ColumnWidthType.ABSOLUTE ).withColumnHorizAlignTextMode( HorizAlignTextMode.RIGHT );
		theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.LEFT ).withColumnSplitTextMode( SplitTextMode.AT_SPACE );
		theTable.withLineBreak( toLineBreak() );

		String theCommand = (_isEscapeCodesEnabled ? _commandEscapeCode : "") + _name + (_isEscapeCodesEnabled ? _resetEscapeCode : "");
		String eUsage;
		for ( Constituent eChild : theChildren ) {
			eUsage = (eChild != null) ? eChild.toSynopsis( toCliContext() ) : "<?>";
			theTable.printRowContinue( theCommand, eUsage );
		}
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		if ( _argsSyntax != null ) {
			_argsSyntax.reset();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setArgsSyntax( Constituent aArgsSyntax ) {
		_argsSyntax = aArgsSyntax;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setArgumentEscapeCode( String aParamEscapeCode ) {
		_argumentEscapeCode = aParamEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerBorderEscapeCode( String aBannerBorderEscapeCode ) {
		_bannerBorderEscapeCode = aBannerBorderEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerEscapeCode( String aBannerEscapeCode ) {
		_bannerEscapeCode = aBannerEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerFont( Font aBannerFont ) {
		_bannerFont = aBannerFont;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerFontPalette( AsciiColorPalette aBannerFontPalette ) {
		_bannerFontPalette = aBannerFontPalette.getPalette();

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerFontPalette( char[] aColorPalette ) {
		_bannerFontPalette = aColorPalette;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCommandEscapeCode( String aCommandEscapeCode ) {
		_commandEscapeCode = aCommandEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConsoleWidth( int aConsoleWidth ) {
		_consoleWidth = _maxConsoleWidth != -1 ? (_maxConsoleWidth < aConsoleWidth ? _maxConsoleWidth : aConsoleWidth) : aConsoleWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCopyright( String aCopyrightNote ) {
		_copyrightNote = aCopyrightNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription( String aDescription ) {
		_description = aDescription;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescriptionEscapeCode( String aDescriptionEscapeCode ) {
		_descriptionEscapeCode = aDescriptionEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setErrorOut( PrintStream aErrorOut ) {
		_errStream = aErrorOut;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEscapeCodesEnabled( boolean isEscapeCodesEnabled ) {
		_isEscapeCodesEnabled = isEscapeCodesEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExamples( Example[] aExamples ) {
		_examples = new ArrayList<>( Arrays.asList( aExamples ) );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLicense( String aLicenseNote ) {
		_licenseNote = aLicenseNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLineBreak( String aLineBreak ) {
		if ( aLineBreak == null ) {
			aLineBreak = Terminal.getLineBreak();
		}
		_lineBreak = aLineBreak;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLineSeparatorEscapeCode( String aLineSeparatorEscapeCode ) {
		_lineSeparatorEscapeCode = aLineSeparatorEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLongOptionPrefix( String aLongOptionPrefix ) {
		_longOptionPrefix = aLongOptionPrefix;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMaxConsoleWidth( int aMaxConsoleWidth ) {
		_maxConsoleWidth = aMaxConsoleWidth;
		_consoleWidth = _maxConsoleWidth != -1 ? (_maxConsoleWidth < _consoleWidth ? _maxConsoleWidth : _consoleWidth) : _consoleWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName( String aName ) {
		_name = aName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptionEscapeCode( String aOptEscapeCode ) {
		_optionEscapeCode = aOptEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setResetEscapeCode( String aResetEscapeCode ) {
		_resetEscapeCode = aResetEscapeCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSeparatorLnChar( char aSeparatorLnChar ) {
		_separatorLnChar = aSeparatorLnChar;
		_hasOverrideSeparatorLnChar = true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setShortOptionPrefix( Character aShortOptionPrefix ) {
		_shortOptionPrefix = aShortOptionPrefix;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setStandardOut( PrintStream aStandardOut ) {
		_stdStream = aStandardOut;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSyntaxMetrics( SyntaxMetrics aSyntaxMetrics ) {
		_syntaxMetrics = aSyntaxMetrics;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTextBoxGrid( TextBoxGrid aTextBoxGrid ) {
		_textBoxGrid = aTextBoxGrid;
		if ( !_hasOverrideSeparatorLnChar ) {
			_separatorLnChar = aTextBoxGrid.getInnerLine();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTitle( String aTitle ) {
		_title = aTitle;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArgsParser withAddExample( Example aExamples ) {
		addExample( aExamples );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArgsParser withAddExample( String aDescription, Operand<?>... aOperands ) {
		addExample( aDescription, aOperands );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArgsParser withArgsSyntax( Constituent aArgsSyntax ) {
		setArgsSyntax( aArgsSyntax );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArgsParser withBannerFontPalette( AsciiColorPalette aBannerFontPalette ) {
		setBannerFontPalette( aBannerFontPalette );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArgsParser withSyntaxMetrics( SyntaxNotation aSyntaxNotation ) {
		setSyntaxMetrics( aSyntaxNotation );
		return this;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns the delimiter to be used by colliding command line args when
	 * creating non colliding arg's aliases (keys for key/value-pairs).
	 * 
	 * @return The according delimiter.
	 */
	protected char getDelimiter() {
		return Delimiter.INDEX.getChar();
	}

	/**
	 * Heuristically loads the arguments without any syntax required, e.g.
	 * without any root {@link Condition} to be set.
	 * 
	 * @param aArgs The arguments to be loaded.
	 * @param aDelimiter The delimiter to resolve name clashes.
	 * 
	 * @return A list of heuristically determined {@link Flag} and
	 *         {@link StringOperand} instances.
	 */
	protected static List<? extends Operand<?>> fromArgs( String[] aArgs, char aDelimiter ) {
		List<Operand<?>> theOperands = new ArrayList<>();
		Map<String, String> theArgs = RuntimeUtility.toProperties( aArgs, ArgsPrefix.toPrefixes(), aDelimiter );
		for ( String eKey : theArgs.keySet() ) {
			if ( Literal.TRUE.getValue().equalsIgnoreCase( theArgs.get( eKey ) ) ) {
				theOperands.add( new Flag( new RelationImpl<>( eKey, Boolean.TRUE ) ) );
			}
			else {
				theOperands.add( new StringOperand( new PropertyImpl( eKey, theArgs.get( eKey ) ) ) );
			}
		}
		return theOperands;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * From text block.
	 *
	 * @param aTextBlock the text block
	 * @param aDelimeter the deimeter
	 * 
	 * @return the string
	 */
	private String fromTextBlock( String[] aTextBlock, String aDelimeter ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( String eString : aTextBlock ) {
			if ( aDelimeter != null && aDelimeter.length() != 0 ) {
				if ( theBuilder.length() > 0 ) {
					theBuilder.append( aDelimeter );
				}
			}
			theBuilder.append( eString );
		}
		return theBuilder.toString() + toLineBreak();
	}

	private boolean hasBorder( String[] aCanvas ) {
		if ( aCanvas != null && aCanvas.length != 0 ) {
			char ePrevChar = aCanvas[0].length() != 0 ? aCanvas[0].charAt( 0 ) : ' ';
			// Top |-->
			for ( int i = 0; i < aCanvas[0].length(); i++ ) {
				if ( ePrevChar != aCanvas[0].charAt( i ) ) {
					return false;
				}
				ePrevChar = aCanvas[0].charAt( i );
			}
			// Bottom |-->
			for ( int i = 0; i < aCanvas[aCanvas.length - 1].length(); i++ ) {
				if ( ePrevChar != aCanvas[aCanvas.length - 1].charAt( i ) ) {
					return false;
				}
				ePrevChar = aCanvas[aCanvas.length - 1].charAt( i );
			}
			// Left |-->
			for ( int i = 0; i < aCanvas.length; i++ ) {
				if ( ePrevChar != aCanvas[i].charAt( 0 ) ) {
					return false;
				}
				ePrevChar = aCanvas[i].charAt( 0 );
			}
			// Right |-->
			for ( int i = 0; i < aCanvas.length; i++ ) {
				if ( ePrevChar != aCanvas[i].charAt( aCanvas[i].length() - 1 ) ) {
					return false;
				}
				ePrevChar = aCanvas[i].charAt( aCanvas[i].length() - 1 );
			}
		}
		return true;
	}

	private CliContext toCliContext() {
		String theNotationProperty = SystemProperty.ARGS_NOTATION.getValue();
		SyntaxNotation theSyntaxNotation = SyntaxNotation.toSyntaxNotation( theNotationProperty );
		CliContext theCliCtx = new CliContext( this, theSyntaxNotation );
		return theCliCtx;
	}

	/**
	 * To line break.
	 *
	 * @return the string
	 */
	private String toLineBreak() {
		if ( Terminal.isLineBreakRequired( _consoleWidth ) ) {
			return _lineBreak;
		}
		return "";
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
