// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import org.refcodes.struct.Relation;

/**
 * Declarative syntactic sugar which may be statically imported in order to
 * allow declarative definitions for the command line {@link Flag},
 * {@link Condition}, {@link Option} and {@link Operand} elements.
 */
public class CliSugar {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sugar for creating an {@link Example} array from a varargs argument.
	 * 
	 * @param aExamples the {@link Example} varargs argument.
	 * 
	 * @return The according array representation.
	 */
	public static Example[] examples( Example... aExamples ) {
		return aExamples;
	}

	/**
	 * Sugar for creating an {@link Example} from a description and the
	 * according {@link Operand} elements.
	 * 
	 * @param aDescription The explanatory text of the example.
	 * @param aOperands The operands required for the {@link Example}.
	 * 
	 * @return The according {@link Example} instance.
	 */
	public static Example example( String aDescription, Operand<?>... aOperands ) {
		return new Example( aDescription, aOperands );
	}

	/**
	 * Constructs a {@link Operation} with the given arguments.
	 * 
	 * @param aOperation The operation to declare.
	 * @param aDescription The description to be used (without any line breaks).
	 * 
	 * @return The accordingly created {@link Operation}.
	 */
	public static Operation operation( String aOperation, String aDescription ) {
		return new Operation( aOperation, aDescription );
	}

	/**
	 * Constructs a {@link Operation} with the given arguments.
	 * 
	 * @param aOperation The operation to declare.
	 * @param aAlias The operation's name to be used when constructing the
	 *        syntax.
	 * @param aDescription The description to be used (without any line breaks).
	 * 
	 * @return The accordingly created {@link Operation}.
	 */
	public static Operation operation( String aOperation, String aAlias, String aDescription ) {
		return new Operation( aOperation, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link AndCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested.
	 * 
	 * @return The according AND condition.
	 * 
	 * @see AndCondition
	 */
	public static Condition and( Constituent... aArgs ) {
		return new AndCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link AllCondition} with the {@link Constituent}
	 * ({@link Condition}) instance to be nested.
	 *
	 * @param aArg The {@link Constituent} ({@link Condition}) instance to be
	 *        nested.
	 * 
	 * @return The according ALL condition.
	 * 
	 * @see AllCondition
	 */
	public static Condition all( Constituent aArg ) {
		return new AllCondition( aArg );
	}

	/**
	 * Instantiates a new {@link OrCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested.
	 * 
	 * @return The according OR condition.
	 * 
	 * @see OrCondition
	 */
	public static Condition or( Constituent... aArgs ) {
		return new OrCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested.
	 * 
	 * @return The according XOR condition.
	 * 
	 * @see XorCondition
	 */
	public static Condition xor( Constituent... aArgs ) {
		return new XorCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link CasesCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested.
	 * 
	 * @return The according CasesCondition (https://www.metacodes.pro XOR)
	 *         condition.
	 * 
	 * @see CasesCondition
	 */
	public static Condition cases( Constituent... aArgs ) {
		return new CasesCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link AnyCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested. Any of the nested
	 * {@link Condition} conditions may match for the {@link AnyCondition} to
	 * match, e.g. all of the nested conditions are optional.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested
	 * 
	 * @return The according {@link AbstractCondition}.
	 * 
	 * @see AnyCondition
	 */
	public static Condition any( Constituent... aArgs ) {
		return new AnyCondition( aArgs );
	}

	/**
	 * Semantically identical synonym for the {@link #optional(Constituent...)}
	 * declaration.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested
	 * 
	 * @return The according {@link AbstractCondition}.
	 * 
	 * @see AnyCondition
	 */
	public static Condition optional( Constituent... aArgs ) {
		return new AnyCondition( aArgs );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aAlias The identifier to be used when printing the syntax via the
	 *        {@link Constituent#toSyntax(CliContext)} method.
	 * @param aDescription The description to be used (without any line breaks).
	 * 
	 * @return the none operand
	 * 
	 * @see NoneOperand
	 */
	public static NoneOperand none( String aAlias, String aDescription ) {
		return new NoneOperand( aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aDescription The description to be used (without any line breaks).
	 * 
	 * @return the none operand
	 * 
	 * @see NoneOperand
	 */
	public static NoneOperand none( String aDescription ) {
		return new NoneOperand( aDescription );
	}

	/**
	 * Instantiates a new {@link NoneOperand}.
	 *
	 * @param aProperty The key (= alias) and the value for the operand.
	 * 
	 * @return the none operand
	 * 
	 * @see NoneOperand
	 */
	public static NoneOperand none( Relation<String, Boolean> aProperty ) {
		return new NoneOperand( aProperty );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments.
	 *
	 * @param <T> The generic type of the enumeration.
	 * 
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link EnumOption}
	 * 
	 * @return The accordingly created {@link EnumOption} instance.
	 * 
	 * @see EnumOption
	 */
	public static <T extends Enum<T>> EnumOption<T> enumOption( Character aShortOption, String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		return new EnumOption<T>( aShortOption, aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param <T> The generic type of the enumeration.
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aDescription The description of the {@link EnumOption}
	 * 
	 * @return The accordingly created {@link EnumOption} instance.
	 * 
	 * @see EnumOption
	 */
	public static <T extends Enum<T>> EnumOption<T> enumOption( Character aShortOption, String aLongOption, Class<T> aType, String aDescription ) {
		return new EnumOption<T>( aShortOption, aLongOption, aType, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments.
	 *
	 * @param <T> The generic type of the enumeration.
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link EnumOption}
	 * 
	 * @return The accordingly created {@link EnumOption} instance.
	 * 
	 * @see EnumOption
	 */
	public static <T extends Enum<T>> EnumOption<T> enumOption( String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		return new EnumOption<T>( aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param <T> The generic type of the enumeration.
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aDescription The description of the {@link EnumOption}
	 * 
	 * @return The accordingly created {@link EnumOption} instance.
	 * 
	 * @see EnumOption
	 */
	public static <T extends Enum<T>> EnumOption<T> enumOption( String aLongOption, Class<T> aType, String aDescription ) {
		return new EnumOption<T>( aLongOption, aType, aDescription );
	}

	/**
	 * Instantiates a new {@link Flag} with the given arguments.
	 *
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link Flag}
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see Flag
	 */
	public static Flag flag( String aLongOption, String aAlias, String aDescription ) {
		return new Flag( aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link Flag} with the given arguments. In case a long
	 * option is provided, the intance's alias will automatically be set with
	 * the long option.
	 *
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link Flag}
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see Flag
	 */
	public static Flag flag( String aLongOption, String aDescription ) {
		return new Flag( aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link Flag} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link Flag}
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see Flag
	 */
	public static Flag flag( Character aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new Flag( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link Flag} with the given arguments. In case a long
	 * option is provided, the intance's alias will automatically be set with
	 * the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link Flag}
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see Flag
	 */
	public static Flag flag( Character aShortOption, String aLongOption, String aDescription ) {
		return new Flag( aShortOption, aLongOption, aDescription );
	}

	/**
	 * Constructs the predefined daemon {@link Flag}.
	 * 
	 * @param aDescription The description to be used (without any line breaks).
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see DaemonFlag
	 */
	public static DaemonFlag daemonFlag( String aDescription ) {
		return new DaemonFlag( aDescription );
	}

	/**
	 * Constructs the predefined daemon {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see DaemonFlag
	 */
	public static DaemonFlag daemonFlag() {
		return new DaemonFlag();
	}

	/**
	 * Constructs the predefined daemon {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see DaemonFlag
	 */
	public static DaemonFlag daemonFlag( boolean hasShortOption ) {
		return new DaemonFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined force {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see ForceFlag
	 */
	public static ForceFlag forceFlag( String aDescription ) {
		return new ForceFlag( aDescription );
	}

	/**
	 * Constructs the predefined force {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see ForceFlag
	 */
	public static ForceFlag forceFlag() {
		return new ForceFlag();
	}

	/**
	 * Constructs the predefined force {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see ForceFlag
	 */
	public static ForceFlag forceFlag( boolean hasShortOption ) {
		return new ForceFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined help {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see HelpFlag
	 */
	public static HelpFlag helpFlag( String aDescription ) {
		return new HelpFlag( aDescription );
	}

	/**
	 * Constructs the predefined help {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see HelpFlag
	 */
	public static HelpFlag helpFlag() {
		return new HelpFlag();
	}

	/**
	 * Constructs the predefined help {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see HelpFlag
	 */
	public static HelpFlag helpFlag( boolean hasShortOption ) {
		return new HelpFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined init {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see InitFlag
	 */
	public static InitFlag initFlag( String aDescription ) {
		return new InitFlag( aDescription );
	}

	/**
	 * Constructs the predefined init {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see InitFlag
	 */
	public static InitFlag initFlag() {
		return new InitFlag();
	}

	/**
	 * Constructs the predefined init {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see InitFlag
	 */
	public static InitFlag initFlag( boolean hasShortOption ) {
		return new InitFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined clean {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see CleanFlag
	 */
	public static CleanFlag cleanFlag( String aDescription ) {
		return new CleanFlag( aDescription );
	}

	/**
	 * Constructs the predefined clean {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see CleanFlag
	 */
	public static CleanFlag cleanFlag() {
		return new CleanFlag();
	}

	/**
	 * Constructs the predefined clean {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see CleanFlag
	 */
	public static CleanFlag cleanFlag( boolean hasShortOption ) {
		return new CleanFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined system info {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see SysInfoFlag
	 */
	public static SysInfoFlag sysInfoFlag( String aDescription ) {
		return new SysInfoFlag( aDescription );
	}

	/**
	 * Constructs the predefined system info {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see SysInfoFlag
	 */
	public static SysInfoFlag sysInfoFlag() {
		return new SysInfoFlag();
	}

	/**
	 * Constructs the predefined system info {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see SysInfoFlag
	 */
	public static SysInfoFlag sysInfoFlag( boolean hasShortOption ) {
		return new SysInfoFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined quiet {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see QuietFlag
	 */
	public static QuietFlag quietFlag( String aDescription ) {
		return new QuietFlag( aDescription );
	}

	/**
	 * Constructs the predefined quiet {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see QuietFlag
	 */
	public static QuietFlag quietFlag() {
		return new QuietFlag();
	}

	/**
	 * Constructs the predefined quiet {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see QuietFlag
	 */
	public static QuietFlag quietFlag( boolean hasShortOption ) {
		return new QuietFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined verbose {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see VerboseFlag
	 */
	public static VerboseFlag verboseFlag( String aDescription ) {
		return new VerboseFlag( aDescription );
	}

	/**
	 * Constructs the predefined verbose {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see VerboseFlag
	 */
	public static VerboseFlag verboseFlag() {
		return new VerboseFlag();
	}

	/**
	 * Constructs the predefined verbose {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see VerboseFlag
	 */
	public static VerboseFlag verboseFlag( boolean hasShortOption ) {
		return new VerboseFlag( hasShortOption );
	}

	/**
	 * Constructs the predefined debug {@link Flag}.
	 *
	 * @param aDescription The description to use when printing out the help
	 *        text.
	 * 
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see DebugFlag
	 */
	public static DebugFlag debugFlag( String aDescription ) {
		return new DebugFlag( aDescription );
	}

	/**
	 * Constructs the predefined debug {@link Flag}.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see DebugFlag
	 */
	public static DebugFlag debugFlag() {
		return new DebugFlag();
	}

	/**
	 * Constructs the predefined debug {@link Flag}.
	 * 
	 * @param hasShortOption True in case to also enable the short option, else
	 *        only the long option takes effect.
	 *
	 * @return The accordingly created {@link Flag} instance.
	 * 
	 * @see DebugFlag
	 */
	public static DebugFlag debugFlag( boolean hasShortOption ) {
		return new DebugFlag( hasShortOption );
	}

	/**
	 * Instantiates a new {@link IntOption} with the given arguments.
	 *
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link IntOption}
	 * 
	 * @return The accordingly created {@link IntOption} instance.
	 * 
	 * @see IntOption
	 */
	public static IntOption intOption( String aLongOption, String aAlias, String aDescription ) {
		return new IntOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link IntOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link IntOption}
	 * 
	 * @return The accordingly created {@link IntOption} instance.
	 * 
	 * @see IntOption
	 */
	public static IntOption intOption( String aLongOption, String aDescription ) {
		return new IntOption( aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link IntOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link IntOption}
	 * 
	 * @return The accordingly created {@link IntOption} instance.
	 * 
	 * @see IntOption
	 */
	public static IntOption intOption( Character aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new IntOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link IntOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link IntOption}
	 * 
	 * @return The accordingly created {@link IntOption} instance.
	 * 
	 * @see IntOption
	 */
	public static IntOption intOption( Character aShortOption, String aLongOption, String aDescription ) {
		return new IntOption( aShortOption, aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link LongOption} with the given arguments.
	 *
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link LongOption}
	 * 
	 * @return The accordingly created {@link LongOption} instance.
	 * 
	 * @see LongOption
	 */
	public static LongOption longOption( String aLongOption, String aAlias, String aDescription ) {
		return new LongOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link LongOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link LongOption}
	 * 
	 * @return The accordingly created {@link LongOption} instance.
	 * 
	 * @see LongOption
	 */
	public static LongOption longOption( String aLongOption, String aDescription ) {
		return new LongOption( aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link LongOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link LongOption}
	 * 
	 * @return The accordingly created {@link LongOption} instance.
	 * 
	 * @see LongOption
	 */
	public static LongOption longOption( Character aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new LongOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link LongOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link LongOption}
	 * 
	 * @return The accordingly created {@link LongOption} instance.
	 * 
	 * @see LongOption
	 */
	public static LongOption longOption( Character aShortOption, String aLongOption, String aDescription ) {
		return new LongOption( aShortOption, aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link FloatOption} with the given arguments.
	 *
	 * @param aFloatOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link FloatOption}
	 * 
	 * @return The accordingly created {@link FloatOption} instance.
	 * 
	 * @see FloatOption
	 */
	public static FloatOption floatOption( String aFloatOption, String aAlias, String aDescription ) {
		return new FloatOption( aFloatOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link FloatOption} with the given arguments. In case
	 * a long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aFloatOption The long option to use.
	 * @param aDescription The description of the {@link FloatOption}
	 * 
	 * @return The accordingly created {@link FloatOption} instance.
	 * 
	 * @see FloatOption
	 */
	public static FloatOption floatOption( String aFloatOption, String aDescription ) {
		return new FloatOption( aFloatOption, aDescription );
	}

	/**
	 * Instantiates a new {@link FloatOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aFloatOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link FloatOption}
	 * 
	 * @return The accordingly created {@link FloatOption} instance.
	 * 
	 * @see FloatOption
	 */
	public static FloatOption floatOption( Character aShortOption, String aFloatOption, String aAlias, String aDescription ) {
		return new FloatOption( aShortOption, aFloatOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link FloatOption} with the given arguments. In case
	 * a long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aFloatOption the float option
	 * @param aDescription The description of the {@link FloatOption}
	 * 
	 * @return The accordingly created {@link FloatOption} instance.
	 * 
	 * @see FloatOption
	 */
	public static FloatOption floatOption( Character aShortOption, String aFloatOption, String aDescription ) {
		return new FloatOption( aShortOption, aFloatOption, aDescription );
	}

	/**
	 * Instantiates a new {@link DoubleOption} with the given arguments.
	 *
	 * @param aDoubleOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link DoubleOption}
	 * 
	 * @return The accordingly created {@link DoubleOption} instance.
	 * 
	 * @see DoubleOption
	 */
	public static DoubleOption doubleOption( String aDoubleOption, String aAlias, String aDescription ) {
		return new DoubleOption( aDoubleOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link DoubleOption} with the given arguments. In case
	 * a long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aDoubleOption The long option to use.
	 * @param aDescription The description of the {@link DoubleOption}
	 * 
	 * @return The accordingly created {@link DoubleOption} instance.
	 * 
	 * @see DoubleOption
	 */
	public static DoubleOption doubleOption( String aDoubleOption, String aDescription ) {
		return new DoubleOption( aDoubleOption, aDescription );
	}

	/**
	 * Instantiates a new {@link DoubleOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aDoubleOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link DoubleOption}
	 * 
	 * @return The accordingly created {@link DoubleOption} instance.
	 * 
	 * @see DoubleOption
	 */
	public static DoubleOption doubleOption( Character aShortOption, String aDoubleOption, String aAlias, String aDescription ) {
		return new DoubleOption( aShortOption, aDoubleOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link DoubleOption} with the given arguments. In case
	 * a long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aDoubleOption the double option
	 * @param aDescription The description of the {@link DoubleOption}
	 * 
	 * @return The accordingly created {@link DoubleOption} instance.
	 * 
	 * @see DoubleOption
	 */
	public static DoubleOption doubleOption( Character aShortOption, String aDoubleOption, String aDescription ) {
		return new DoubleOption( aShortOption, aDoubleOption, aDescription );
	}

	/**
	 * Instantiates a new {@link StringOption} with the given arguments.
	 *
	 * @param aStringOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link StringOption}
	 * 
	 * @return The accordingly created {@link StringOption} instance.
	 * 
	 * @see StringOption
	 */
	public static StringOption stringOption( String aStringOption, String aAlias, String aDescription ) {
		return new StringOption( aStringOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link StringOption} with the given arguments. In case
	 * a long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aStringOption The long option to use.
	 * @param aDescription The description of the {@link StringOption}
	 * 
	 * @return The accordingly created {@link StringOption} instance.
	 * 
	 * @see StringOption
	 */
	public static StringOption stringOption( String aStringOption, String aDescription ) {
		return new StringOption( aStringOption, aDescription );
	}

	/**
	 * Instantiates a new {@link StringOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aStringOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link StringOption}
	 * 
	 * @return The accordingly created {@link StringOption} instance.
	 * 
	 * @see StringOption
	 */
	public static StringOption stringOption( Character aShortOption, String aStringOption, String aAlias, String aDescription ) {
		return new StringOption( aShortOption, aStringOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link StringOption} with the given arguments. In case
	 * a long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aStringOption the string option
	 * @param aDescription The description of the {@link StringOption}
	 * 
	 * @return The accordingly created {@link StringOption} instance.
	 * 
	 * @see StringOption
	 */
	public static StringOption stringOption( Character aShortOption, String aStringOption, String aDescription ) {
		return new StringOption( aShortOption, aStringOption, aDescription );
	}

	/**
	 * Instantiates a new {@link CharOption} with the given arguments.
	 *
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link CharOption}
	 * 
	 * @return The accordingly created {@link CharOption} instance.
	 * 
	 * @see CharOption
	 */
	public static CharOption charOption( String aLongOption, String aAlias, String aDescription ) {
		return new CharOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link CharOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link CharOption}
	 * 
	 * @return The accordingly created {@link CharOption} instance.
	 * 
	 * @see CharOption
	 */
	public static CharOption charOption( String aLongOption, String aDescription ) {
		return new CharOption( aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link CharOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link CharOption}
	 * 
	 * @return The accordingly created {@link EnumOption}instance.
	 * 
	 * @see CharOption
	 */
	public static CharOption charOption( Character aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new CharOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link CharOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link CharOption}
	 * 
	 * @return The accordingly created {@link EnumOption}instance.
	 * 
	 * @see CharOption
	 */
	public static CharOption charOption( Character aShortOption, String aLongOption, String aDescription ) {
		return new CharOption( aShortOption, aLongOption, aDescription );
	}

	/**
	 * Creates a {@link ConfigOption} representing value specifying a
	 * configuration resource (file).
	 *
	 * @return the according {@link ConfigOption}.
	 * 
	 * @see ConfigOption
	 */
	public static ConfigOption configOption() {
		return new ConfigOption();
	}

	/**
	 * Instantiates a new config (file) {@link Option}.
	 *
	 * @param aDescription The description to use.
	 *
	 * @return the according {@link ConfigOption}.
	 * 
	 * @see ConfigOption
	 */
	public static ConfigOption configOption( String aDescription ) {
		return new ConfigOption( aDescription );
	}

	/**
	 * Instantiates a new {@link FileOption} with the given arguments.
	 *
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link FileOption}
	 * 
	 * @return The according {@link FileOption}.
	 * 
	 * @see FileOption
	 */
	public static FileOption fileOption( String aLongOption, String aAlias, String aDescription ) {
		return new FileOption( aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link FileOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aLongOption The long option to use.
	 * @param aDescription the description
	 * 
	 * @return The according {@link FileOption}.
	 * 
	 * @see FileOption
	 */
	public static FileOption fileOption( String aLongOption, String aDescription ) {
		return new FileOption( aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link FileOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link FileOption}
	 * 
	 * @return The according {@link FileOption}.
	 * 
	 * @see FileOption
	 */
	public static FileOption fileOption( Character aShortOption, String aLongOption, String aAlias, String aDescription ) {
		return new FileOption( aShortOption, aLongOption, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link FileOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aDescription The description of the {@link FileOption}
	 * 
	 * @return The according {@link FileOption}.
	 * 
	 * @see FileOption
	 */
	public static FileOption fileOption( Character aShortOption, String aLongOption, String aDescription ) {
		return new FileOption( aShortOption, aLongOption, aDescription );
	}

	/**
	 * Instantiates a new {@link StringOperand}.
	 *
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description to be used (without any line breaks).
	 * 
	 * @return The according {@link StringOperand}.
	 * 
	 * @see StringOperand
	 */
	public static StringOperand stringOperand( String aAlias, String aDescription ) {
		return new StringOperand( aAlias, aDescription );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Option}. This way any {@link Option} can also be used as an array
	 * {@link Option}, e.g. it can be provided multiple times in the command
	 * line arguments.
	 * 
	 * @param <T> The type of the {@link Option} for which to create an
	 *        {@link ArrayOption}.
	 * @param aOption The {@link Option} which's array counterpart is to be
	 *        defined.
	 * 
	 * @return The according {@link ArrayOption}.
	 */
	public static <T> ArrayOption<T> asArray( Option<T> aOption ) {
		return new ArrayOption<T>( aOption );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Option}. This way any {@link Option} can also be used as an array
	 * {@link Option}, e.g. it can by provided multiple times in the command
	 * line arguments.
	 * 
	 * @param <T> The type of the {@link Option} for which to create an
	 *        {@link ArrayOption}.
	 * @param aOption The {@link Option} which's array counterpart is to be
	 *        defined.
	 * @param aLength The number of array elements, or -1 if there is no limit.
	 * 
	 * @return The according {@link ArrayOption}.
	 */
	public static <T> ArrayOption<T> asArray( Option<T> aOption, int aLength ) {
		return new ArrayOption<T>( aOption, aLength );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Option}. This way any {@link Option} can also be used as an array
	 * {@link Option}, e.g. it can by provided multiple times in the command
	 * line arguments.
	 * 
	 * @param <T> The type of the {@link Option} for which to create an
	 *        {@link ArrayOption}.
	 * @param aOption The {@link Option} which's array counterpart is to be
	 *        defined.
	 * @param aMinLength The minimum number of array elements, or -1 if there is
	 *        no limit.
	 * @param aMaxLength The maximum number of array elements, or -1 if there is
	 *        no limit.
	 * 
	 * @return The according {@link ArrayOption}.
	 */
	public static <T> ArrayOption<T> asArray( Option<T> aOption, int aMinLength, int aMaxLength ) {
		return new ArrayOption<T>( aOption, aMinLength, aMaxLength );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Operand}. This way any {@link Operand} can also be used as an
	 * array {@link Operand}, e.g. it can be provided multiple times in the
	 * command line arguments.
	 * 
	 * @param <T> The type of the {@link Operand} for which to create an
	 *        {@link ArrayOperand}.
	 * @param aOperand The {@link Operand} which's array counterpart is to be
	 *        defined.
	 * 
	 * @return The according {@link ArrayOperand}.
	 */
	public static <T> ArrayOperand<T> asArray( Operand<T> aOperand ) {
		return new ArrayOperand<T>( aOperand );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Operand}. This way any {@link Operand} can also be used as an
	 * array {@link Operand}, e.g. it can by provided multiple times in the
	 * command line arguments.
	 * 
	 * @param <T> The type of the {@link Operand} for which to create an
	 *        {@link ArrayOperand}.
	 * @param aOperand The {@link Operand} which's array counterpart is to be
	 *        defined.
	 * @param aLength The number of array elements, or -1 if there is no limit.
	 * 
	 * @return The according {@link ArrayOperand}.
	 */
	public static <T> ArrayOperand<T> asArray( Operand<T> aOperand, int aLength ) {
		return new ArrayOperand<T>( aOperand, aLength );
	}

	/**
	 * Creates an array representation facade for the encapsulated
	 * {@link Operand}. This way any {@link Operand} can also be used as an
	 * array {@link Operand}, e.g. it can by provided multiple times in the
	 * command line arguments.
	 * 
	 * @param <T> The type of the {@link Operand} for which to create an
	 *        {@link ArrayOperand}.
	 * @param aOperand The {@link Operand} which's array counterpart is to be
	 *        defined.
	 * @param aMinLength The minimum number of array elements, or -1 if there is
	 *        no limit.
	 * @param aMaxLength The maximum number of array elements, or -1 if there is
	 *        no limit.
	 * 
	 * @return The according {@link ArrayOperand}.
	 */
	public static <T> ArrayOperand<T> asArray( Operand<T> aOperand, int aMinLength, int aMaxLength ) {
		return new ArrayOperand<T>( aOperand, aMinLength, aMaxLength );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
