// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * This utility class provides method useful for the refcodes-cli artifact and
 * whose implementation has been motivated by the implementation of the
 * refcodes-cli artifact.
 */
public final class CliUtility {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new console utility.
	 */
	private CliUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Takes all {@link Operand} instances found in the provided {@link List}s
	 * and adds all therein found argument arrays (as of
	 * {@link Operand#toParsedArgs()}) to the result.
	 * 
	 * @param aOperands The lists containing the {@link Operand} instances whose
	 *        command line arguments are to be added to the result.
	 * 
	 * @return All the command line arguments detected in the provided
	 *         {@link Operand}s {@link List}s.
	 */
	@SuppressWarnings("unchecked")
	public static String[] toParsedArgs( List<? extends Operand<?>>... aOperands ) {
		if ( aOperands != null ) {
			List<String> theArgs = new ArrayList<String>();
			for ( List<? extends Operand<?>> eOperands : aOperands ) {
				if ( eOperands != null ) {
					for ( Operand<?> eOperand : eOperands ) {
						if ( eOperand.toParsedArgs() != null ) {
							theArgs.addAll( Arrays.asList( eOperand.toParsedArgs() ) );
						}
					}
				}
			}
			return theArgs.toArray( new String[theArgs.size()] );
		}
		return null;
	}

	/**
	 * Creates the difference between the provided set and the provided subset.
	 * 
	 * @param aArgs The set to be used for the diff operation.
	 * @param aArgsSubset The subset to be used for the diff operation.
	 * 
	 * @return The difference between the set and the subset.
	 */
	public static String[] toArgsDiff( String[] aArgs, String[] aArgsSubset ) {
		List<String> theDiff = new ArrayList<String>( Arrays.asList( aArgs ) );
		int index = 0;
		String eeElement;
		for ( String eElement : aArgsSubset ) {
			out: {
				// Try to get subsequent elements |-->
				for ( int l = index; l < theDiff.size(); l++ ) {
					eeElement = theDiff.get( l );
					if ( eElement.equals( eeElement ) ) {
						theDiff.remove( l );
						index = l;
						break out;
					}
				}
				// Try to get subsequent elements <--|

				// No subsequent elelemt found, start from beginning |-->
				for ( int l = 0; l < theDiff.size(); l++ ) {
					eeElement = theDiff.get( l );
					if ( eElement.equals( eeElement ) ) {
						theDiff.remove( l );
						index = l;
						break out;
					}
				}
				// No subsequent elelemt found, start from beginning <--|
			}
		}
		return theDiff.toArray( new String[theDiff.size()] );
	}

	/**
	 * Creates the difference between the provided set and the provided
	 * {@link List}s therein found argument arrays subset (as of
	 * {@link Operand#toParsedArgs()}).
	 * 
	 * @param aArgs The set to be used for the diff operation.
	 * @param aArgsSubset The subset to be used for the diff operation being the
	 *        lists containing the {@link Operand} instances whose command line
	 *        arguments are to be diffed.
	 * 
	 * @return The difference between the set and the subset.
	 */
	@SuppressWarnings("unchecked")
	public static String[] toArgsDiff( String[] aArgs, List<? extends Operand<?>> aArgsSubset ) {
		return toArgsDiff( aArgs, toParsedArgs( aArgsSubset ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
