// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import org.refcodes.struct.Relation;

/**
 * The {@link EnumOption} represents an {@link Option} of enumeration types.
 *
 * @param <T> The generic type of the enumeration.
 */
public class EnumOption<T extends Enum<T>> extends AbstractOption<T> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link EnumOption}
	 */
	public EnumOption( Character aShortOption, String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		super( aShortOption, aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option, else the short option is used ass alias.
	 *
	 * @param aShortOption The short option to use.
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aDescription The description of the {@link EnumOption}
	 */
	public EnumOption( Character aShortOption, String aLongOption, Class<T> aType, String aDescription ) {
		super( aShortOption, aLongOption, aType, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments.
	 *
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aAlias The alias to be used for naming purposes.
	 * @param aDescription The description of the {@link EnumOption}
	 */
	public EnumOption( String aLongOption, Class<T> aType, String aAlias, String aDescription ) {
		super( aLongOption, aType, aAlias, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the given arguments. In case a
	 * long option is provided, the intance's alias will automatically be set
	 * with the long option.
	 *
	 * @param aLongOption The long option to use.
	 * @param aType The type of the enumeration to be used.
	 * @param aDescription The description of the {@link EnumOption}
	 */
	public EnumOption( String aLongOption, Class<T> aType, String aDescription ) {
		super( aLongOption, aType, aDescription );
	}

	/**
	 * Instantiates a new {@link EnumOption} with the alias being the proerty's
	 * key and the value being the property's value. Depending on the provided
	 * property's key, the key is either used for the short option or the long
	 * option.
	 * 
	 * @param aProperty The key (=alias) and the value for the operand.
	 * @param aType The type of the enumeration to be used.
	 */
	public EnumOption( Relation<String, T> aProperty, Class<T> aType ) {
		super( aProperty, aType );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected T toType( String aArg ) throws ParseArgsException {
		IllegalArgumentException theFirstCause = null;
		try {
			return Enum.valueOf( getType(), aArg );
		}
		catch ( IllegalArgumentException e1 ) {
			theFirstCause = e1;
			try {
				return Enum.valueOf( getType(), aArg.toUpperCase() );
			}
			catch ( IllegalArgumentException e2 ) {
				try {
					return Enum.valueOf( getType(), aArg.toLowerCase() );
				}
				catch ( IllegalArgumentException e3 ) {
					throw new ParseArgsException( new String[] { aArg }, "Invalid argument \"" + aArg + "\" for option <" + getAlias() + ">.", theFirstCause );
				}
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
