// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

/**
 * An {@link OrCondition} represents a list of {@link Condition}
 * ({@link Constituent}) instances of which at least one must be parsed
 * successfully when the {@link Constituent} s'
 * {@link Constituent#parseArgs(String[], String[], CliContext)} methods are
 * invoked. The command line arguments syntax <code>{ -a | -b | -c }</code>
 * specifies that at least "-a", "-b" or "-c" must be set. In case none is set ,
 * then the {@link OrCondition} will terminate the
 * {@link #parseArgs(String[], String[], CliContext)} method with an exception.
 */
public class OrCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link OrCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested.
	 */
	public OrCondition( Constituent... aArgs ) {
		super( "At least one (OR) nested syntaxable must match from the arguments.", aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {
		List<Operand<?>> theResult = null;
		List<Operand<?>> eOperands;
		for ( Constituent eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions, aCliCtx );
				if ( eOperands != null ) {
					if ( theResult == null ) {
						theResult = new ArrayList<Operand<?>>();
						;
					}
					theResult.addAll( eOperands );
				}
				aArgs = CliUtility.toArgsDiff( aArgs, eOperands );
			}
			catch ( UnknownArgsException | AmbiguousArgsException exc ) {}
		}
		if ( theResult != null ) {
			return theResult;
		}
		throw new UnknownArgsException( aArgs, "Not any syntax matched the provided command line arguments, though at least one elective must match." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( CliContext aCliCtx ) {
		String theSynopsis = "";
		for ( Constituent eChild : getChildren() ) {
			if ( theSynopsis.length() != 0 ) {
				theSynopsis += aCliCtx.getSyntaxMetrics().getOrSymbol() != null && aCliCtx.getSyntaxMetrics().getOrSymbol().length() != 0 ? " " + aCliCtx.getSyntaxMetrics().getOrSymbol() + " " : " ";
			}
			theSynopsis += eChild.toSyntax( aCliCtx );
		}
		return theSynopsis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return toSchema().toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
