// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.refcodes.textual.VerboseTextBuilder;

/**
 * An {@link XorCondition} represents a list of {@link Condition}
 * ({@link Constituent}) instances of which only one is allowed to be parsed
 * successfully when the
 * {@link Condition#parseArgs(String[], String[], CliContext)} methods are
 * invoked. The command line arguments syntax <code>{ -a ^ -b ^ -c }</code>
 * specifies that only "-a", only "-b" or only "-c" must be set. In case more
 * then one is set or none, then the {@link XorCondition} will terminate the
 * {@link #parseArgs(String[], String[], CliContext)} method with an exception.
 */
public class XorCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested
	 */
	public XorCondition( Constituent... aArgs ) {
		super( "Exactly one (XOR) of the nested syntaxables must match from the arguments.", aArgs );
	}

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Constituent}
	 * ({@link Condition}) instances to be nested.
	 * 
	 * @param aDescription The description of this {@link Condition}.
	 *
	 * @param aArgs The {@link Constituent} ({@link Condition}) instances to be
	 *        nested
	 */
	protected XorCondition( String aDescription, Constituent... aArgs ) {
		super( aDescription, aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {

		// In case if one child only |-->
		//	if ( getChildren().size() == 1 ) {
		//		return getChildren().get( 0 ).parseArgs( aArgs );
		//	}
		// In case if one child only <--|

		List<Condition> theOptionals = new ArrayList<Condition>();
		List<Operand<?>> theResult = null;
		Synopsisable theResultSyntaxable = null;
		List<Operand<?>> eOperands = null;
		int theMatchCount = getChildren().size();
		for ( Constituent eChild : getChildren() ) {
			if ( eChild instanceof AnyCondition ) theOptionals.add( (AnyCondition) eChild );
			try {
				if ( theResult != null && eChild instanceof Operand && theResultSyntaxable instanceof Option ) {
					eOperands = (List<Operand<?>>) eChild.parseArgs( CliUtility.toArgsDiff( aArgs, theResult ), aOptions, aCliCtx );
				}
				else {
					eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions, aCliCtx );
				}
				if ( eOperands != null && eOperands.isEmpty() ) eOperands = null;

			}
			catch ( ArgsSyntaxException ignore ) {
				theMatchCount--;
			}

			// Same operands in different children |-->
			theResult = removeDuplicates( theResult, eOperands );
			// Same operands in different children <--|

			if ( theResult != null && eOperands != null ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match: " + new VerboseTextBuilder().withElements( CliUtility.toParsedArgs( theResult, eOperands ) ).toString() );
			}
			if ( eOperands != null && !eOperands.isEmpty() && theResult == null ) {
				theResult = eOperands;
				theResultSyntaxable = eChild;
			}
			eOperands = null;
		}

		if ( theMatchCount == 0 ) {
			throw new UnknownArgsException( aArgs, "Not one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match!" );
		}
		if ( theMatchCount > 1 ) {
			if ( theOptionals.size() > 1 && (aArgs != null && aArgs.length > 0) ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive optional syntax matched the provided command line arguments, causing exclusive contextual ambiguity: " + new VerboseTextBuilder().withElements( toSpec( theOptionals ) ).toString() );
			}
			theMatchCount -= theOptionals.size();
			if ( theMatchCount > 1 ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match:" );
			}
		}
		if ( theResult != null ) {
			return theResult;
		}
		if ( theOptionals.size() == 0 ) {
			return Collections.EMPTY_LIST;
		}

		throw new UnknownArgsException( aArgs, "Not one exclusive syntax matched the provided command line arguments, though exactly one exclusiveness must match!" );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( CliContext aCliCtx ) {
		String theSynopsis = "";
		for ( Constituent eChild : getChildren() ) {
			if ( theSynopsis.length() != 0 ) {
				theSynopsis += aCliCtx.getSyntaxMetrics().getXorSymbol() != null && aCliCtx.getSyntaxMetrics().getXorSymbol().length() != 0 ? " " + aCliCtx.getSyntaxMetrics().getXorSymbol() + " " : " ";
			}
			theSynopsis += eChild.toSyntax( aCliCtx );
		}
		return theSynopsis;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static List<String> toSpec( List<Condition> aConditions ) {
		List<String> theParameters = new ArrayList<>();
		for ( Condition eArgsSyntax : aConditions ) {
			for ( Operand<?> eOperand : eArgsSyntax.toOperands() ) {
				theParameters.add( new CliContext( false ).toSpec( eOperand ) );
			}
		}
		return theParameters;
	}

	private static List<Operand<?>> removeDuplicates( List<Operand<?>> aElements, List<Operand<?>> aDuplicates ) {
		if ( aElements != null && aDuplicates != null ) {
			for ( Operand<?> eOperand : aDuplicates ) {
				if ( aElements.contains( eOperand ) ) {
					aElements.remove( eOperand );
				}
			}
			if ( aElements.isEmpty() ) {
				aElements = null;
			}
		}
		return aElements;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
