// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.refcodes.cli.CliException.ConsoleArgsException;
import org.refcodes.mixin.SourceAccessor;

/**
 * Thrown in case of a command line arguments mismatch regarding provided and
 * expected args.
 */
public class ArgsSyntaxException extends ConsoleArgsException implements SourceAccessor<Term>, MatchCountAccessor {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Term _source;
	private int _matchCount;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aSource The {@link Term} responsible for this exception (source).
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, Term aSource ) {
		super( aArgs, aMessage );
		_source = aSource;
		_matchCount = aSource.getMatchCount();
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aErrorCode The error code identifying this exception.
	 * @param aSource The {@link Term} responsible for this exception (source).
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, String aErrorCode, Term aSource ) {
		super( aArgs, aMessage, aErrorCode );
		_source = aSource;
		_matchCount = aSource.getMatchCount();
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aSource The {@link Term} responsible for this exception (source).
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, Throwable aCause, Term aSource ) {
		super( aArgs, aMessage, aCause );
		_source = aSource;
		_matchCount = aSource.getMatchCount();
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aErrorCode The error code identifying this exception.
	 * @param aSource The {@link Term} responsible for this exception (source).
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, Throwable aCause, String aErrorCode, Term aSource ) {
		super( aArgs, aMessage, aCause, aErrorCode );
		_source = aSource;
		_matchCount = aSource.getMatchCount();
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aSource The {@link Term} responsible for this exception (source).
	 */
	public ArgsSyntaxException( String[] aArgs, Throwable aCause, Term aSource ) {
		super( aArgs, aCause );
		_source = aSource;
		_matchCount = aSource.getMatchCount();
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aErrorCode The error code identifying this exception.
	 * @param aSource The {@link Term} responsible for this exception (source).
	 */
	public ArgsSyntaxException( String[] aArgs, Throwable aCause, String aErrorCode, Term aSource ) {
		super( aArgs, aCause, aErrorCode );
		_source = aSource;
		_matchCount = aSource.getMatchCount();
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage ) {
		super( aArgs, aMessage );
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aErrorCode The error code identifying this exception.
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, String aErrorCode ) {
		super( aArgs, aMessage, aErrorCode );
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, Throwable aCause ) {
		super( aArgs, aMessage, aCause );
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aMessage The aMessage describing this exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aErrorCode The error code identifying this exception.
	 */
	public ArgsSyntaxException( String[] aArgs, String aMessage, Throwable aCause, String aErrorCode ) {
		super( aArgs, aMessage, aCause, aErrorCode );
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 */
	public ArgsSyntaxException( String[] aArgs, Throwable aCause ) {
		super( aArgs, aCause );
	}

	/**
	 * Instantiates a new {@link ArgsSyntaxException} exception.
	 *
	 * @param aArgs The arguments participating in the exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aErrorCode The error code identifying this exception.
	 */
	public ArgsSyntaxException( String[] aArgs, Throwable aCause, String aErrorCode ) {
		super( aArgs, aCause, aErrorCode );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the number of args being matched upon encountering this
	 * exception.
	 * 
	 * @return The number of args matching till this exception occurred.
	 */
	@Override
	public int getMatchCount() {
		return _matchCount; // Don't talk to strangers, directly provide the match count.
	}

	/**
	 * Returns the {@link Term} responsible for this exception.
	 * 
	 * @return The {@link Term} where this exception occurred.
	 */
	@Override
	public Term getSource() {
		return _source;
	}

	/**
	 * Tries to determine the most likely cause of this exception by crawling
	 * the exception tree down the causes and suppressed exceptions and
	 * evaluating their {@link Term} source's {@link Term#getMatchCount()}
	 * argument match counts.
	 * 
	 * {@link #getSuppressed()} exceptions, the cause {@link #getCause()} is at
	 * a CHILD level of THIS exception, the cause's suppressed exceptions are at
	 * the SAME level as the cause itself. A wrong idea of the hierarchy of
	 * exceptions has erroneously been implemented, this implementation got to
	 * corrected as described.
	 * 
	 * @return The heuristic cause of this exception.
	 */
	public String toHeuristicMessage() {
		ArgsSyntaxException theRootCause = toRootMatchCause();
		int theMatchCount = theRootCause.getMatchCount();
		List<ArgsSyntaxException> theCauses = new ArrayList<>();
		theCauses.add( theRootCause );
		if ( getSuppressed() != null ) {
			for ( Throwable eSuppressed : getSuppressed() ) {
				if ( eSuppressed instanceof ArgsSyntaxException theException ) {
					theException = theException.toRootMatchCause();
					if ( theException.getMatchCount() == theMatchCount ) {
						theCauses.add( theException );
					}
					else if ( theException.getMatchCount() > theMatchCount ) {
						theMatchCount = theException.getMatchCount();
						theCauses.clear();
						theCauses.add( theException );
					}
				}
			}
		}

		if ( !theCauses.isEmpty() ) {
			StringBuilder theMessage = new StringBuilder();
			Iterator<ArgsSyntaxException> e = theCauses.iterator();
			ArgsSyntaxException eException;
			while ( e.hasNext() ) {
				eException = e.next();
				theMessage.append( eException.toMessage() );
				if ( e.hasNext() ) {
					theMessage.append( ' ' );
				}
			}
			return theMessage.toString();
		}
		return toMessage();
	}

	/**
	 * Determines the root cause of this exception with the same match count
	 * ({@link #getMatchCount()}) as this exception.
	 * 
	 * @return The actual root cause of this exception or this exception if no
	 *         (more specific) root cause was determined.
	 */
	public ArgsSyntaxException toRootMatchCause() {
		return toRootMatchCause( this );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static ArgsSyntaxException toRootMatchCause( ArgsSyntaxException aException ) {
		ArgsSyntaxException theException = null;
		while ( aException.getCause() instanceof ArgsSyntaxException eException && aException.getCause() != aException && eException.getMatchCount() >= aException.getMatchCount() ) {
			theException = eException;
			if ( theException != null && theException != aException ) return theException;
		}
		if ( aException.getSuppressed() != null && aException.getSuppressed().length != 0 ) {
			for ( Throwable eSuppressed : aException.getSuppressed() ) {
				if ( eSuppressed instanceof ArgsSyntaxException eException ) {
					theException = toRootMatchCause( theException );
					if ( theException != null && theException != aException ) return theException;
				}
			}
		}
		return aException;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
