// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * An {@link XorCondition} represents a list of {@link Condition} ({@link Term})
 * instances of which only one is allowed to be parsed successfully when the
 * {@link Condition#parseArgs(String[], String[], CliContext)} methods are
 * invoked. The command line arguments syntax <code>{ -a ^ -b ^ -c }</code>
 * specifies that only "-a", only "-b" or only "-c" must be set. In case more
 * then one is set or none, then the {@link XorCondition} will terminate the
 * {@link #parseArgs(String[], String[], CliContext)} method with an exception.
 */
public class XorCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Term}
	 * ({@link Condition}) instances to be nested.
	 *
	 * @param aArgs The {@link Term} ({@link Condition}) instances to be nested
	 */
	public XorCondition( Term... aArgs ) {
		super( "Exactly one (XOR) syntax branch must match from the command line arguments.", aArgs );
	}

	/**
	 * Instantiates a new {@link XorCondition} with the {@link Term}
	 * ({@link Condition}) instances to be nested.
	 * 
	 * @param aDescription The description of this {@link Condition}.
	 *
	 * @param aArgs The {@link Term} ({@link Condition}) instances to be nested
	 */
	protected XorCondition( String aDescription, Term... aArgs ) {
		super( aDescription, aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions, CliContext aCliCtx ) throws ArgsSyntaxException {
		List<Condition> theOptionals = new ArrayList<Condition>();
		List<Operand<?>> theResult = null;
		Synopsisable theResultSyntaxable = null;
		List<Operand<?>> eOperands = null;
		int theMatchCount = _children.length;
		ArgsSyntaxException theCause = null;
		for ( Term eChild : _children ) {
			if ( eChild instanceof AnyCondition ) theOptionals.add( (AnyCondition) eChild );
			try {
				if ( theResult != null && eChild instanceof Operand && theResultSyntaxable instanceof Option ) {
					eOperands = (List<Operand<?>>) eChild.parseArgs( toArgsDiff( aArgs, theResult ), aOptions, aCliCtx );
				}
				else {
					eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions, aCliCtx );
				}
				if ( eOperands != null && eOperands.isEmpty() ) eOperands = null;

			}
			catch ( ArgsSyntaxException e ) {
				if ( theCause == null ) {
					theCause = e;
				}
				else {
					theCause.addSuppressed( e );
				}
				theMatchCount--;
			}
			// Same operands in different children |-->
			theResult = removeDuplicates( theResult, eOperands );
			// Same operands in different children <--|
			if ( theResult != null && eOperands != null ) {
				throw _exception = new AmbiguousArgsException( aArgs, "More than one syntax branch matched the command line arguments, though exactly one (XOR) syntax branch must match!", theCause, this );
			}
			if ( eOperands != null && !eOperands.isEmpty() && theResult == null ) {
				theResult = eOperands;
				theResultSyntaxable = eChild;
			}
			eOperands = null;
		}
		if ( theMatchCount == 0 ) {
			throw _exception = new UnknownArgsException( aArgs, "No syntax branch (fully) matched the command line arguments, though one (XOR) syntax branch must match!", theCause, this );
		}
		if ( theMatchCount > 1 ) {
			if ( theOptionals.size() > 1 && (aArgs != null && aArgs.length > 0) ) {
				throw _exception = new AmbiguousArgsException( aArgs, "No syntax branch (fully) matched the command line arguments causing exclusive (XOR) ambiguity for the according syntax branch!", theCause, this );
			}
			theMatchCount -= theOptionals.size();
			if ( theMatchCount > 1 ) {
				throw _exception = new AmbiguousArgsException( aArgs, "No syntax branch (fully) matched the command line arguments causing exclusive (XOR) ambiguity for the according syntax branch!", theCause, this );
			}
		}
		if ( theResult != null ) {
			return theResult;
		}
		if ( theOptionals.size() == 0 ) {
			return Collections.EMPTY_LIST;
		}
		throw _exception = new UnknownArgsException( aArgs, "No syntax branch (fully) matched the provided command line arguments, though one (XOR) syntax branch must match!", theCause, this );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( CliContext aCliCtx ) {
		String theSynopsis = "";
		for ( Term eChild : _children ) {
			if ( theSynopsis.length() != 0 ) {
				theSynopsis += aCliCtx.getSyntaxMetrics().getXorSymbol() != null && aCliCtx.getSyntaxMetrics().getXorSymbol().length() != 0 ? " " + aCliCtx.getSyntaxMetrics().getXorSymbol() + " " : " ";
			}
			theSynopsis += eChild.toSyntax( aCliCtx );
		}
		return theSynopsis;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static List<Operand<?>> removeDuplicates( List<Operand<?>> aElements, List<Operand<?>> aDuplicates ) {
		if ( aElements != null && aDuplicates != null ) {
			for ( Operand<?> eOperand : aDuplicates ) {
				if ( aElements.contains( eOperand ) ) {
					aElements.remove( eOperand );
				}
			}
			if ( aElements.isEmpty() ) {
				aElements = null;
			}
		}
		return aElements;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
