// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import static org.junit.jupiter.api.Assertions.*;

import java.util.Arrays;
import java.util.List;

import org.junit.jupiter.api.Test;

public class ArgsFilterTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testNoneFilter1() {
		String[] theArgs = { "-Dconsole.width=130", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-Dconsole.width=130", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		validate( ArgsFilter.NONE, theArgs, theExpected );
	}

	@Test
	public void testNoneFilter2() {
		String[] theArgs = { "-Dconsole.width=130\n", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-Dconsole.width=130\n", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		validate( ArgsFilter.NONE, theArgs, theExpected );
	}

	@Test
	public void testDFilter1() {
		String[] theArgs = { "-Dconsole.width=130", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "<somePath>" };
		validate( ArgsFilter.D, theArgs, theExpected );
	}

	@Test
	public void testDFilter2() {
		String[] theArgs = { "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "<somePath>" };
		validate( ArgsFilter.D, theArgs, theExpected );
	}

	@Test
	public void testDFilter3() {
		String[] theArgs = { "-a", "-b", "<someB>", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "<somePath>" };
		validate( ArgsFilter.D, theArgs, theExpected );
	}

	@Test
	public void testDFilter4() {
		String[] theArgs = { "-a", "-b", "<someB>", "--Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "--Dlog.debug=true", "<somePath>" };
		validate( ArgsFilter.D, theArgs, theExpected );
	}

	@Test
	public void testDFilter5() {
		String[] theArgs = { "-a", "-b", "<someB>", "Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "Dlog.debug=true", "<somePath>" };
		validate( ArgsFilter.D, theArgs, theExpected );
	}

	@Test
	public void testDFilter6() {
		String[] theArgs = { "-Dconsole.width=130", "-Dconsole.linebreak=\n", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "<somePath>" };
		validate( ArgsFilter.D, theArgs, theExpected );
	}

	@Test
	public void testXXFilter1() {
		String[] theArgs = { "-XX:codecachetotal", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		validate( ArgsFilter.XX, theArgs, theExpected );
	}

	@Test
	public void testXXFilter2() {
		String[] theArgs = { "-XXwhatever", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-XXwhatever", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		validate( ArgsFilter.XX, theArgs, theExpected );
	}

	@Test
	public void testDXXFilter1() {
		String[] theArgs = { "-XX:codecachetotal", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-a", "-b", "<someB>", "<somePath>" };
		validate( ArgsFilter.D_XX, theArgs, theExpected );
	}

	@Test
	public void testDXXFilter2() {
		String[] theArgs = { "-XXwhatever", "-XX:codecachetotal", "-a", "-b", "<someB>", "-Dlog.debug=true", "<somePath>" };
		String[] theExpected = { "-XXwhatever", "-a", "-b", "<someB>", "<somePath>" };
		validate( ArgsFilter.D_XX, theArgs, theExpected );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static void validate( ArgsFilter aArgsFilter, String[] aArgs, String[] aExpected ) {
		String[] theResult = aArgsFilter.toFiltered( aArgs );
		List<String> theList = Arrays.asList( aArgs );
		List<String> theFiltered = aArgsFilter.toFiltered( theList );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Args := " + Arrays.toString( aArgs ) );
			System.out.println( "Expected := " + Arrays.toString( aExpected ) );
			System.out.println( "Result := " + Arrays.toString( theResult ) );
			System.out.println( "Filtered := " + theFiltered );
			System.out.println();
		}
		assertArrayEquals( aExpected, theResult );
		assertArrayEquals( aExpected, theFiltered.toArray( new String[theFiltered.size()] ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
